/****************************************************************************
 *
 * Copyright (c) 2009-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/

package com.ebmwebsourcing.easyviper.core.impl.engine;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Logger;

import org.jdom.Element;
import org.oasisopen.sca.ServiceReference;
import org.oasisopen.sca.annotation.PolicySets;
import org.oasisopen.sca.annotation.Reference;
import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.fraclet.annotations.Lifecycle;
import org.objectweb.fractal.fraclet.types.Step;
import org.osoa.sca.annotations.Property;
import org.ow2.frascati.tinfi.api.control.ContentInstantiationException;
import org.ow2.frascati.tinfi.api.control.SCAContentController;

import com.ebmwebsourcing.easycommons.sca.helper.api.SCAException;
import com.ebmwebsourcing.easycommons.sca.helper.impl.Binding;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAComponentImpl;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAHelper;
import com.ebmwebsourcing.easycommons.thread.ConcurrentHashMapWithDefaults;
import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.Node;
import com.ebmwebsourcing.easyviper.core.api.engine.Scope;
import com.ebmwebsourcing.easyviper.core.api.engine.fault.Fault;
import com.ebmwebsourcing.easyviper.core.api.engine.variable.Variable;
import com.ebmwebsourcing.easyviper.core.api.soa.Partner;

/**
 * @author Nicolas Salatge - EBM WebSourcing
 */
@org.oasisopen.sca.annotation.Scope("COMPOSITE")
@org.oasisopen.sca.annotation.Service(value = Execution.class, names = "service")
@PolicySets("frascati:scaEasyPrimitive")
public class ExecutionImpl extends SCAComponentImpl implements Execution {

	private Logger log = Logger.getLogger(ExecutionImpl.class.getSimpleName());

	/**
	 * variable for debugging
	 */
	@Property(name = "stepByStep", required = true)
	private boolean stepByStep = false;

	private AtomicReference<State> currentState = new AtomicReference<Execution.State>(
			State.INACTIVE);

	private Node currentTarget;

	@Reference(name = "parentExec", required = false)
	private Execution parent = null;

	/** are concurrent childExecutions that related to this execution. */
	@Reference(name = "childs", required = false)
	private final List<Execution> childExecutions = new ArrayList<Execution>();

	private Exception exception;

	private ConcurrentHashMapWithDefaults<String, Object> variableValues = new ConcurrentHashMapWithDefaults<String, Object>();

	private ConcurrentHashMapWithDefaults<String, Element> partnerValues = new ConcurrentHashMapWithDefaults<String, Element>();

	public Node getCurrentTarget() {
		return currentTarget;
	}

	public synchronized void run() throws CoreException {
		log.fine("execution run begins for " + getName());

		setState(Execution.State.ACTIVE);

		while ((getState() != State.ENDED) && (getState() != State.CANCELLED)
				&& (getState() != State.SUSPENDED)) {
			step();
		}

	}

	public void removeChildExecution(final Execution execution) throws CoreException {
		if (execution != null) {
			Component exeComp;
			try {
				Object obj = SCAHelper.getSCAHelper().getParent(this.getComponent()).getFcInterface("service");
				if(obj instanceof Scope) {
					Scope scope = (Scope) obj;
					exeComp = SCAHelper.getSCAHelper().getComponentByInterface(
							scope.getComponent(),
							(org.objectweb.fractal.api.Interface) execution.getComponent()
							.getFcInterface("service"), "service");

					if (exeComp != null) {
						SCAHelper.getSCAHelper().deleteComponent(exeComp);
					}
				}
			} catch (NoSuchInterfaceException e) {
				throw new CoreException(e);
			} catch (SCAException e) {
				throw new CoreException(e);
			}

		}
	}

	@Override
	public final boolean hasCurrentTarget() {
		return currentTarget != null;
	}

	public final void setInitialTarget(Node initialTarget) {
		setCurrentTarget(initialTarget);
	}

	private final void setCurrentTarget(Node currentTarget) {
		String oldTargetName = this.currentTarget != null ? this.currentTarget.getName() : "none";

		this.currentTarget = currentTarget;

		if (this.currentTarget == null)
			return;

		try {
			Node target = null;
			if (currentTarget instanceof ServiceReference<?>) {
				ServiceReference<?> sr = (ServiceReference<?>) currentTarget;
				target = (Node) SCAHelper.getSCAHelper().getBoundInterface(sr);
			} else {
				target = currentTarget;
			}

			final Component targetComp = SCAHelper.getSCAHelper().getComponentByInterface(
					SCAHelper.getSCAHelper().getParent(this.getComponent()),
					(org.objectweb.fractal.api.Interface) target, "service");

			// start the target component
			if ((targetComp != null) && (!SCAHelper.getSCAHelper().isStarted(targetComp))) {
				SCAHelper.getSCAHelper().startComponent(targetComp);
			}

		} catch (SCAException e) {
			throw new CoreException(e);
		}
		String newTargetName = this.currentTarget != null ? this.currentTarget.getName() : "none";

		this.log.fine(String.format("current target = '%s' ; next target = '%s'", oldTargetName,
				newTargetName));

	}

	public void end() throws CoreException {
		setCurrentTarget(null);
	}

	public List<Execution> getChildExecutions() {
		return Collections.unmodifiableList(childExecutions);
	}

	public Execution getParentExecution() {
		return this.parent;
	}

	public State getState() {
		return this.currentState.get();
	}

	public void setStateRecursively(final State state) {
		// TODO : in the case we set recursively SUSPENDED state,
		// there is a chance that a parent execution becomes SUSPENDED
		// whereas some child execution are still ACTIVE.
		// Instead of that, we should mark a parent execution as SUSPENDED
		// if and only if all child executions are ENDED or SUSPENDED.
		this.setState(state);
		if (getParentExecution() != null) {
			getParentExecution().setStateRecursively(state);
		}
	}

	public void setState(final State state) {
		this.currentState.set(state);
		log.fine("Execution " + this.getName() + " is now in state " + this.getState());
	}

	public void addExecution(final Execution childExecution) throws CoreException {
		try {
			// link child to execution
			List<Binding> listOfBindings = new ArrayList<Binding>();
			listOfBindings.add(new Binding("childs_" + String.valueOf(this.childExecutions.size()),
					(org.objectweb.fractal.api.Interface) (childExecution).getComponent()
					.getFcInterface("service")));
			SCAHelper.getSCAHelper().addBindings(this.getComponent(), listOfBindings);

			// link execution to child
			listOfBindings = new ArrayList<Binding>();
			listOfBindings.add(new Binding("parentExec", (org.objectweb.fractal.api.Interface) this
					.getComponent().getFcInterface("service")));
			SCAHelper.getSCAHelper().addBindings(childExecution.getComponent(), listOfBindings);

			// inherit parent execution variable and partner values
			ExecutionImpl childExecutionImpl;
			try {
				childExecutionImpl = (ExecutionImpl) ((SCAContentController) childExecution
						.getComponent().getFcInterface(SCAContentController.NAME)).getFcContent();
			} catch (ContentInstantiationException cie) {
				throw new CoreException(cie);
			}
			childExecutionImpl.variableValues = variableValues;
			childExecutionImpl.partnerValues = partnerValues;

		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		} catch (final SCAException e) {
			throw new CoreException(e);
		}
	}

	public void removeChildExecutions() throws CoreException {
		Execution child = null;
		while ((this.childExecutions != null) && (this.childExecutions.size() > 0)) {
			child = this.childExecutions.iterator().next();
			log.fine("remove execution: " + child.getName());
			this.removeChildExecution(child);
		}
	}

	public void setParentExecution(final Execution parent) {
		assert parent != null;
		this.parent = parent;
		this.variableValues = new ConcurrentHashMapWithDefaults<String, Object>(
				((ExecutionImpl) parent).variableValues);
		this.partnerValues = new ConcurrentHashMapWithDefaults<String, Element>(
				((ExecutionImpl) parent).partnerValues);
	}

	public final synchronized void step() throws CoreException {
		if (getState() == State.ENDED)
			return;
		//assert hasCurrentTarget();

		Node nextTarget = null;
		if (hasCurrentTarget()) {
			try {
				nextTarget = currentTarget.execute(this);
			} catch (CoreException e) {
				Scope faultScope = findFaultScope(e);

				if (faultScope == null || faultScope.getException() != null) {
					log.fine("fault not catched: " + e.getClass().getName());
					setException(e);
					setState(State.ENDED);
					throw e;
				} else {
					log.fine("fault catched by scope: " + faultScope.getName());
					faultScope.setException(e);
					try {
						nextTarget = (Node) faultScope.getComponent().getFcInterface("service");
					} catch (NoSuchInterfaceException e1) {
						throw new CoreException(e1);
					}
				}

			}
			setCurrentTarget(nextTarget);
		}

		if (!hasCurrentTarget()) {
			if (getParentExecution() != null) {
				synchronized(getParentExecution()) {
					getParentExecution().removeChildExecution(this);
				}
			}
			setState(State.ENDED);
		}

	}

	public void runStepByStep() throws CoreException {
		this.stepByStep = true;
		this.setState(Execution.State.ACTIVE);
	}

	public synchronized void signal() throws CoreException {
		step();

	}

	public boolean isStepByStep() {
		return this.stepByStep;
	}

	public void setStepByStep(final boolean stepByStep) {
		this.stepByStep = stepByStep;
	}

	public void setLog(final Logger logger) {
		this.log = logger;
	}

	/**
	 * Destroy the ExecutionImpl behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = Step.DESTROY)
	public void destroySCAComponent() throws SCAException {
		this.log.fine("Fractal execution destroyed: " + this.getName());
		this.setState(Execution.State.CANCELLED);
	}

	@Override
	public String toString() {
		String res = super.toString();

		if (this.getName() != null) {
			res = this.getName();
		}

		return res;
	}

	private static final void doGetSuspendedExecutions(Execution execution,
			Map<String, Execution> suspendedExecutions) {
		
		if (execution.getState() == Execution.State.SUSPENDED) {
		
			suspendedExecutions.put(execution.getName(), execution);
		}
		for (Execution childExecution : execution.getChildExecutions()) {
		
			doGetSuspendedExecutions(childExecution, suspendedExecutions);
		}
	}

	public Map<String, Execution> getSuspendedExecutions() throws CoreException {
		Map<String, Execution> result = new HashMap<String, Execution>();
		doGetSuspendedExecutions(this, result);
		return result;
	}

	private final void setException(CoreException exception) {
		this.exception = exception;
	}

	@Override
	public final Exception getException() {
		return exception;
	}


	@Override
	public final Object getVariableValue(String name) {
		return variableValues.getValueOrDefault(name);
	}

	@Override
	public Element getPartnerValue(String name) {
		return partnerValues.getValueOrDefault(name);
	}

	@Override
	public final void putVariableValue(String name, Object value) {
		variableValues.put(name, value);
	}

	// TODO : rename put => init
	@Override
	public void putPartnerValue(String name, Element value) {
		partnerValues.put(name, value);
	}


	@Override
	public final void assignVariableValue(String name, Object value) {
		Map<String, Object> map = variableValues;
		while (true) {
			assert map != null;
			if (map.containsKey(name)) {
				map.put(name, value);
				return;
			}

			assert map instanceof ConcurrentHashMapWithDefaults;
			map = ((ConcurrentHashMapWithDefaults<String, Object>) map).getDefaults();
		}

	}


	@Override
	public void assignPartnerValue(String name, Element value) {
		Map<String, Element> map = partnerValues;
		while (true) {
			assert map != null;
			if (map.containsKey(name)) {
				map.put(name, value);
				return;
			}

			assert map instanceof ConcurrentHashMapWithDefaults;
			map = ((ConcurrentHashMapWithDefaults<String, Element>) map).getDefaults();
		}
	}


	@Override
	public final void enterScope(Scope scope) {
		System.err.println("---------------------------------ENTER SCOPE " + scope.getName());
		variableValues = new ConcurrentHashMapWithDefaults<String, Object>(variableValues);
		partnerValues = new ConcurrentHashMapWithDefaults<String, Element>(partnerValues);

		for (Variable var : scope.getVariables().values()) {
			var.initialize(this);
		}


		for (Partner partner : scope.getPartners().values()) {
			System.err.println("---------------------------------INIT partner " + partner.getName());
			partner.initialize(this);
		}

	}

	@Override
	public final void leaveScope() {
		variableValues = (ConcurrentHashMapWithDefaults<String, Object>) variableValues
				.getDefaults();
		partnerValues = (ConcurrentHashMapWithDefaults<String, Element>) partnerValues
				.getDefaults();
	}


	private final Scope findFaultScope(CoreException e) throws CoreException {
		Scope faultScope = null;
		// test if currentScope is not already a fault scope
		Exception exTest = null;
		Scope current = getCurrentScope();

		while (current != null) {
			exTest = current.getException();
			if (exTest == null) {
				for (Entry<Fault, Scope> entry : current.getExceptions().entrySet()) {
					if (entry.getValue().getException() == null
							&& entry.getKey().getFaultHandler() != null
							&& entry.getKey().getFaultHandler()
							.match(this, entry.getKey(), e, entry.getValue())) {
						faultScope = entry.getValue();
						if (entry.getKey().getVariable() != null) {
							e.setFaultVariableName(entry.getKey().getVariable().getName());
						}
						return faultScope;
					}

				}
			}
			exTest = null;
			current = current.getParentScope();
		}
		return null;
	}

	@Override
	public final Scope getCurrentScope() {
		Node currentTarget;
		try {
			currentTarget = (Node) getCurrentTarget().getComponent().getFcInterface("service");
			if (currentTarget instanceof Scope) {
				return (Scope) currentTarget;
			} else {
				return currentTarget.getScope();
			}
		} catch (NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
	}

	
}
