/****************************************************************************
 *
 * Copyright (c) 2009-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/
 
package com.ebmwebsourcing.easyviper.core.impl.engine.time;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.functionnal.WaitBehaviour;
import com.ebmwebsourcing.easyviper.core.api.engine.time.TimerFinishedEvent;

/**
 * @author Nicolas Salatge - EBM WebSourcing
 */
public class WaitingThreadPool {

	private static Logger log = Logger.getLogger(WaitingThreadPool.class.getSimpleName());
	
	private static WaitingThreadPool instance;
	
	private static Map<Execution,List<WaitingThread>> threads = Collections.synchronizedMap(new HashMap<Execution,List<WaitingThread>>());
	private static Map<Execution,List<CoreException>> exceptions = Collections.synchronizedMap(new HashMap<Execution,List<CoreException>>());
	
	private WaitingThreadPool() {}
	
	public static synchronized void addWaitingThread(Execution exec, WaitBehaviour waitBehaviour, long timeToWait) {
		WaitingThread wt = getInstance().new WaitingThread(exec, waitBehaviour, timeToWait);
		if(!threads.containsKey(exec)) {
			threads.put(exec,new ArrayList<WaitingThread>());
		}
		threads.get(exec).add(wt);
		
		log.finest("new Waiting thread for "+waitBehaviour.getName()+" "+timeToWait);
		
		wt.start();
	}
	
	public static synchronized void removeWaitingThread(Execution exec, WaitBehaviour wb) {
		if(threads.containsKey(exec)) {
			WaitingThread toBeRemoved = null;
			for(WaitingThread wt : threads.get(exec)) {
				if(wt.behaviour.equals(wb)) {
					toBeRemoved = wt;
					break;
				}
			}
			threads.get(exec).remove(toBeRemoved);
		}
		
		//TODO also stop the thread
	}
	
	private synchronized void onWaitTermination(Execution exec, WaitBehaviour b) {
		if(threads.containsKey(exec)) {
			threads.get(exec).remove(Thread.currentThread());
			try {
				exec.getCurrentScope().getProcess().getEngine().onTimerFinish(exec, new TimerFinishedEvent(exec,b));
			} catch (CoreException e) {
				// TODO maybe end the process or at least notify the engine
				e.printStackTrace();
				addException(exec, e);
			}
		}
	}
	
	private void addException(Execution exec, CoreException e) {
		if(!exceptions.containsKey(exec)) {
			exceptions.put(exec, new ArrayList<CoreException>());
		}
		exceptions.get(exec).add(e);
	}
	
	private class WaitingThread extends Thread {
		
		private Execution exec;
		private WaitBehaviour behaviour;
		private long time;
	
		protected WaitingThread(Execution exec, WaitBehaviour wb, long timeToWait) {
			this.time = timeToWait;
			this.behaviour = wb;
			this.exec = exec;
		}
		
		@Override
		public void run() {
			try {
				Thread.sleep(time);
				log.finest("Sleep ended for "+behaviour.getName());
			} catch (InterruptedException e) {
				return;
			}
			WaitingThreadPool.this.onWaitTermination(exec,behaviour);
		}
	}
	
	private static WaitingThreadPool getInstance() {
		if(instance == null) {
			instance = new WaitingThreadPool();
		}
		return instance;
	}
}
