/****************************************************************************
 *
 * Copyright (c) 2009-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/

package com.ebmwebsourcing.easyviper.core.impl.engine.behaviour.functionnal;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;

import org.oasisopen.sca.annotation.PolicySets;
import org.osoa.sca.annotations.Property;

import com.ebmwebsourcing.easycommons.lang.UncheckedException;
import com.ebmwebsourcing.easycommons.uuid.SimpleUUIDGenerator;
import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Engine;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.Node;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.AbstractBehaviourImpl;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.AbstractComplexBehaviourImpl;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.Behaviour;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.functionnal.ExclusiveBehaviour;
import com.ebmwebsourcing.easyviper.core.api.env.ExternalContext;
import com.ebmwebsourcing.easyviper.core.impl.engine.ExecutionThread;
import com.ebmwebsourcing.easyviper.core.impl.engine.ProcessImpl;

/**
 * @author Nicolas Salatge - EBM WebSourcing
 */
@org.oasisopen.sca.annotation.Scope("COMPOSITE")
@org.oasisopen.sca.annotation.Service(value = ExclusiveBehaviour.class, names = "service")
@PolicySets("frascati:scaEasyCompositeWithContent")
public class ExclusiveBehaviourImpl extends AbstractComplexBehaviourImpl implements ExclusiveBehaviour {

	private static final SimpleUUIDGenerator uuidGenerator = new SimpleUUIDGenerator();

	@Property(name = "triggeringNodes", required = false)
	private Set<Node> triggeringNodes;

	@Property(name = "exclusiveNodeToExecute", required = false)
	private Node exclusiveNodeToExecute = null;

	private List<Thread> childExecutionThreads = new ArrayList<Thread>();

	private Execution mainExec = null;

	private Object lock = new Object();

	@Override
	protected State executeOnStarted(Execution execution) {
		if(triggeringNodes != null) {
			if(mainExec == null) {
				mainExec = execution;
			}
			List<Node> childNodes = getNode().getChildNodes();
			assert childNodes != null;
			for (Node childNode : triggeringNodes) {
				ExecutionThread executionThread = FlowBehaviourImpl.startChildExecution(getNode().getProcess(), execution, childNode, uuidGenerator, log);
				System.err.println("------------- start exec " + executionThread + " on node " + childNode.getName());
				childExecutionThreads.add(executionThread);
			}
			com.ebmwebsourcing.easyviper.core.api.engine.Process process = getNode().getProcess();
			Engine engine = process.getEngine();
			if (!engine.getConfiguration().getSingleThreadedExecution()) {
				for (Thread thread : childExecutionThreads) {
					try {
						thread.join();
					} catch (InterruptedException e) {
						throw new UncheckedException(e);
					}
				}
			}

			if(triggeringNodes.size() > 0) {
				if(exclusiveNodeToExecute == null) {
					synchronized(lock) {
						try {
							lock.wait();
						} catch (InterruptedException e) {
							throw new CoreException(e);
						}
					}
				}

				if(exclusiveNodeToExecute != null) {
					execution.setStateRecursively(Execution.State.ACTIVE);
					selectNextNodeToExecute(execution, exclusiveNodeToExecute);
				}
			} else {
				selectNextNodeToExecute(execution, null);
			}
		} else {
			selectNextNodeToExecute(execution, null);
		}
		return State.ACTIVITY_ENDED;
	}





	@Override
	public synchronized Node onChildBehaviourNotification(Execution childExecution, Node nextNode,
			Behaviour childBehaviour) throws CoreException {
		log.fine("Notification received from " + childBehaviour.getName());

		if(childExecutionThreads.size() == 1 && nextNode == null) {
			selectNextNodeToExecute(mainExec, null);
			exclusiveNodeToExecute = this.nextTarget;
			synchronized (lock) {
				lock.notify();
			}
		}
		log.fine("next node " + nextNode);
		if(exclusiveNodeToExecute == null) {
			exclusiveNodeToExecute = nextNode;
			synchronized (lock) {
				lock.notify();
			}
			childExecution.getCurrentTarget().getBehaviour().setCurrentExecution(mainExec);
			childExecution.setState(Execution.State.CANCELLED);
			childExecutionThreads.remove(childExecution);
			
		} else {
			for(ExecutionThread childExec : Arrays.asList(childExecutionThreads.toArray(new ExecutionThread[childExecutionThreads.size()]))) {
				if(childExec.getExecution() == childExecution) {
					childExec.getExecution().setState(Execution.State.CANCELLED);
					childExecutionThreads.remove(childExec);
				}
			}
			return null;
		}
		return this.getNode();
	}



	@Override
	public void reset() {
		super.reset();
		exclusiveNodeToExecute = null;;
		childExecutionThreads = new ArrayList<Thread>();
		mainExec = null;
	}



}
