/****************************************************************************
 *
 * Copyright (c) 2009-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/

package com.ebmwebsourcing.easyviper.core.impl.engine.behaviour.functionnal;

import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.QName;

import org.osoa.sca.annotations.Property;

import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.Node;
import com.ebmwebsourcing.easyviper.core.api.engine.Scope;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.AbstractComplexBehaviourImpl;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.functionnal.ReceiverBehaviour;
import com.ebmwebsourcing.easyviper.core.api.engine.variable.Variable;
import com.ebmwebsourcing.easyviper.core.api.soa.correlation.CorrelationMatcher;
import com.ebmwebsourcing.easyviper.core.api.soa.message.BindingInputMessageAdapter;
import com.ebmwebsourcing.easyviper.core.api.soa.message.BindingMessageAdapter.Direction;
import com.ebmwebsourcing.easyviper.core.api.soa.message.Message;
import com.ebmwebsourcing.easyviper.core.api.soa.message.MessageMatcher;

/**
 * @author Nicolas Salatge - EBM WebSourcing
 */
@org.oasisopen.sca.annotation.Scope("COMPOSITE")
@org.oasisopen.sca.annotation.Service(value=ReceiverBehaviour.class,names="service")
public class ReceiverBehaviourImpl extends AbstractComplexBehaviourImpl implements
ReceiverBehaviour {

	private Message message;

	@Property(name="variableName", required=true)
	private String variableName = null;

	@Property(name="messageMatcher", required=true)
	private MessageMatcher messageMatcher = null;

	@Property(name="correlationsMatcher", required=true)
	private List<CorrelationMatcher> correlationsMatcher = new ArrayList<CorrelationMatcher>();

	@Property(name="interfaceName", required=false)
	private QName interfaceName;

	@Property(name="inputMessageAdapter", required=false)
	private BindingInputMessageAdapter messageAdapter;

	private Node childNodeSelected = null;

	public boolean accept(Execution execution, 
			final Message message)
					throws CoreException {
		boolean accept = false;
		final Scope scope = getNode().getScope();

		final Variable v = scope.findVariable(variableName);
		if (v == null) {
			throw new CoreException(
					"Impossible to find in process, the variable used in the received : "
							+ this.getName());
		}

		// check the message
		boolean check = false;
		if (this.messageMatcher != null) {
			check = this.messageMatcher.match(v, message);
		} else {
			this.log.warning("no matcher found");
			check = true;
		}

		if(!check) {
			log.fine("message not valid for variable: " + variableName);
		} else {

			boolean correlationOK = false;
			if(this.correlationsMatcher != null && this.correlationsMatcher.size() > 0) {
				log.fine("Correlation found: execute correlation for receiver: " + this.getName());
				for(CorrelationMatcher corr: this.correlationsMatcher) {
					correlationOK = corr.match(execution, variableName, message, messageAdapter, interfaceName);
					if(correlationOK) {
						break;
					}
				}
			} else {
				log.fine("No correlation ");
				correlationOK = true;
			}

			if ((check)&&(correlationOK)) {
				boolean isBindingStyleRpc = messageAdapter.isBindingStyleRpc(
						message.getEndpoint(), message.getService(), interfaceName, message.getOperationName());
				messageAdapter.adaptFromBindingInput(execution, message, v, Direction.REQUEST, isBindingStyleRpc);

				// valid the accept
				accept = true;
				selectNextNodeToExecute(execution, getNode());
				this.log.fine("Message accepted on receive by variable: "
						+ variableName);
				if (!getNode().getChildNodes().isEmpty()) {
					this.childNodeSelected = getNode().getChildNodes().get(0);
				}
			}
		}

		return accept;
	}



	public void setMessage(final Message message) {
		this.message = message;
	}

	public Message getMessage() {
		return this.message;
	}

	public Node getChildNodeSelected() {
		return this.childNodeSelected;
	}

	private void executeOnSuspended(Execution execution) throws CoreException {
		if (message != null) {
			final boolean accept = this.accept(execution, message);
			if (accept) {
				this.log.fine("Message accepted by the receive activity: "
						+ this.getName());

				Node nextNode = notifyParentBehaviour(execution, getChildNodeSelected());
				if(nextNode == null) {
					selectNextNodeToExecute(execution, getChildNodeSelected());
				} else {
					selectNextNodeToExecute(execution, null);
				}
				execution.setStateRecursively(Execution.State.ACTIVE);
			} else {
				execution.setStateRecursively(
						Execution.State.SUSPENDED);
				this.log
				.fine("Message does not correspond - execution suspended: "
						+ execution.getState());
				selectNextNodeToExecute(execution, getNode());
			}
			message = null;
		} else {
			execution.setStateRecursively(Execution.State.SUSPENDED);
			this.log.fine("No Message - execution suspended: "
					+ execution.getState());
			selectNextNodeToExecute(execution, getNode());
		}
	}


	@Override
	protected State executeOnInactive(Execution execution) throws CoreException {
		executeOnSuspended(execution);
		return State.ACTIVITY_STARTED;
	}

	@Override
	protected State executeOnStarted(Execution execution) throws CoreException {
		if (this.message != null) {
			executeOnSuspended(execution);
			return State.ACTIVITY_STARTED;
		}  else {
			return State.ACTIVITY_ENDED;
		}
	}



	@Override
	protected State executeOnEnded(Execution execution) throws CoreException {
		this.message = null;
		return super.executeOnEnded(execution);
	}

	//    
	//    @Override
	//    public void reset() {
	//        this.message = null;
	//        super.reset();
	//    }

}
