/****************************************************************************
 *
 * Copyright (c) 2009-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/
 
package com.ebmwebsourcing.easyviper.core.impl.engine;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.logging.Logger;

import org.oasisopen.sca.annotation.PolicySets;
import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.Interface;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.osoa.sca.annotations.Property;
import org.ow2.frascati.tinfi.api.control.ContentInstantiationException;
import org.ow2.frascati.tinfi.api.control.SCAContentController;

import com.ebmwebsourcing.easycommons.sca.helper.api.SCAException;
import com.ebmwebsourcing.easycommons.sca.helper.impl.Binding;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAHelper;
import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.ExpressionEvaluator;
import com.ebmwebsourcing.easyviper.core.api.engine.Node;
import com.ebmwebsourcing.easyviper.core.api.engine.PartnerEvaluator;
import com.ebmwebsourcing.easyviper.core.api.engine.Scope;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.Behaviour;
import com.ebmwebsourcing.easyviper.core.api.engine.fault.Fault;
import com.ebmwebsourcing.easyviper.core.api.engine.handler.TerminationHandler;
import com.ebmwebsourcing.easyviper.core.api.engine.variable.Variable;
import com.ebmwebsourcing.easyviper.core.api.soa.Partner;
import com.ebmwebsourcing.easyviper.core.api.soa.correlation.CorrelationGroup;
import com.ebmwebsourcing.easyviper.tools.InstanceOfUtil;

/**
 * @author Nicolas Salatge - EBM WebSourcing
 */
@org.oasisopen.sca.annotation.Scope("COMPOSITE")
@org.oasisopen.sca.annotation.Service(value = Scope.class, names = "service")
@PolicySets("frascati:scaEasyCompositeWithContent")
public class ScopeImpl extends NodeImpl implements Scope {

    private static int cpt_childsNode = 0;

    private static int cpt_incomingNode = 0;

    private static int cpt_outgoingNode = 0;

    private static Logger log = Logger.getLogger(ScopeImpl.class.getSimpleName());

    protected Component initialNode;

    @Property(name = "exceptions", required = false)
    private Map<Fault, Scope> exceptions = new HashMap<Fault, Scope>();

    @Property(name = "partners", required = false)
    protected Map<String, Partner> partners = new HashMap<String, Partner>();

    @Property(name = "correlationGroups", required = false)
    protected List<CorrelationGroup> correlationGroups = new ArrayList<CorrelationGroup>();

    @Property(name = "faultScope", required = false)
    protected Exception faultScope = null;

    @Property(name = "terminationHandler", required = false)
    protected TerminationHandler terminationHandler = null;

    @Property(name = "variables", required = true)
    private final Map<String, Variable> variables = new HashMap<String, Variable>();

    @Property(name = "expressionEvaluator", required = true)
    protected ExpressionEvaluator expressionEvaluator = null;

    @Property(name = "partnerEvaluator", required = true)
    protected PartnerEvaluator partnerEvaluator = null;

    /**
     * Default Constructor
     * 
     */
    public ScopeImpl() throws CoreException {
        super();
    }

    public Node createNode(String behaviourName, Class<? extends Behaviour> behaviourContextClass,
            Map<String, Object> context) throws CoreException {
        ScopeImpl.log.fine("start create node");
        Node node = null;
        String nodeName = null;
        try {

            if (behaviourName == null) {
                behaviourName = behaviourContextClass.getSimpleName();
                nodeName = "node_supporting_" + behaviourName + "_" + this.getAllNodes().size();
            } else {
                nodeName = "node_supporting_" + behaviourName;
            }

            // create node
            final Component fractalNode = SCAHelper.getSCAHelper().createNewComponent(
                    NodeImpl.class.getName(), null);

            SCAHelper.getSCAHelper().addComponent(fractalNode, getComponent(), null);

            SCAContentController scacc = (SCAContentController) fractalNode
                    .getFcInterface(SCAContentController.NAME);
            try {
                node = (Node) scacc.getFcContent();
            } catch (ContentInstantiationException e) {
                throw new CoreException(e);
            }
            ((Node) node).setName(nodeName);
            if (!ScopeImpl.log.getName().equals(ScopeImpl.class.getName())) {
                node.setLog(ScopeImpl.log);
            }

            node = (Node) fractalNode.getFcInterface("service");

            // create behaviour
            final Component fractalBehaviour = SCAHelper.getSCAHelper().createNewComponent(
                    behaviourContextClass.getName(), context);

            SCAHelper.getSCAHelper().changeName(fractalBehaviour, behaviourName);

            SCAHelper.getSCAHelper().addComponent(fractalBehaviour, getComponent(), null);

            Behaviour behaviour = null;
            scacc = (SCAContentController) fractalBehaviour
                    .getFcInterface(SCAContentController.NAME);

            try {
                behaviour = (Behaviour) scacc.getFcContent();
            } catch (ContentInstantiationException e) {
                throw new CoreException(e);
            }

            behaviour = (Behaviour) fractalBehaviour.getFcInterface("service");

            this.linkedNodeAndFunctionnalBehaviour(node, behaviour);

            ScopeImpl.log.fine("Creation of the fractal node: " + behaviourName);

        } catch (final NoSuchInterfaceException e) {
            ScopeImpl.log.severe(e.getMessage());
            throw new CoreException(e);
        } catch (final SCAException e) {
            ScopeImpl.log.severe(e.getMessage());
            throw new CoreException(e);
        }
        ScopeImpl.log.fine("end of create node");

        return node;
    }

    public Node getInitialNode() throws CoreException {
        Node res = null;
        try {
            if (this.initialNode != null) {
                res = (Node) this.initialNode.getFcInterface("service");
            }
        } catch (final NoSuchInterfaceException e) {
            throw new CoreException(e);
        }
        return res;
    }

    public Node getNode(final String name) throws CoreException {
        Node res = null;
        try {
            final List<Component> comps = SCAHelper.getSCAHelper().getComponentsByName(
                    getComponent(), name);

            final List<Node> nodes = new ArrayList<Node>();

            if (comps != null) {
                for (final Component comp : comps) {
                    if (comp.getFcInterface("service") instanceof Node) {
                        nodes.add((Node) comp.getFcInterface("service"));
                    }
                }
            }

            if (nodes.size() > 1) {
                throw new CoreException("Several node components are the same name");
            }

            if (nodes.size() > 0) {
                res = nodes.get(0);
            }

        } catch (final NoSuchInterfaceException e) {
            throw new CoreException(e);
        } catch (final SCAException e) {
            throw new CoreException(e);
        }
        return res;
    }

    public void end(boolean clean) throws CoreException {
        try {
            // apply termination handler
            if (this.terminationHandler != null) {
                terminationHandler.end(this);
            }

            if (clean) {
                ScopeImpl.log.fine("clean all scopes");
                this.cleanScopes(this);

                ScopeImpl.log.fine("stop all sub components");
                SCAHelper.getSCAHelper().stopAllSubComponents(getComponent());
            }
        } catch (final SCAException e) {
            throw new CoreException(e);
        }
    }

    protected void cleanScopes(Scope scope) throws CoreException {
        for (Scope faultScope : scope.getExceptions().values()) {
            cleanScopes(faultScope);
        }

        scope.setException(null);
        

//        try {
//            // find under scopes
//            List<Component> components = SCAHelper.getSCAHelper().getComponents(
//                    scope.getComponent());
//            for (Component component : components) {
//                if (SCAHelper.getSCAHelper().isStarted(component)) {
//                    try {
//                        // TODO lesbegu add ... check
//                        if (component.getFcInterface("service") instanceof Scope) {
//                            Scope underScope = (Scope) component.getFcInterface("service");
//                            cleanScopes(underScope);
//                        }
//                    } catch (NoSuchInterfaceException e) {
//                        // do nothing
//                    }
//                }
//            }
//        } catch (CoreException e) {
//            throw new CoreException(e);
//        } catch (final SCAException e) {
//            throw new CoreException(e);
//        }
    }


    public boolean removeNode(final Node node) throws CoreException {
        final boolean res = true;
        try {
            final Component nodeComp = SCAHelper.getSCAHelper().getComponentByInterface(
                    this.getComponent(), (org.objectweb.fractal.api.Interface) node, "service");

            // delete all child nodes
            Node child = null;
            final String clientItfName = "childNodes_";

            Map<String, org.objectweb.fractal.api.Interface> map = SCAHelper.getSCAHelper()
                    .getServerInterfacesLinkedToClientInterfacesOfComponent(nodeComp);

            Iterator<Entry<String, org.objectweb.fractal.api.Interface>> entries = map.entrySet()
                    .iterator();
            Entry<String, org.objectweb.fractal.api.Interface> entry = null;
            while ((entries != null) && (entries.hasNext())) {
                entry = entries.next();
                if (entry.getKey().startsWith(clientItfName)) {
                    child = (Node) entry.getValue();
                    this.removeNode(child);

                    map = SCAHelper.getSCAHelper()
                            .getServerInterfacesLinkedToClientInterfacesOfComponent(nodeComp);
                    entries = map.entrySet().iterator();
                }
            }

            // delete the component
            SCAHelper.getSCAHelper().deleteComponent(nodeComp);
        } catch (final SCAException e) {
            throw new CoreException(e);
        }
        return res;
    }

    public void setInitialNode(final Node initial) throws CoreException {
        try {
            this.initialNode = SCAHelper.getSCAHelper().getComponentByInterface(
                    this.getComponent(), (Interface) initial, "service");
        } catch (final SCAException e) {
            throw new CoreException(e);
        }
    }

    public List<Node> getAllNodes() throws CoreException {
        final List<Node> nodes = new ArrayList<Node>();
        try {
            final List<Component> comps = SCAHelper.getSCAHelper().getComponents(getComponent());
            for (final Component component : comps) {
                try {
                    if (component.getFcInterface("service") instanceof Node) {
                        nodes.add((Node) component.getFcInterface("service"));
                    }
                } catch (final NoSuchInterfaceException e) {
                    // do nothing
                }
            }
        } catch (final SCAException e) {
            throw new CoreException(e);
        }
        return nodes;
    }

    public List<Scope> getAllScopes() throws CoreException {
        final List<Scope> scope = new ArrayList<Scope>();
        try {
            scope.add(this);
            final List<Component> comps = SCAHelper.getSCAHelper().getAllSubComponents(
                    getComponent());
            for (final Component component : comps) {
                try {
                    if (component.getFcInterface("service") instanceof Scope) {
                        scope.add((Scope) component.getFcInterface("service"));
                    }
                } catch (final NoSuchInterfaceException e) {
                    // do nothing
                }
            }
        } catch (final SCAException e) {
            throw new CoreException(e);
        }
        return scope;
    }

    public Node getNodeByName(final String name) throws CoreException {
        Node res = null;
        try {
            final List<Component> comps = SCAHelper.getSCAHelper().getComponentsByName(
                    getComponent(), name);

            final List<Node> nodes = new ArrayList<Node>();

            if (comps != null) {
                for (final Component comp : comps) {
                    if (comp.getFcInterface("service") instanceof Node) {
                        nodes.add((Node) comp.getFcInterface("service"));
                    }
                }
            }

            if (nodes.size() > 1) {
                throw new CoreException("Several node components are the same name");
            }

            if (nodes.size() > 0) {
                res = nodes.get(0);
            }

        } catch (final NoSuchInterfaceException e) {
            throw new CoreException(e);
        } catch (final SCAException e) {
            throw new CoreException(e);
        }
        return res;
    }

    public void linkedNodeAndFunctionnalBehaviour(final Node targetNode, final Behaviour behaviour)
            throws CoreException {
        try {
            if ((behaviour != null) && (targetNode != null)) {
                final Component targetComp = SCAHelper.getSCAHelper().getComponentByInterface(
                        this.getComponent(), (org.objectweb.fractal.api.Interface) targetNode,
                        "service");

                final Component activityComp = SCAHelper.getSCAHelper().getComponentByInterface(
                        this.getComponent(), (org.objectweb.fractal.api.Interface) behaviour,
                        "service");

                if ((targetComp == null) || (activityComp == null)) {
                    throw new CoreException("Impossible to link node to a transition");
                }

                // Add binding between target node and behaviour
                List<Binding> listOfBindings = new ArrayList<Binding>();
                listOfBindings.add(new Binding("behaviour",
                        (org.objectweb.fractal.api.Interface) activityComp
                        .getFcInterface("service")));
                SCAHelper.getSCAHelper().addBindings(targetComp, listOfBindings);

                // Add binding between behaviour and target node
                listOfBindings = new ArrayList<Binding>();
                listOfBindings
                .add(new Binding("node", (org.objectweb.fractal.api.Interface) targetComp
                        .getFcInterface("service")));
                SCAHelper.getSCAHelper().addBindings(activityComp, listOfBindings);
            }
        } catch (final NoSuchInterfaceException e) {
            log.fine(e.getMessage());
            throw new CoreException(e);
        } catch (final SCAException e) {
            throw new CoreException(e);
        }
    }

    public void unlinkedNodeAndFunctionnalBehaviour(final Node targetNode, final Behaviour behaviour)
            throws CoreException {
        try {
            if ((behaviour != null) && (targetNode != null)) {
                final Component targetComp = SCAHelper.getSCAHelper().getComponentByInterface(
                        this.getComponent(), (org.objectweb.fractal.api.Interface) targetNode,
                        "service");

                final Component activityComp = SCAHelper.getSCAHelper().getComponentByInterface(
                        this.getComponent(), (org.objectweb.fractal.api.Interface) behaviour,
                        "service");

                if ((targetComp == null) || (activityComp == null)) {
                    throw new CoreException("Impossible to link node to a transition");
                }

                // Add binding between target node and behaviour
                List<Binding> listOfBindings = new ArrayList<Binding>();
                listOfBindings.add(new Binding("behaviour",
                        (org.objectweb.fractal.api.Interface) activityComp
                        .getFcInterface("service")));
                SCAHelper.getSCAHelper().deleteBindings(targetComp, listOfBindings);

                // delete binding between behaviour and target node
                listOfBindings = new ArrayList<Binding>();
                listOfBindings
                .add(new Binding("node", (org.objectweb.fractal.api.Interface) targetComp
                        .getFcInterface("service")));
                SCAHelper.getSCAHelper().deleteBindings(activityComp, listOfBindings);
            }
        } catch (final NoSuchInterfaceException e) {
            ScopeImpl.log.fine(e.getMessage());
            throw new CoreException(e);
        } catch (final SCAException e) {
            throw new CoreException(e);
        }
    }

    public Node unlinkBrotherNodes(final Node sourceNode, final Node targetNode)
            throws CoreException {

        Node res = null;
        try {

            Map<String, Object> mapOfBindings = SCAHelper.getSCAHelper().getListOfBindings(
                    sourceNode.getComponent());
            Node afterNode = null;
            String afterName = null;
            for (Entry<String, Object> e : mapOfBindings.entrySet()) {
                if (e.getKey().startsWith("outgoingNodes")) {

                    afterName = e.getKey();
                    afterNode = (Node) e.getValue();
                    break;
                }
            }

            res = afterNode;

            // TODO This returns every component linked to sourceNode. How to
            // get only the one who is the outgoinnode ?
            List<Component> cs = SCAHelper.getSCAHelper()
                    .getClientComponentsLinkedToServerInterfacesOfComponent(this.getComponent(),
                            (Interface) sourceNode.getComponent().getFcInterface("service"));
            List<Boolean> started = new ArrayList<Boolean>();
            for (Component css : cs) {
                if (!SCAHelper.getSCAHelper().isStarted(css)) {
                    SCAHelper.getSCAHelper().startComponent(css);
                    started.add(Boolean.FALSE);
                } else {
                    started.add(Boolean.TRUE);
                }
            }

            Binding b = new Binding(afterName, (Interface) afterNode.getComponent().getFcInterface(
                    "service"));
            List<Binding> bindToDelete = new ArrayList<Binding>();
            bindToDelete.add(b);
            SCAHelper.getSCAHelper().deleteBindings(sourceNode.getComponent(), bindToDelete);

            // TODO This returns every component linked to sourceNode. How to
            // get only the one who is the outgoinnode ?
            int i = 0;
            for (Component css : cs) {
                if (!started.get(i)) {
                    SCAHelper.getSCAHelper().stopComponent(css);
                }
                i++;
            }

        } catch (SCAException e) {
            throw new CoreException(e);
        } catch (NoSuchInterfaceException e) {
            throw new CoreException(e);
        }

        return res;
    }

    public void linkedBrotherNodes(final Node sourceNode, final Node targetNode)
            throws CoreException {
        try {
            final Component sourceComp = SCAHelper.getSCAHelper().getComponentByInterface(
                    this.getComponent(), (org.objectweb.fractal.api.Interface) sourceNode,
                    "service");
            final Component targetComp = SCAHelper.getSCAHelper().getComponentByInterface(
                    this.getComponent(), (org.objectweb.fractal.api.Interface) targetNode,
                    "service");

            if ((sourceComp == null) || (targetComp == null)) {
                throw new CoreException("Impossible to link brother nodes");
            }

            // Add binding between source and target
            List<Binding> listOfBindings = new ArrayList<Binding>();
            listOfBindings.add(new Binding("incomingNodes_"
                    + this.formatCounter(ScopeImpl.cpt_incomingNode),
                    (org.objectweb.fractal.api.Interface) sourceComp.getFcInterface("service")));
            SCAHelper.getSCAHelper().addBindings(targetComp, listOfBindings);
            ScopeImpl.cpt_incomingNode++;

            // Add binding between target and source
            listOfBindings = new ArrayList<Binding>();
            listOfBindings.add(new Binding("outgoingNodes"
                    + this.formatCounter(ScopeImpl.cpt_outgoingNode),
                    (org.objectweb.fractal.api.Interface) targetComp.getFcInterface("service")));
            SCAHelper.getSCAHelper().addBindings(sourceComp, listOfBindings);
            ScopeImpl.cpt_outgoingNode++;

        } catch (final NoSuchInterfaceException e) {
            throw new CoreException(e);
        } catch (final SCAException e) {
            throw new CoreException(e);
        }
    }


    public void linkedChildNodeToParent(final Node parentNode, final Node childNode)
            throws CoreException {
        try {
            final Component parentComp = SCAHelper.getSCAHelper().getComponentByInterface(
                    this.getComponent(), (org.objectweb.fractal.api.Interface) parentNode,
                    "service");
            final Component childComp = SCAHelper.getSCAHelper().getComponentByInterface(
                    this.getComponent(), (org.objectweb.fractal.api.Interface) childNode,
                    "service");

            if ((parentComp == null) || (childComp == null)) {
                throw new CoreException("Impossible to link parent and children node");
            }

            // Add binding between parent and child
            List<Binding> listOfBindings = new ArrayList<Binding>();
            listOfBindings.add(new Binding("childNodes_"
                    + this.formatCounter(ScopeImpl.cpt_childsNode),
                    (org.objectweb.fractal.api.Interface) childComp.getFcInterface("service")));
            SCAHelper.getSCAHelper().addBindings(parentComp, listOfBindings);
            ScopeImpl.cpt_childsNode++;

            // Add binding between child and parent
            listOfBindings = new ArrayList<Binding>();
            listOfBindings.add(new Binding("parentNode",
                    (org.objectweb.fractal.api.Interface) parentComp.getFcInterface("service")));
            SCAHelper.getSCAHelper().addBindings(childComp, listOfBindings);

            SCAContentController scacc = (SCAContentController) childComp
                    .getFcInterface(SCAContentController.NAME);
            NodeImpl nodeContent = null;
            try {
                nodeContent = (NodeImpl) scacc.getFcContent();
            } catch (ContentInstantiationException e) {
                throw new CoreException(e);
            }
            if (nodeContent.getParentNode() == null) {
                nodeContent.setParentNode(parentNode);
            }

        } catch (final NoSuchInterfaceException e) {
            throw new CoreException(e);
        } catch (final SCAException e) {
            throw new CoreException(e);
        }
    }

    public void linkedChildNodeToParent2(final Node parentNode, final Node childNode)
            throws CoreException {
        try {
            final Component parentComp = SCAHelper.getSCAHelper().getComponentByInterface(
                    this.getComponent(), (org.objectweb.fractal.api.Interface) parentNode,
                    "service");
            final Component childComp = SCAHelper.getSCAHelper().getComponentByInterface(
                    this.getComponent(), (org.objectweb.fractal.api.Interface) childNode,
                    "service");

            if ((parentComp == null) || (childComp == null)) {
                throw new CoreException("Impossible to link parent and children node");
            }

            // Add binding between parent and child
            List<Binding> listOfBindings = new ArrayList<Binding>();
            listOfBindings.add(new Binding("childNodes_"
                    + this.formatCounter(ScopeImpl.cpt_childsNode),
                    (org.objectweb.fractal.api.Interface) childComp.getFcInterface("service")));
            SCAHelper.getSCAHelper().addBindings(parentComp, listOfBindings);
            ScopeImpl.cpt_childsNode++;

        } catch (final NoSuchInterfaceException e) {
            throw new CoreException(e);
        } catch (final SCAException e) {
            throw new CoreException(e);
        }
    }



    public void linkedExecutableElement2Execution(final Node source, final Execution exe)
            throws CoreException {
        try {
            final Component sourceComp = SCAHelper.getSCAHelper().getComponentByInterface(
                    this.getComponent(), (org.objectweb.fractal.api.Interface) source, "service");

            final Component exeComp = SCAHelper.getSCAHelper().getComponentByInterface(
                    this.getComponent(), (org.objectweb.fractal.api.Interface) exe, "service");

            if ((exeComp == null) || (sourceComp == null)) {
                throw new CoreException("Impossible to link the axecution and the target node");
            }
        } catch (final SCAException e) {
            throw new CoreException(e);
        }
    }

    public Map<String, Variable> getVariables() {
        return this.variables;
    }

    public Variable findVariable(String name) throws CoreException {
        Variable v = null;
        v = this.variables.get(name);

        if (v == null) {
            Scope current = this.getParentScope(this);
            while ((v == null) && (current != null)) {
                v = current.getVariables().get(name);
                current = this.getParentScope(current);
            }
        }

        return v;
    }

    public final void getInScopeVariables(Map<String, Variable> result) throws CoreException {
        Scope parentScope = getParentScope();
        if (parentScope != null) {
            parentScope.getInScopeVariables(result);
        }
        result.putAll(getVariables());
    }

    @Override
    public Map<String, Variable> getInScopeVariables() throws CoreException {
        Map<String, Variable> inScopeVariables = new HashMap<String, Variable>();
        getInScopeVariables(inScopeVariables);
        return inScopeVariables;
    }

    public List<CorrelationGroup> findCorrelationGroups(final String name) throws CoreException {
        List<CorrelationGroup> res = new ArrayList<CorrelationGroup>();

        for (CorrelationGroup correlationGroup : correlationGroups) {
            if (correlationGroup.getCorrelations().get(name) != null) {
                res.add(correlationGroup);
            }
        }

        Scope current = this.getParentScope(this);
        if (current != null) {
            res.addAll(current.findCorrelationGroups(name));
            current = this.getParentScope(current);
        }

        return res;
    }

    @Override
    public Partner findPartner(final String partnerName) throws CoreException {
        Partner ep = null;
        ep = this.partners.get(partnerName);

        if (ep == null) {
            Scope current = this.getParentScope(this);
            while ((ep == null) && (current != null)) {
                ep = current.getPartners().get(partnerName);
                current = this.getParentScope(current);
            }
        }

        return ep;
    }

    public <B extends Behaviour> List<B> findBehaviours(Class<B> behaviourClass)
            throws CoreException {
        return findBehaviours(behaviourClass, this.getComponent(), false);
    }

    public <B extends Behaviour> List<B> findBehaviours(Class<B> behaviourClass,
            boolean ignoreStoppedComponent) throws CoreException {
        return findBehaviours(behaviourClass, this.getComponent(), ignoreStoppedComponent);
    }

    @SuppressWarnings("unchecked")
    public static <B extends Behaviour> List<B> findBehaviours(Class<B> behaviourClass,
            Component comp, boolean ignoreStopComponent) throws CoreException {
        List<B> behaviours = new ArrayList<B>();
        HashSet<B> behavioursSet = new HashSet<B>();
        try {
            List<Component> components = SCAHelper.getSCAHelper().getAllSubComponents(comp);
            for (Component component : components) {

                boolean isStarted = SCAHelper.getSCAHelper().isStarted(component);
                try {

                    B behaviour = (B) component.getFcInterface("service");

                    if (InstanceOfUtil.isClassExtendOfClass2found(behaviour.getClass(),
                            behaviourClass)) {
                        if (ignoreStopComponent == true) {
                            SCAHelper.getSCAHelper().startComponent(component);
                            behavioursSet.add(behaviour);
                        } else if (isStarted) {
                            behavioursSet.add(behaviour);
                        }
                    }

                } catch (NoSuchInterfaceException e1) {
                    // do nothing
                } catch (ClassCastException e1) {
                    // do nothing
                }

                try {

                    if (component.getFcInterface("service") instanceof Scope) {
                        boolean wasStarted = SCAHelper.getSCAHelper().isStarted(component);
                        if (!wasStarted) {
                            SCAHelper.getSCAHelper().startComponent(component);
                        }
                        behavioursSet.addAll(findBehaviours(behaviourClass, component,
                                ignoreStopComponent));

                        if (!wasStarted) {
                            SCAHelper.getSCAHelper().stopComponent(component);
                        }
                    }

                } catch (NoSuchInterfaceException e) {
                    // do nothing
                }

            }
        } catch (final SCAException e) {
            throw new CoreException(e);
        }

        for (B b : behavioursSet) {
            behaviours.add(b);
        }
        return behaviours;
    }

    public void setVariable(Variable v) throws CoreException {
        Scope current = this;
        Variable var = current.getVariables().get(v.getName());
        if (var == null) {
            while ((var == null) && (current != null)) {
                var = current.getVariables().get(v.getName());
                current = this.getParentScope();
            }
        }

        if (current != null) {
            current.getVariables().put(v.getName(), v);
        } else {
            this.getVariables().put(v.getName(), v);
        }
    }

    public Scope getParentScope() throws CoreException {
        Scope parent = null;
        try {
            if (!(this instanceof com.ebmwebsourcing.easyviper.core.api.engine.Process)) {
                final Component parentComp = SCAHelper.getSCAHelper()
                        .getParent(this.getComponent());
                if (parentComp != null) {
                    try {
                        parent = (com.ebmwebsourcing.easyviper.core.api.engine.Process) parentComp
                                .getFcInterface("service");
                    } catch (final NoSuchInterfaceException e) {
                        // parent = (Scope) parentComp.getFcInterface("scope");
                        parent = (Scope) parentComp.getFcInterface("service");
                    } catch (final ClassCastException e) {
                        // parent = (Scope) parentComp.getFcInterface("scope");
                        parent = (Scope) parentComp.getFcInterface("service");
                    }
                }
            }
        } catch (final NoSuchInterfaceException e) {
            throw new CoreException(e);
        } catch (final SCAException e) {
            throw new CoreException(e);
        }
        return parent;
    }

    private Scope getParentScope(Scope child) throws CoreException {
        Scope parent = null;
        try {
            if (!(child instanceof com.ebmwebsourcing.easyviper.core.api.engine.Process)) {
                final Component parentComp = SCAHelper.getSCAHelper().getParent(
                        child.getComponent());
                if (parentComp != null) {
                    parent = (Scope) parentComp.getFcInterface("service");
                }
            }
        } catch (final NoSuchInterfaceException e) {
            // do nothing
        } catch (final SCAException e) {
            // do nothing
        }
        return parent;
    }

    @Override
    public void setLog(final Logger logger) {
        super.setLog(logger);
        ScopeImpl.log = logger;
    }

    private String formatCounter(int cpt) {
        String res = "";
        if (cpt < 10) {
            res = "000" + cpt;
        } else if (cpt < 100) {
            res = "00" + cpt;
        } else if (cpt < 1000) {
            res = "0" + cpt;
        } else {
            res = String.valueOf(cpt);
        }
        return res;
    }

    public Map<String, Partner> getPartners() {
        return partners;
    }

    public List<CorrelationGroup> getCorrelationGroups() {
        return correlationGroups;
    }

    @Override
    public final Map<Fault, Scope> getExceptions() {
        return Collections.unmodifiableMap(this.exceptions);
    }


    @Override
    public final void addException(Fault fault, Scope faultScope) {
        this.exceptions.put(fault, faultScope);
    }

    public Exception getException() {
        return this.faultScope;
    }

    @Override
    public Node getParentNode() {
        return this.parentNode;
    }

    @Override
    public void setParentNode(Node parentNode) {
        this.parentNode = parentNode;
    }

    public void setException(Exception faultScope) {
        System.err.println("$$$$$$$$$$$$$$$$$ setException(" + (faultScope == null ? "null" : faultScope.getClass()) + ") on " + getName());
        this.faultScope = faultScope;
    }

    public void setTerminationHandler(TerminationHandler th) {
        this.terminationHandler = th;
    }

    public TerminationHandler getTerminationHandler() {
        return this.terminationHandler;
    }

    @Override
    public final ExpressionEvaluator getExpressionEvaluator() {
        return expressionEvaluator;
    }

    @Override
    public final void setExpressionEvaluator(ExpressionEvaluator expressionEvaluator) {
        this.expressionEvaluator = expressionEvaluator;
    }

    @Override
    public final PartnerEvaluator getPartnerEvaluator() {
        return partnerEvaluator;
    }

    @Override
    public final void setPartnerEvaluator(PartnerEvaluator partnerEvaluator) {
        this.partnerEvaluator = partnerEvaluator;
    }


  
}
