/****************************************************************************
 *
 * Copyright (c) 2009-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/
 
package com.ebmwebsourcing.easyviper.extended.service.behaviour.impl.util.jarLoader;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Map.Entry;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarInputStream;
import java.util.jar.Manifest;
import java.util.logging.Logger;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/**
 * @author Nicolas Salatge - EBM WebSourcing
 */
public class JarLoader extends URLClassLoader {

	private List<Class<?>> classList = new ArrayList<Class<?>> ();
	private Map<String, List<String>> resources = new HashMap<String, List<String>>();
	private Logger log = Logger.getLogger(JarLoader.class.getCanonicalName());


	public JarLoader(URL url, ClassLoader parent) throws JarException {
		this(url, getAllUrlsInManifest(url), parent);
	}

	private JarLoader(URL parentUrl, URL[] urlsInManifest, ClassLoader parent) {
		super(urlsInManifest, parent);
		load(parentUrl);
	}

	private static URL[] getAllUrlsInManifest(URL url) throws JarException {
		List<URL> urls = new ArrayList<URL>();
		if(url != null) {
			JarInputStream jin = null;
			try {
				Manifest manifest = null;

				File file;
				try {
					file = new File(url.toURI());
				} catch(URISyntaxException e) {
					file = new File(url.getPath());
				}
				jin = new JarInputStream(new FileInputStream(file));

				manifest = jin.getManifest();


				if(manifest != null) {
					Attributes attr =  manifest.getMainAttributes();
					List<String> classpath = new ArrayList<String>();
					if(attr != null && attr.getValue("Class-Path") != null) {

						StringTokenizer st = new StringTokenizer(attr.getValue("Class-Path"), " ");

						while(st.hasMoreElements()) {
							classpath.add(file.getAbsolutePath().substring(0,file.getAbsolutePath().lastIndexOf(File.separatorChar)+1) + st.nextToken());
						}

					}

					classpath.add(0, file.toString());

					// reverse order
					for(String fileS: classpath) {
						urls.add(0, new File(fileS).toURI().toURL());
					}
				}
			} catch (FileNotFoundException e) {
				throw new JarException(e);
			} catch (IOException e) {
				throw new JarException(e);
			} finally {
				if (jin!=null) {
					try {
						jin.close();
					} catch (IOException e) {
						// Never mind
					}
				}
			}
		}
		URL[] res = new URL[urls.size()];
		return urls.toArray(res);
	}

	/**
	 * retourne la liste des classe lues
	 * 
	 * @return Vector<Class<?>>
	 */
	public List<Class<?>> getClassList() {
		return this.classList;
	}

	public Map<String, List<String>> getResources() {
		return resources;
	}

	/**
	 * load jar
	 */
	private void load(URL url) {
		JarInputStream jin = null;
		try {

			Manifest manifest = null;

			File file;
			try {
				file = new File(url.toURI());
			} catch(URISyntaxException e) {
				file = new File(url.getPath());
			}
			jin = new JarInputStream(new FileInputStream(file));

			/**
			 * We want to store classes in an array.
			 * Then we have to retrieve their sizes 
			 */

			ZipFile zf = new ZipFile(file);
			Enumeration<? extends ZipEntry> zen = zf.entries();
			ZipEntry e;
			Hashtable<String, Integer>	sizeOfClass		= new Hashtable<String, Integer>();
			while (zen.hasMoreElements()) {
				e = zen.nextElement();
				String name = e.getName();
				sizeOfClass.put(name.replace("\\", "/"), (int) e.getSize());

			}
			zf.close();



			// we retrieve the class names

			Hashtable<String, byte[]>	hs = new Hashtable<String, byte[]>();
			JarEntry je = null;
			while ((je = jin.getNextJarEntry()) != null) {
				String name = je.getName().replace("\\", "/");
				int size = sizeOfClass.get(name);
				byte[] b = new byte[size];
				int byte_read = 0;

				if (manifest == null && name.equals("META-INF/MANIFEST.MF")) {
					manifest = new Manifest();
					while (byte_read != size) {
						int blu = jin.read(b, byte_read, size - byte_read);
						if (blu > 0)
							byte_read += blu;
					}
					manifest.read(new ByteArrayInputStream(b));
				} else {
					if (!name.endsWith(".class")) {
						addResources(name);
						continue;
					}
					while (byte_read != size) {
						int blu = jin.read(b, byte_read, size - byte_read);
						if (blu > 0)
							byte_read += blu;
					}

					hs.put(name, b);
					//				}
				}
			}
			this.defineClassList(hs);

		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			try {
				// on ferme le flux
				if (jin!=null){
					jin.close();
				}
			} catch (IOException ioe) {
				// Never mind
			}
		}

	}


	private void addResources(String name) {
		if(name.contains(".")) {
			String extension = name.substring(name.indexOf(".")+1, name.length());
			if(extension.length() == 3) {
				List<String> urls = this.resources.get(extension.toLowerCase());
				if(urls == null) {
					urls = new ArrayList<String>();
				}
				urls.add(name);
				this.resources.put(extension, urls);
			}
		}
	}

	/**
	 * recupere la liste des classes contenues dans le jar
	 * 
	 * @return un Vector<Class<?>>
	 */
	public List<Class<?>> defineClassList(Hashtable<String, byte[]>	hs) {
		Enumeration<String> en = hs.keys();

		Map<String, byte[]> classes2reload = new HashMap<String, byte[]>();
		while (en.hasMoreElements()) {
			String s = en.nextElement();
			byte[] b = hs.get(s);

			if(s.endsWith(".class")) {
				// on degage le .class
				String name = s.substring(0, s.indexOf('.'));
				name = name.replace("/", ".");
				classes2reload.put(name, b);
			}
		}

		classes2reload = classes2reload(classes2reload);
		int cpt = 0;
		while(classes2reload.size() > 0 && cpt < 100) {
			classes2reload = classes2reload(classes2reload);
			cpt ++;
		}
		return getClassList();
	}

	private Map<String, byte[]> classes2reload(Map<String, byte[]> classes2reload) {
		Map<String, byte[]> res = new HashMap<String, byte[]>();
		Iterator<Entry<String, byte[]>> it = classes2reload.entrySet().iterator();
		Class<?> c = null;
		while(it.hasNext()) {
			Entry<String, byte[]> entry = it.next();
			try {
				c = this.defineClass(entry.getKey(), entry.getValue(), 0, entry.getValue().length);
			} catch(LinkageError e) {
				log.warning("Impossible to load: " + e.getMessage());

				
				//if((e.getMessage().contains("attempted  duplicate class definition for name")) && classList.contains(entry.getKey())){
				try {
					System.err.println("entry.getKey() = " + entry.getKey());
					c = this.loadClass(entry.getKey());
				} catch (ClassNotFoundException e1) {
					// TODO Auto-generated catch block
					//e1.printStackTrace();
					res.put(entry.getKey(), entry.getValue());
				} catch (NoClassDefFoundError e1) {
					// TODO Auto-generated catch block
					//e1.printStackTrace();
					res.put(entry.getKey(), entry.getValue());
				}

			}


			if((c != null)&&!classList.contains(entry.getKey())) {
				classList.add(c);
			}
			c = null;
		}
		return res;
	}


	/**
	 * appeler une methode en connaissant les types des argument
	 * 
	 * @param c
	 *            la classe
	 * @param inst
	 *            une instance de classe
	 * @param method
	 *            nom de la methode
	 * @param type
	 *            type des argument
	 * @param args
	 *            arguments
	 * @return
	 * @throws SecurityException
	 * @throws NoSuchMethodException
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 */
	public static Object callMethod(Class<?> c, Object inst, String method, Object... args) throws JarException {
		Object res = null; 
		try {
			Method m = null;
			for(int i = 0; i < c.getMethods().length; i++) {
				if(c.getMethods()[i].getName().equals("setMessage")) {
					m = c.getMethods()[i];
					break;
				}
			}
			res = m.invoke(inst, args);
		} catch (IllegalArgumentException e) {
			throw new JarException(e);
		} catch (IllegalAccessException e) {
			throw new JarException(e);
		} catch (InvocationTargetException e) {
			throw new JarException(e);
		}
		return res;

	}


	public String toString() {
		String res = "";
		for(Class<?> clazz: this.getClassList()) {
			res = clazz.getName() + "\n";
		}
		return res;
	}
}
