/**
 * geasy-diagram-editor - A project for editing diagrams based on OMG Diagram Definition standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.geasytools.diagrameditor;

import java.util.Iterator;

import com.ebmwebsourcing.geasytools.diagrameditor.api.events.builder.IBuildDiagramInterchangeCompleteEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.api.events.builder.IModelBuilderHandler;
import com.ebmwebsourcing.geasytools.diagrameditor.api.graphic.IDiagramElementView;
import com.ebmwebsourcing.geasytools.diagrameditor.api.graphic.IDiagramView;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.diagramcommon.layout.Point;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IDiagram;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IDiagramElement;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IEdge;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IModelElement;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IShape;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.IConnector;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.IConnectorPoint;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.IConnectorStart;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.events.IAddIntermediateConnectorPointEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.events.IAddWayPointEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.events.IConnectionEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.events.IDisconnectionEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.events.IRefreshEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.events.IRemoveIntermediateConnectorPointEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.events.IRemoveWayPointEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.core.IContainer;
import com.ebmwebsourcing.geasytools.geasyui.api.core.events.IAddUIElementEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.core.events.IBoundsUpdateHandler;
import com.ebmwebsourcing.geasytools.geasyui.api.core.events.IContainerHandler;
import com.ebmwebsourcing.geasytools.geasyui.api.core.events.IPositionChangeEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.core.events.ISizeChangeEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.uipanel.events.INewUIElementEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.uipanel.events.IRemoveUIElementEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.uipanel.events.IResizeRequestEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.uipanel.events.ISelectedElementsEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.uipanel.events.IUIPanelHandler;
import com.ebmwebsourcing.geasytools.geasyui.api.uipanel.events.IUnselectedElementsEvent;
import com.ebmwebsourcing.geasytools.geasyui.impl.connectable.events.ConnectorHandler;
import com.ebmwebsourcing.geasytools.geasyui.impl.core.UIPanelVisitor;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.modelbinder.IWatchedModelProxy;
import com.google.gwt.user.client.Window;

/**
 * Builds a Diagram Object based on user interactions only.
 * Adds default handlers to DiagramUI to build a default Diagram Interchange.
 * 
 * @author nfleury
 *
 */
public class DiagramInterchangeModelBuilder {

	private IDiagramView diagramView;
	private DiagramController diagramController;

	protected IUIPanelHandler panelHandler;


	public DiagramInterchangeModelBuilder(IDiagramView diagram, DiagramController diagramController) {
		this.diagramView = diagram;
		this.diagramController = diagramController;

		//initialize diagrams models
		this.initialize();

		addListeners();
	}

	public IDiagram getDiagramInterchangeModel(){		
		return diagramView.getDiagram();
	}

	private void initialize(){

	}


	private void addListeners(){
		this.panelHandler = new IUIPanelHandler() {

			@Override
			public void onUnSelectUIElement(IUnselectedElementsEvent unselectedElementsEvent) {

			}

			@Override
			public void onSelectUIElement(ISelectedElementsEvent selectedElementsEvent) {

			}

			@Override
			public void onNewUIElement(INewUIElementEvent addElementEvent) {
				//1- When adding an element we 
				//a- add it to diagrams model plane
				//b- register the editors model to actual undoredosession
				//c- assign model element to diagramElement
				if (addElementEvent.getNewElement() instanceof IDiagramElementView){
					final IDiagramElementView newDiView = (IDiagramElementView) addElementEvent.getNewElement();

					IDiagramElement diElement = newDiView.getDiagramElement();

					//a - add diagram element to plane only if parent is NULL;
					if (diElement.getOwningElement()==null) getDiagramInterchangeModel().addDiagramElement(diElement);

					//b - register editor model to undoredosession
					//undoRedoSession.registerModel(diElement);					

					//c- assign model element to diagramElement
					IWatchedModelProxy<? extends IModelElement> proxy = (IWatchedModelProxy<? extends IModelElement>) newDiView.getEditorModel();
					diElement.setModelElement(proxy.getBindedModel());

					//2- If actual diagram element is a container add 
					//receive di into di container
					if (newDiView instanceof IContainer){
						IContainer container = (IContainer) newDiView;

						container.addContainerHandler(new IContainerHandler() {

							@Override
							public void onRemoveUIElement(
									com.ebmwebsourcing.geasytools.geasyui.api.core.events.IRemoveUIElementEvent event) {

								if (event.getUIElement() instanceof IDiagramElementView){
									IDiagramElementView removedElement = (IDiagramElementView) event.getUIElement() ;
									newDiView.getDiagramElement().removeDiagramElement(removedElement.getDiagramElement());
								}
							}

							@Override
							public void onAddUIElement(IAddUIElementEvent event) {
								if (event.getUIElement() instanceof IDiagramElementView) {
									IDiagramElementView addedElement = (IDiagramElementView) event.getUIElement() ;
									newDiView.getDiagramElement().addDiagramElement(addedElement.getDiagramElement());
									addedElement.getDiagramElement().setOwningElement(newDiView.getDiagramElement());
								}
							}
						});

					}


					//4- Add bound update handlers if actual DI is a Shape
					//so that we update corresponding di bounds
					if (newDiView.getDiagramElement() instanceof IShape){
						final IShape shape  = (IShape) newDiView.getDiagramElement();

						newDiView.addBoundUpdateHandler(new IBoundsUpdateHandler() {
							@Override
							public void onSizeChange(ISizeChangeEvent event) {
								shape.getBounds().setWidth(event.getNewWidth());
								shape.getBounds().setHeight(event.getNewHeight());
							}

							@Override
							public void onPositionChange(IPositionChangeEvent event) {
								shape.getBounds().setX(event.getNewRelativeX());
								shape.getBounds().setY(event.getNewRelativeY());
							}
						});
					}

					if (newDiView instanceof IConnector && diElement instanceof IEdge) {
						final IEdge edge = (IEdge) diElement;

						((IConnector)addElementEvent.getNewElement()).addConnectorHandler(new ConnectorHandler() {
							@Override
							public void onConnection(IConnectionEvent event) {
								if(event.getConnectionSource()!=null && event.getConnectionSource() instanceof IDiagramElementView) {
									edge.setSource(((IDiagramElementView)event.getConnectionSource()).getDiagramElement());
								}
								if(event.getConnectionTarget()!=null
										&& event.getConnectionTarget() instanceof IDiagramElementView) {
									edge.setTarget(((IDiagramElementView)event.getConnectionTarget()).getDiagramElement());
								}
							}

							@Override
							public void onAddWayPointRequest(
									IAddWayPointEvent event) {
								// TODO Auto-generated method stub
							}

							@Override
							public void onRemoveWayPoint(
									IRemoveWayPointEvent event) {
								// TODO Auto-generated method stub
							}

							@Override
							public void onRefresh(IRefreshEvent event) {
								// TODO Auto-generated method stub
							}

							@Override
							public void onAddIntermediateConnectorPoint(
									IAddIntermediateConnectorPointEvent event) {
								//TODO
							}

							@Override
							public void onRemoveIntermediateConnectorPoint(
									IRemoveIntermediateConnectorPointEvent event) {
								//TODO
							}

							@Override
							public void onDisconnection(
									IDisconnectionEvent event) {
								if(event.getDisconnectedSource()!=null) {
									edge.setSource(null);
									if(event.getDisconnectedSource() instanceof IDiagramElementView) {
										((IDiagramElementView)event.getDisconnectedSource()).getDiagramElement().removeOutgoingEdge(edge);
									}
								}
								if(event.getDisconnectedTarget()!=null) {
									edge.setTarget(null);
									if(event.getDisconnectedTarget() instanceof IDiagramElementView) {
										((IDiagramElementView)event.getDisconnectedTarget()).getDiagramElement().removeIncomingEdge(edge);
									}
								}
							}
						});
					}
				}
			}


			@Override
			public void onResizeRequest(IResizeRequestEvent event) {

			}

			@Override
			public void onRemoveUIElement(IRemoveUIElementEvent removeElementEvent) {
				IDiagramElementView elView = (IDiagramElementView) removeElementEvent.getRemovedElement();
				getDiagramInterchangeModel().getRootElement().removeDiagramElement(elView.getDiagramElement());
			}
		};



		this.diagramController.addModelBuilderHandler(new IModelBuilderHandler() {

			@Override
			public void onBuildDiagramInterchangeComplete(
					IBuildDiagramInterchangeCompleteEvent event) {

				IDiagramView diagramView = event.getDiagramView();

				new UIPanelVisitor(diagramView) {
					@Override
					protected void visitConnector(IConnector connector) {
						//build waypoints
						IDiagramElementView view = (IDiagramElementView) connector;
						IEdge edge = (IEdge) view.getDiagramElement();

						//remove previous waypoints from current edge
						edge.clearWayPoints();

						//Add new ones
						Iterator<IConnectorPoint> it = connector.getAllConnectorPoints().iterator();
						IConnectorPoint first = null;
						while (it.hasNext() && first==null) {
							IConnectorPoint current = it.next();
							if(current instanceof IConnectorStart) {
								first = current;
							}
						}
						while(first!=null) {
							Point wayPoint = new Point(first.getRelativeX()+first.getUIPanel().getScrollLeft()+Window.getScrollLeft(),
									first.getRelativeY()+first.getUIPanel().getScrollTop()+Window.getScrollTop());
							edge.addWayPoint(wayPoint);
							first = first.getNextPoint();
						}
						

						///////Set source / target
						IDiagramElementView sourceView = (IDiagramElementView) connector.getSource();
						if(sourceView!=null) {
							edge.setSource(sourceView.getDiagramElement());
						}
						
						IDiagramElementView targetView = (IDiagramElementView) connector.getTarget();
						if(targetView!=null) {
							edge.setTarget(targetView.getDiagramElement());
						}
					}
				};
			}
		});		

		diagramView.addUIPanelHandler(panelHandler);		
	}



}
