/**
 * geasy-diagram-editor - A project for editing diagrams based on OMG Diagram Definition standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.ebmwebsourcing.geasytools.diagrameditor.impl.syntax;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.ebmwebsourcing.geasytools.diagrameditor.api.syntax.ISyntaxModelBuilder;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IDiagramElement;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IEdge;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IModelElement;

public abstract class SyntaxModelBuilder<T extends IModelElement> implements ISyntaxModelBuilder<T>{
	
	private IDiagramElement mainDiagramElement;
	private SyntaxModelBuilder<?> parentSyntaxBuilder;
	
	private HashMap<Class<?>,SyntaxModelBuilder<?>> builders;
	
	public SyntaxModelBuilder(IDiagramElement diagramElement) {
		this();
		this.mainDiagramElement = diagramElement;
	}
	
	public SyntaxModelBuilder() {
		this.builders = new HashMap<Class<?>, SyntaxModelBuilder<?>>();
	}
	
	public void setParentSyntaxBuilder(SyntaxModelBuilder<?> parentSyntaxBuilder) {
		this.parentSyntaxBuilder = parentSyntaxBuilder;
	}
	
	public SyntaxModelBuilder<?> getParentSyntaxBuilder() {
		return parentSyntaxBuilder;
	}
	
	protected void register(SyntaxModelBuilder<?> builder){
		this.builders.put(builder.getSyntaxModelType(), builder);
		builder.setParentSyntaxBuilder(this);
	}
	
	public void setMainDiagramElement(IDiagramElement mainDiagramElement) {
		this.mainDiagramElement = mainDiagramElement;
	}
	
	public <E extends IModelElement> Set<E> getAll(IDiagramElement owningElement,Class<? extends E> type,boolean recursively){
		
		HashSet<E> result = new HashSet<E>();
		
		for(IDiagramElement child:owningElement.getOwnedElements()){

				if (isAssignableTo(child.getModelElement().getClass(), type)) result.add((E) child.getModelElement());
				
				if (recursively){
					
					result.addAll(getAll(child, type, true));
					
				}
				
				//if some builders are registered search
				SyntaxModelBuilder<?> builder = builders.get(child.getModelElement().getClass());
				
				if (builder!=null){
				
					builder.setMainDiagramElement(child);
					E m = (E) builder.getSyntaxModel();
					
					if (isAssignableTo(m.getClass(), type)){
						result.add(m);
					}
					
				}
				
				
		}
		
		
		return result;
	}	
	
	
	
	
	
	public <E extends IModelElement> Set<E> getAll(Class<? extends E> type,boolean recursively){
		return this.getAll(mainDiagramElement, type,recursively);
	}
	
	/**
	 * Returns all the edges of the specified type that are in the scope of current DiagramElement
	 * @param <E>
	 * @param type
	 * @return
	 */
	public <E extends IModelElement> Map<IDiagramElement,E> getEdges(Class<? extends E> type){
		HashMap<IDiagramElement,E> result = new HashMap<IDiagramElement,E>();
		HashSet<IDiagramElement> children = getChildren(mainDiagramElement);

		for(IDiagramElement child:children){
			if(child instanceof IEdge) {
				if (isAssignableTo(child.getModelElement().getClass(), type)) {
					result.put(child,(E) child.getModelElement());
				}
			}
			else {
				//using shapes incoming and outgoing edges is necessary because
				//the edges always are directly owned by the plane
				//--> if the builder is not the plane builder you do not have edges in the children list
				for(IEdge edge : child.getIncomingEdges()) {
					if (isAssignableTo(edge.getModelElement().getClass(), type)) {
						result.put(edge,(E) edge.getModelElement());
					}
				}
				for(IEdge edge : child.getOutgoingEdges()) {
					if (isAssignableTo(edge.getModelElement().getClass(), type)) {
						result.put(edge,(E) edge.getModelElement());
					}
				}
			}
		}
		
		
		return result;
	}
	
	
	public <E extends IModelElement> Set<E> getConnectedTo(Class<? extends E> typeToFilterOn,Class<? extends E> connectorType){
		
		HashSet<E> result = new HashSet<E>();
		
		Map<IDiagramElement,E> edges = this.getEdges(connectorType);
		
		for(IDiagramElement di:edges.keySet()){
			
			IEdge edge = (IEdge) di;
			
			//check in source
			if (isAssignableTo(edge.getSource().getClass(), typeToFilterOn)){
				result.add((E) edge.getSource().getModelElement());
			}
			
			//check in target
			if (isAssignableTo(edge.getTarget().getClass(), typeToFilterOn)){
				result.add((E) edge.getTarget().getModelElement());
			}
			
		}
		
		
		return result;
	}

	
	
	/**
	 * Recursively collect every diagramElements in current element
	 * @param element
	 * @return
	 */
	private HashSet<IDiagramElement> getChildren(IDiagramElement element){
		
		HashSet<IDiagramElement> children = new HashSet<IDiagramElement>();
		
		for(IDiagramElement el:element.getOwnedElements()){
			
			children.add(el);
			children.addAll(getChildren(el));
		}
		
		return children;
	}
	
	
	private boolean isAssignableTo(Class<?> type,Class<?> typeToCheck){
		
		Class<?> currentType = type;
		
		while(currentType!=Object.class){
			
			if (currentType==typeToCheck){
				return true;
			}
			
			currentType = currentType.getSuperclass();
			
		}
		
		return false;
	}
	
	public abstract T getSyntaxModel();
	
	public abstract Class<? extends T> getSyntaxModelType();
	
	public IDiagramElement getMainDiagramElement() {
		return mainDiagramElement;
	}
	
}
