package org.ow2.petals.bc.mail.steps

import org.ow2.petals.groovycommons.util.PlatformDependantObject;
import org.ow2.petals.groovycommons.util.LinuxProcess;
import org.ow2.petals.groovycommons.util.MavenProject;

import static org.jbehave.core.io.CodeLocations.*;


class JamesServer {

    static final def JAMES_CLI_RUN_COMMAND = PlatformDependantObject.create(
    [
        "Linux" : [
            "sh",
            "james-cli.sh"]
    ]);

    static final def JAMES_SERVER_RUN_COMMAND = PlatformDependantObject.create(
    [
        "Linux" : [
            "sh",
            "run.sh"]
    ]);

    static final String executionDir = MavenProject.getExecutionDir();

    static final String JAMES_ZIP_PATH = "${executionDir}/apache-james.zip";
    static final String JAMES_UNZIPPED_DIR_PATH = "${executionDir}/james-server";
    static final String JAMES_UNZIPPED_CONF_DIR_PATH = "${JAMES_UNZIPPED_DIR_PATH}/apache-james-3.0-beta3/conf/";
    static final String JAMES_UNZIPPED_BIN_DIR_PATH = "${JAMES_UNZIPPED_DIR_PATH}/apache-james-3.0-beta3/bin/";

    static final String POP3_SCHEME = "pop3";
    static final String IMAP_SCHEME = "imap";
    static final String SMTP_SCHEME = "smtp";

    def pop3Port = 8110;
    def imapPort = 8143;
    def smtpPort = 8025;


    private boolean isPortOpen(String host, int port) {
        try {
            new Socket(host, port);
            return true;
        } catch (IOException ioe) {
            return false;
        }
    }

    private boolean isLocalPortOpen(int localPort) {
        isPortOpen("localhost", localPort);
    }


    private boolean isRunning() {
        return isLocalPortOpen(pop3Port) && isLocalPortOpen(imapPort) && isLocalPortOpen(smtpPort);
    }


    def launch() {
        def ant = new AntBuilder();
        File f = new File(JAMES_ZIP_PATH);
        ant.delete(dir: JAMES_UNZIPPED_DIR_PATH, failonerror: false);
        ant.unzip(src:JAMES_ZIP_PATH, dest:JAMES_UNZIPPED_DIR_PATH);

        ant.replace(file:"${JAMES_UNZIPPED_CONF_DIR_PATH}/pop3server.xml",
                token:"<bind>0.0.0.0:110</bind>", value:"<bind>0.0.0.0:${pop3Port}</bind>");
        ant.replace(file:"${JAMES_UNZIPPED_CONF_DIR_PATH}/imapserver.xml",
                token:"<bind>0.0.0.0:143</bind>", value:"<bind>0.0.0.0:${imapPort}</bind>");
        ant.replace(file:"${JAMES_UNZIPPED_CONF_DIR_PATH}/smtpserver.xml",
                token:"<bind>0.0.0.0:25</bind>", value:"<bind>0.0.0.0:${smtpPort}</bind>");

        def proc = JAMES_SERVER_RUN_COMMAND.execute(null, new File(JAMES_UNZIPPED_BIN_DIR_PATH));
        while (!isRunning()) {
            sleep(1000);
        }
    }

    def getSchemePort(String scheme) {
        switch (scheme) {
            case POP3_SCHEME:
                return pop3Port;
                break;
            case IMAP_SCHEME:
                return imapPort;
                break;
            case SMTP_SCHEME:
                return smtpPort;
                break;
            default:
                break;
        }
    }

    def runCommand(String[] args) {
        def jamesCliRunCommandWithArgs = (JAMES_CLI_RUN_COMMAND + args).flatten();
        def jamesCliProcess = jamesCliRunCommandWithArgs.execute(null, new File(JAMES_UNZIPPED_BIN_DIR_PATH));
        jamesCliProcess.waitFor();
        if (jamesCliProcess.exitValue() != 0) {
            assert jamesCliProcess.exitValue() == 0;
        }        
        return jamesCliProcess.text;
    }

    private List<String> listDomains() {
        String command = "listdomains";
        return runCommand("-h", "localhost", command).replaceAll("${command} command executed sucessfully.*", "").trim().readLines();
    }

    private List<String> listUsers() {
        String command = "listusers";
        return runCommand("-h", "localhost", command).replaceAll("${command} command executed sucessfully.*", "").trim().readLines();
    }



    private void addDomain(String domain) {
        runCommand("-h", "localhost", "adddomain", domain);
    }


    private void addUser(String user, String password) {
        String command = "adduser";
        runCommand("-h", "localhost", command, user, password);
    }


    private void removeUser(String user) {
        String command = "removeuser";
        runCommand("-h", "localhost", command, user);
    }

    private void removeDomain(String domain) {
        String command = "removedomain";
        runCommand("-h", "localhost", command, domain);
    }


    def reset() {
        listUsers().each({ removeUser(it) });
        listDomains().each({ removeDomain(it) });
    }


    def addMailbox(String user, String password, String domain) {
        def jamesUser = "${user}@${domain}";
        assert (!listUsers().contains(jamesUser));
        if (!listDomains().contains(domain)) {
            addDomain(domain);
        }
        addUser(jamesUser, password);
    }


    def sendEmail(String fromParam, String toParam, String mailSubject,
    String mailContent) {
        def ant = new AntBuilder();
        ant.mail(mailhost: "localhost", mailport: smtpPort, subject: mailSubject) {
            from(address: fromParam)
            to(address: toParam)
            message(mailContent)
        };
    }


    static public void killAll() {
        LinuxProcess.kill("james-server", 9);
    }
}
