/****************************************************************************
 *
 * Copyright (c) 2011-2012, EBM WebSourcing
 * This code is not made public and should not be shared outside EBM WebSourcing. 
 *
 *****************************************************************************/

package org.ow2.petals.components.steps;

import static org.junit.Assert.assertFalse

import java.util.logging.ConsoleHandler
import java.util.logging.Handler
import java.util.logging.LogManager
import java.util.logging.Logger

import javax.xml.namespace.QName

import org.hamcrest.Matchers
import org.jbehave.core.annotations.AfterScenario
import org.jbehave.core.annotations.Alias
import org.jbehave.core.annotations.BeforeScenario
import org.jbehave.core.annotations.BeforeStory
import org.jbehave.core.annotations.Given
import org.jbehave.core.annotations.Named
import org.jbehave.core.annotations.ScenarioType
import org.jbehave.core.annotations.Then
import org.jbehave.core.failures.UUIDExceptionWrapper
import org.jbehave.core.model.ExamplesTable
import org.jbehave.core.model.OutcomesTable
import org.jbehave.core.model.OutcomesTable.OutcomesFailed
import org.ow2.easywsdl.wsdl.api.abstractItf.AbsItfOperation
import org.ow2.petals.commons.log.FlowAttributes
import org.ow2.petals.commons.log.Level
import org.ow2.petals.commons.log.PetalsExecutionContext
import org.ow2.petals.commons.log.TraceCode
import org.ow2.petals.component.api.Component
import org.ow2.petals.component.api.ComponentConfiguration
import org.ow2.petals.component.api.Message
import org.ow2.petals.component.framework.logger.AbstractFlowLogData
import org.ow2.petals.components.stories.util.LogInfoFinder
import org.ow2.petals.components.stories.util.RegexMatcher
import org.ow2.petals.components.stories.util.StoryContext
import org.ow2.petals.log.formatter.BasicLogDataAppender
import org.ow2.petals.log.formatter.LogDataFormatter
import org.ow2.petals.log.handler.PetalsFileHandler
import org.ow2.petals.log.handler.PetalsPayloadDumperFileHandler
import org.ow2.petals.log.handler.TestFileHandler

import com.ebmwebsourcing.easycommons.io.FileSystemHelper
import com.ebmwebsourcing.easycommons.io.IOHelper
import com.ebmwebsourcing.easycommons.lang.UncheckedException
import com.ebmwebsourcing.easycommons.thread.ExecutionContext
import com.ebmwebsourcing.easycommons.xml.XMLComparator
import com.ebmwebsourcing.jbi.adaptor.impl.ComponentType
import com.ebmwebsourcing.jbi.adaptor.impl.WrappedComponent
import com.ebmwebsourcing.jbi.adaptor.impl.WrappedRequestToProviderMessage


public abstract class AbstractComponentSteps extends StoryContext {

    static int count = 0;

    public static final String CDK_NAMESPACE_URI = "http://petals.ow2.org/components/extensions/version-5";
    public static final String OPERATION_PARAM_NAME = new QName(CDK_NAMESPACE_URI, "operation").toString();
    public static final String MEP_PARAM_NAME = new QName(CDK_NAMESPACE_URI, "mep").toString();
    public static final String PETALS_HOME_DIR = FileSystemHelper.getSystemTempDir().getAbsolutePath();

    @AfterScenario(uponType=ScenarioType.NORMAL)
    def stopComponent() {
        if(this.component!=null && component.isStarted()) {
            component.uninstallAllServices();
            component.stop();
        }
    }

    @AfterScenario(uponType=ScenarioType.EXAMPLE)
    def stopComponentAfterEachExample() {
        stopComponent();
    }

    @BeforeScenario(uponType=ScenarioType.NORMAL)
    def beforeScenario() {
        LogManager.logManager.reset();
        this.anotherLogEntry=null;
        this.requestPayload=null;
        this.responsePayload=null;
        this.logEntries=[];
    }

    @BeforeScenario(uponType=ScenarioType.EXAMPLE)
    def beforeScenarioBeforeEachExample() {
        beforeScenario();
    }

    @BeforeStory
    def beforeStory() {
        System.setProperty(PetalsFileHandler.PETALS_LOG_DIR_PROPERTY_NAME, PETALS_HOME_DIR)
    }



    @Given("the ESB only composed of one node")
    def esbOnlyComposedOf() {
        this.nodeName = "node1";
    }

    @Given("the ESB only composed of <nodeHost> known as <nodeName>")
    def esbOnlyComposedOf(String nodeHost, String nodeName) {
        this.nodeHost = nodeHost;
        this.nodeName = nodeName;
    }

    @Given("MONIT enabled on this node")
    def monitmsgLogLevelIsEnabledOnCurrentHost() {
        this.logLevel = Level.MONIT;
        initLogger(this.nodeName, false);
    }

    @Given("MONIT enabled on <nodeName>")
    def monitLogLevelIsEnabledOnHost(String nodeName) {
        this.logLevel = Level.MONIT;
        initLogger(this.nodeName, false);
    }

    @Given("INFO enabled on <nodeName>")
    def infoLogLevelIsEnabledOnHost(String nodeName) {
        this.logLevel = Level.INFO;
        initLogger(this.nodeName, false);
    }

    @Given("MONIT enabled on this node and configured with default configuration to dump payloads")
    def monitLogEnablePayloadDump() {
        this.logLevel = Level.MONIT;
        initLogger(this.nodeName, true);
    }

    @Then("no MONIT record is logged")
    def noMonitRecordLogged() {
        testFileHandler.getLogEntries().each {
            assertFalse(LogInfoFinder.isValidMonitLogTrace(it))
        };
    }

    private def doStopComponent() {
    }

    def private initLogger(nodeName, withPayloadDumper) {
        Logger rootLogger = Logger.getLogger("");

        LogDataFormatter formatter = new LogDataFormatter();
        formatter.setExtraParametersDelimiter("", "");
        ExecutionContext.getProperties().setProperty(PetalsExecutionContext.CONTAINER_NAME_PROPERTY_NAME, nodeName);
        formatter.addLogDataAppender(new BasicLogDataAppender());
        testFileHandler = new TestFileHandler(getClass().getName() + "_" + count++);

        if (withPayloadDumper) {
            PetalsPayloadDumperFileHandler payloadDumperFileHandler = new PetalsPayloadDumperFileHandler();

            rootLogger.addHandler(payloadDumperFileHandler);
        }
        rootLogger.addHandler(testFileHandler);
        rootLogger.addHandler(new ConsoleHandler());
        Handler[] handlers = rootLogger.getHandlers();

        for(Handler handler : handlers) {
            handler.setFormatter(formatter);
            handler.setLevel(logLevel);
        }
    }


    protected ComponentConfiguration createComponentConfiguration(
            String componentName = "",
            List<URL> resources = new ArrayList<URL>()) {
        componentLogger = Logger.getLogger(componentName);
        componentLogger.setLevel(logLevel);
        ComponentConfiguration componentConfiguration = new ComponentConfiguration(componentName, componentLogger);
        if(resources != null && ! resources.isEmpty()) {
            for(URL resource : resources) {
                componentConfiguration.addResource(resource);
            }
        }
        return componentConfiguration;
    }


    protected Component createAndStartComponent(ComponentType componentType,
            ComponentConfiguration componentConfiguration) {

        if (componentType.equals(ComponentType.SE_BPEL)) {
            componentConfiguration.setParameter("{http://petals.ow2.org/components/petals-bpel-engine/version-1.0}internal-logger", "true")
        }
        Component component = new WrappedComponent(componentType, componentConfiguration);
        component.start();
        return component;
    }


    InputStream createPayloadInputStream(String payloadAsString) {
        return new ByteArrayInputStream(payloadAsString.getBytes());
    }

    protected boolean isOutcomesMapVerified(Map<String, String> actual, ExamplesTable expected) {
        OutcomesTable outcomes = new OutcomesTable();
        for (int i = 0 ; i < expected.getRowCount() ; ++i) {
            Map<String, String> row = expected.getRow(i);
            String matcherName = row.get("matcher");
            String key = row.get("key");
            def actualGet = actual.get(key);
            def expectedGetRowAsParameters = expected.getRowAsParameters(i, true);
            String expectedValue = expectedGetRowAsParameters.valueAs("value", String.class);
            if ("equals".equals(matcherName)) {
                outcomes.addOutcome(key, actualGet,
                        Matchers.equalTo(expectedValue));
            } else if ("regex".equals(matcherName)) {
                outcomes.addOutcome(key, actualGet,
                        new RegexMatcher(expectedValue));
            } else {
                throw new UncheckedException("Unknown matcher '${matcherName}'");
            }
        }
        try {
            outcomes.verify();
        } catch (UUIDExceptionWrapper uew) {
            if (uew.getCause() instanceof OutcomesFailed) {
                return false;
            } else {
                throw uew;
            }
        }
        return true;
    }

    protected List<String> matchLogEntries(String nodeName, String componentName, Level level, ExamplesTable logDataTable) {
        List<String> matchedLogEntries = new ArrayList<String>();
        int expectedLogEntryCount = 0;
        for (String logEntry : testFileHandler.getLogEntries()) {
            if (isExpectedLogEntry(logEntry, nodeName, componentName, level, logDataTable)) {
                matchedLogEntries.add(logEntry);
                ++expectedLogEntryCount;
            }
        }
        return matchedLogEntries;
    }

    protected boolean isExpectedLogEntry(String logEntry,
            String expectedNodeName, String expectedComponentName, Level expectedLevel, ExamplesTable expectedLogDataTable) {
        if (!LogInfoFinder.isValidLogTrace(logEntry)) return false;
        if (!LogInfoFinder.getNodeName(logEntry)) return false;
        if (!expectedLevel.toString().equals(LogInfoFinder.getLogLevel(logEntry))) return false;
        if (!expectedComponentName.equals(LogInfoFinder.getLoggerName(logEntry))) return false;
        actualMonitLogDataMap = LogInfoFinder.getMonitLogData(logEntry);
        if (actualMonitLogDataMap.size() != expectedLogDataTable.getRowCount()) return false;
        return isOutcomesMapVerified(actualMonitLogDataMap, expectedLogDataTable);
    }

    @Then("""after <processingDelay> second(s),
    a first MONIT record is logged on <nodeName> by <componentName>
    containing strictly following info in this order:\$logDataTable""")
    @Alias("""after <processingDelay> second(s),
    a MONIT record is logged on <nodeName> by <componentName>
    containing strictly following info in this order:\$logDataTable""")
    def afterDelayAMonitRecordIsLoggedContainingFollowingData(
            int processingDelay,
            String nodeName,
            String componentName,
            ExamplesTable logDataTable) {
        sleep(processingDelay * 1000);
        this.aMonitRecordIsLoggedContainingFollowingInfo(nodeName, componentName, logDataTable);
    }

    @Then("a provider step begin MONIT record is logged")
    def aProviderStepBeginMonitRecordIsLogged(){
        ExamplesTable expectedLogParameters = new ExamplesTable("""| key | matcher | value |
            | traceCode             | equals  | provideFlowStepBegin |
            | flowInstanceId        | regex  | [a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12} |
            | flowStepId            | regex   | [a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12} |
            | flowStepInterfaceName | equals  | """+this.interfaceName+""" |
            | flowStepServiceName   | equals  |  """+this.serviceName+""" |
            | flowStepEndpointName  | equals  |  """+this.endpointName+""" |
            | flowStepOperationName | equals  |  """+this.operation+""" |
            | flowPreviousStepId    | regex  |  [a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12} |""");
        this.aMonitRecordIsLoggedContainingFollowingInfo(this.nodeName, this.componentName, expectedLogParameters );
    }

    @Then("""a first MONIT record is logged on <nodeName> by <componentName>
	containing strictly following info in this order:\$logDataTable""")
    @Alias("""a MONIT record is logged on <nodeName> by <componentName>
	containing strictly following info in this order:\$logDataTable""")
    public void aMonitRecordIsLoggedContainingFollowingInfo(
            String nodeName,
            String componentName,
            ExamplesTable logDataTable) {
        List<String> matchedLogEntries = matchLogEntries(nodeName, componentName, Level.MONIT, logDataTable)
        assert matchedLogEntries.size() == 1;
        this.logEntries[0] = matchedLogEntries.get(0);
    }

    def aMonitMsgRecordIsLogged(TraceCode traceCode){
        ExamplesTable expectedLogParameters = new ExamplesTable("""| key | matcher | value |
        | traceCode             | equals  | """ + traceCode.toString() + """ |
        | flowInstanceId        | regex  | [a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12} |
        | flowStepId            | regex   | [a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12} |
        | payloadContentDumpFile | regex  | .*([a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12})\\/([a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12})_""" + traceCode.toString() + """.xml |""");
        List<String> matchedLogEntries = matchLogEntries(this.nodeName, this.componentName, Level.MONIT_MSG, expectedLogParameters)
        assert matchedLogEntries.size() == 1;
        this.monitMsgLogEntry = matchedLogEntries.get(0);
    }

    @Then("a provider step begin MONIT record is logged with the reference of the dump file")
    def aProviderStepBeginMonitRecordWithPayloadDumpIsLogged(){
        ExamplesTable expectedLogParameters = new ExamplesTable("""| key | matcher | value |
            | traceCode             | equals  | provideFlowStepBegin |
            | flowInstanceId        | regex  | [a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12} |
            | flowStepId            | regex   | [a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12} |
            | flowStepInterfaceName | equals  | """+this.interfaceName+""" |
            | flowStepServiceName   | equals  |  """+this.serviceName+""" |
            | flowStepEndpointName  | equals  |  """+this.endpointName+""" |
            | flowStepOperationName | equals  |  """+this.operation+""" |
            | flowPreviousStepId    | regex  |  [a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12} |
            | payloadContentDumpFile | regex  | .*([a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12})\\/([a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12})_provideFlowStepBegin.xml |""");
        this.aMonitRecordIsLoggedContainingFollowingInfo(this.nodeName, this.componentName, expectedLogParameters );
    }

    @Then("""later, a second MONIT record is logged on <nodeName> by <componentName> containing strictly following info in this order:\$logDataTable""")
    public void thenLaterASecondMONITRecordIsLoggedOnNodeNameByComponentNameContainingStrictlyFollowingInfoInThisOrder(
            @Named("nodeName") String nodeName,
            @Named("componentName") String componentName,
            @Named("logDataTable") ExamplesTable logDataTable){
        this.logEntries[1] = checkCurrentLogEntry(nodeName, componentName, Level.MONIT, logDataTable, this.logEntries[0]);
    }

    @Then("a provider step end MONIT record is logged")
    def aProviderStepEndMonitRecordIsLogged(){
        ExamplesTable expectedLogParameters = new ExamplesTable("""| key | matcher | value |
            | traceCode             | equals  | provideFlowStepEnd |
            | flowInstanceId        | regex  | [a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12} |
            | flowStepId            | regex   | [a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12} |""");
        this.aMonitRecordIsLoggedContainingFollowingInfo(this.nodeName, this.componentName, expectedLogParameters );
    }

    @Then("a provider step end MONIT record is logged with the reference of the dump file")
    def aProviderStepEndMonitRecordWithPayloadDumpIsLogged(){
        ExamplesTable expectedLogParameters = new ExamplesTable("""| key | matcher | value |
            | traceCode             | equals  | provideFlowStepEnd |
            | flowInstanceId        | regex  | [a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12} |
            | flowStepId            | regex   | [a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12} |
            | payloadContentDumpFile | regex  | .*([a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12})\\/([a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12})_provideFlowStepEnd.xml |""");
        this.aMonitRecordIsLoggedContainingFollowingInfo(this.nodeName, this.componentName, expectedLogParameters );
    }

    @Then("""later,
        an other MONIT record is logged on <nodeName> by <componentName>
        containing strictly following info in this order:\$logDataTable""")
    public void laterAnOtherOrSecondMONITRecordIsLoggedContainingFollowingInfo(
            String nodeName,
            String componentName,
            ExamplesTable logDataTable) {
        this.anotherLogEntry = checkCurrentLogEntry(nodeName, componentName, Level.MONIT, logDataTable, this.logEntries[0]);
    }

    @Then("""after first record, an other MONIT record, known as third record, is logged on <nodeName> by <componentName> containing strictly following info in this order:\$logDataTable""")
    public void thenAfterFirstRecordAThirdMONITRecordIsLoggedOnNodeNameByComponentNameContainingStrictlyFollowingInfoInThisOrder(
            @Named("nodeName") String nodeName,
            @Named("componentName") String componentName,
            @Named("logDataTable") ExamplesTable logDataTable) {
        this.logEntries[2] = checkCurrentLogEntry(nodeName, componentName, Level.MONIT, logDataTable, this.logEntries[0]);
    }

    @Then("""later, a fourth MONIT record is logged on <nodeName> by <componentName> containing strictly following info in this order:\$logDataTable""")
    public void thenLaterAFourthMONITRecordIsLoggedOnNodeNameByComponentNameContainingStrictlyFollowingInfoInThisOrder(
            @Named("nodeName") String nodeName,
            @Named("componentName") String componentName,
            @Named("logDataTable") ExamplesTable logDataTable) {
        this.logEntries[3] = checkCurrentLogEntry(nodeName, componentName, Level.MONIT, logDataTable, this.logEntries[2]);
    }

    @Then("""later, a third MONIT record is logged on <nodeName> by <componentName> containing strictly following info in this order:\$logDataTable""")
    public void thenLaterAThirdMONITRecordIsLoggedOnNodeNameByComponentNameContainingStrictlyFollowingInfoInThisOrder(
            @Named("nodeName") String nodeName,
            @Named("componentName") String componentName,
            @Named("logDataTable") ExamplesTable logDataTable) {
        this.logEntries[2] = checkCurrentLogEntry(nodeName, componentName, Level.MONIT, logDataTable, this.logEntries[1]);
    }

    protected String checkCurrentLogEntry(String nodeName, String componentName, Level level, ExamplesTable logDataTable, previousLogEntry) {
        List<String> matchedLogEntries = matchLogEntries(nodeName, componentName, level, logDataTable)
        assert matchedLogEntries.size() == 1;
        String currentLogEntry = matchedLogEntries.get(0);
        boolean laterInTime = LogDataFormatter.DATE_FORMAT.parse(LogInfoFinder.getLogTime(previousLogEntry)).getTime() < LogDataFormatter.DATE_FORMAT.parse(LogInfoFinder.getLogTime(currentLogEntry)).getTime();
        boolean equalInTimeAndLaterInList = LogDataFormatter.DATE_FORMAT.parse(LogInfoFinder.getLogTime(previousLogEntry)).getTime() == LogDataFormatter.DATE_FORMAT.parse(LogInfoFinder.getLogTime(currentLogEntry)).getTime() && matchedLogEntries.indexOf(previousLogEntry) < matchedLogEntries.indexOf(currentLogEntry)
        assert (laterInTime || equalInTimeAndLaterInList)
        return currentLogEntry
    }

    @Then("flowStepId of first record must be equal to flowStepId of the other record")
    public void thenFlowStepIdOfFirstRecordMustBeEqualToFlowStepIdOfTheOtherRecord() {
        assertSameFlowStepId(this.logEntries[0], anotherLogEntry);
    }

    @Then("flowInstanceId of first record must be equal to flowInstanceId of the other record")
    public void thenFlowInstanceIdOfFirstRecordMustBeEqualToFlowInstanceIdOfTheOtherRecord() {
        assertSameFlowInstanceId(this.logEntries[0], anotherLogEntry);
    }

    @Then("flowStepId of all records must be the same")
    public void thenFlowStepIdOfFirstRecordMustBeEqualToFlowStepIdOfOthersRecords(){
        assert this.logEntries.size() > 1
        for (int i = 1; i < this.logEntries.size(); i++) {
            assertSameFlowStepId(this.logEntries[0], this.logEntries[i]);
        }
    }

    @Then("flowInstanceId of all records must be the same")
    public void thenFlowInstanceIdOfAllRecordsMustBeTheSame(){
        assert this.logEntries.size() > 1
        for (int i = 1; i < this.logEntries.size(); i++) {
            assertSameFlowInstanceId(this.logEntries[0], this.logEntries[i]);
        }
    }

    protected assertSameFlowStepId(String aLogEntry, String anOtherLogEntry) {
        String flowStepIdOfALogRecord =
                LogInfoFinder.getMonitLogData(aLogEntry).get(PetalsExecutionContext.FLOW_STEP_ID_PROPERTY_NAME);
        String flowStepIdOfAnOtherLogRecord =
                LogInfoFinder.getMonitLogData(anOtherLogEntry).get(PetalsExecutionContext.FLOW_STEP_ID_PROPERTY_NAME);
        assert flowStepIdOfALogRecord.equals(flowStepIdOfAnOtherLogRecord), "The following log entries doesn't have the same flow step id: \n" +aLogEntry + "\n"+anOtherLogEntry
    }
    protected assertSameFlowInstanceId(String aLogEntry, String anOtherLogEntry) {
        String flowInstanceIdOfALogRecord =
                LogInfoFinder.getMonitLogData(aLogEntry).get(PetalsExecutionContext.FLOW_INSTANCE_ID_PROPERTY_NAME);
        String flowInstanceIdOfAnOtherLogRecord =
                LogInfoFinder.getMonitLogData(anOtherLogEntry).get(PetalsExecutionContext.FLOW_INSTANCE_ID_PROPERTY_NAME);
        assert flowInstanceIdOfALogRecord.equals(flowInstanceIdOfAnOtherLogRecord), "The following log entries doesn't have the same flow instance id: \n" +aLogEntry + "\n"+anOtherLogEntry
    }

    protected assertSameTraceCode(String aLogEntry, String anOtherLogEntry) {
        if (!StringHelper.isNullOrEmpty(aLogEntry) && !StringHelper.isNullOrEmpty(anOtherLogEntry)){
            String traceCodeOfALogRecord =
                    LogInfoFinder.getMonitLogData(aLogEntry).get(AbstractFlowLogData.FLOW_STEP_TRACE_CODE);
            String traceCodeOfAnOtherLogRecord =
                    LogInfoFinder.getMonitLogData(anOtherLogEntry).get(AbstractFlowLogData.FLOW_STEP_TRACE_CODE);
            assert traceCodeOfALogRecord.equals(traceCodeOfAnOtherLogRecord), "The following log entries doesn't have the same trace code: \n" +aLogEntry + "\n"+anOtherLogEntry
        }
    }

    def dumpFileCreatedInFlowInstanceFolder(String payload) {
        LinkedHashMap<String, String> monitLogData = LogInfoFinder.getMonitLogData(this.logEntries[0]);
        File createdFile = new File(PETALS_HOME_DIR, monitLogData.get(PetalsPayloadDumperFileHandler.PAYLOAD_CONTENT_DUMP_FILE_LOGDATA_NAME));
        assert createdFile.exists();
        assert XMLComparator.isEquivalent(payload,IOHelper.readFileAsString(createdFile));
    }

    @Then("""the dump file (name pattern flowInstanceId/flowStepId_traceCode.xml)
    is created in the flow instance logs folder containing the request payload""")
    public void requestPayloadDumpFileCreatedInFlowInstanceFolder(){
        assert this.requestPayload != null;
        this.dumpFileCreatedInFlowInstanceFolder(this.requestPayload);
    }

    @Then("""the dump file (name pattern flowInstanceId/flowStepId_traceCode.xml)
    is created in the flow instance logs folder containing the response payload""")
    public void responsePayloadDumpFileCreatedInFlowInstanceFolder(){
        assert this.responsePayload != null;
        this.dumpFileCreatedInFlowInstanceFolder(this.responsePayload);
    }

    protected doRequestReceivedByService(String operationName,
            URI mep,
            String requestPayload,
            String flowInstanceId,
            String previousStepId) {
        this.requestPayload = requestPayload;
        Message message = new WrappedRequestToProviderMessage(provideServiceConfiguration,  QName.valueOf(operationName),
                mep, createPayloadInputStream(this.requestPayload), new FlowAttributes(flowInstanceId,
                previousStepId));
        this.component.pushRequestToProvider(message);

        doTransportIfNecessary();

        if(mep.equals(AbsItfOperation.MEPPatternConstants.IN_OUT.value())) {
            // wait for the end of processing of the message
            Message response = this.component.pollResponseFromProvider();
            this.responsePayload = response.getPayload();
        } else {
            // the meaning of end step when invoking an in only operation is not
            // defined (functional pb). all we can do is sleep ...
            sleep(3000);
        }
    }


    protected void doTransportIfNecessary() {
    }
}

