package org.ow2.petals.components.stories.util

import org.apache.tools.ant.types.resources.selectors.InstanceOf;
import org.junit.Test
import static org.junit.Assert.*

class LogInfoFinderTest {

    static final String logTemplate = "nodeName logTime logLevel [loggerName] : logData";
    static final String NODE_NAME_SAMPLE = "node1"
    static final String LOG_TIME_SAMPLE = "2011/08/25 15:44:44,532 GMT+0200"
    static final String MONIT_LOG_LEVEL = "MONIT"
    static final String LOGGER_NAME_SAMPLE = "Petals.Container.Components.petals-se-bpel"
    static final String LOG_DATA_SAMPLE = "traceCode = 'provideFlowStepBegin',flowInstanceId = '62cdea40-cf20-11e0-bdfa-000c29523cc7',flowStepId = '64083f00-cf20-11e0-87fc-000c29523cc7',flowStepInterfaceName = '{http://usecase1/bpel-P12/}interfaceName1',flowStepServiceName = '{http://usecase1/bpel-P12/}serviceName1',flowStepOperationName = '{http://usecase1/bpel-P12/}process',flowStepEndpointName = 'endpointName1',flowPreviousStepId = '62ce3860-cf20-11e0-bdfa-000c29523cc7'"

    static final String monitLogSample = NODE_NAME_SAMPLE +" "+ LOG_TIME_SAMPLE + " "+ MONIT_LOG_LEVEL+" [" + LOGGER_NAME_SAMPLE+ "] : "+LOG_DATA_SAMPLE;
    static final String otherLevelLogSample = "node1 2011/08/25 15:44:44,532 GMT+0200 INFO [Petals.Container.Components.petals-se-bpel] : BPEL-Processes deployment on Core server has been performed";
    static final  String problematicLogSample = "node1 2011/08/25 15:44:44,532 GMT+0200 INFO [Petals.Container.Components.petals-se-bpel] : node2 2012/08/25 16:44:44,532 GMT-0300 MONIT [Petals.Container.Components.petals-se-quartz]";
    static final String emptyLog = "";
    static final Map<String,String> emptyMap = [:]

    @Test
    public void testIsMonitLogTraceForMonitLog(){
        assertTrue(LogInfoFinder.isValidMonitLogTrace(monitLogSample));
    }
    @Test
    public void testIsMonitLogTraceForOtherLevelLog(){
        assertFalse(LogInfoFinder.isValidMonitLogTrace(otherLevelLogSample));
    }
    @Test
    public void testIsMonitLogTraceForProblematicLog(){
        assertFalse(LogInfoFinder.isValidMonitLogTrace(problematicLogSample));
    }
    @Test
    public void testIsMonitLogTraceForEmptyLog(){
        assertFalse(LogInfoFinder.isValidMonitLogTrace(emptyLog));
    }
    @Test
    public void testIsMonitLogTraceForNullLog(){
        assertFalse(LogInfoFinder.isValidMonitLogTrace(null));
    }

    @Test
    public void testIsValidLogTraceForMonitLog(){
        assertTrue(LogInfoFinder.isValidLogTrace(monitLogSample));
    }
    @Test
    public void testIsValidLogTraceForOtherLevelLog(){
        assertTrue(LogInfoFinder.isValidLogTrace(otherLevelLogSample));
    }
    @Test
    public void testIsValidLogTraceForProblematicLog(){
        assertTrue(LogInfoFinder.isValidLogTrace(problematicLogSample));
    }
    @Test
    public void testIsValidLogTraceForEmptyLog(){
        assertFalse(LogInfoFinder.isValidLogTrace(emptyLog));
    }
    @Test
    public void testIsValidLogTraceForNullLog(){
        assertFalse(LogInfoFinder.isValidLogTrace(null));
    }

    @Test
    public void testGetNodeNameFromMonitLog() {
        def templateForLogNameTest = logTemplate.replaceFirst("logTime", LOG_TIME_SAMPLE).replaceFirst("logLevel",MONIT_LOG_LEVEL).replaceFirst("loggerName", LOGGER_NAME_SAMPLE).replaceFirst("logData", LOG_DATA_SAMPLE);
        def nodeNames = [
            NODE_NAME_SAMPLE,
            "2e3r5r4t",
            "111",
            "aaaaa",
            ""
        ];
        nodeNames.each {
            def parametrizedMonitLog = templateForLogNameTest.replaceFirst("nodeName", it);testGetNodeName(parametrizedMonitLog, it)
        }
    }

    @Test
    public void testGetNodeNameFromProblematicLog() {
        testGetNodeName(problematicLogSample, NODE_NAME_SAMPLE);
    }

    private void testGetNodeName(String logString, String expectedNodeName) {
        final String actualNodeName = LogInfoFinder.getNodeName(logString);
        assertEquals("Bad node name", expectedNodeName, actualNodeName);
    }

    @Test
    public void testGetLogTimeFromMonitLog() {
        def templateForLogTimeTest = logTemplate.replaceFirst("nodeName", NODE_NAME_SAMPLE).replaceFirst("logLevel",MONIT_LOG_LEVEL).replaceFirst("loggerName", LOGGER_NAME_SAMPLE).replaceFirst("logData", LOG_DATA_SAMPLE)
        def logTimes = [
            LOG_TIME_SAMPLE,
            "2011/08/25 15:44:44,532 GMT-0200",
            "2011/08/25 15:44:44,532 GMT-0700",
            "2011/08/25 00:00:00,000 GMT+0200"
        ];
        logTimes.each {
            def parametrizedMonitLog = templateForLogTimeTest.replaceFirst("logTime", it);testGetLogTime(parametrizedMonitLog, it)
        }
    }

    @Test
    public void testGetLogTimeFromProblematicLog() {
        testGetLogTime(problematicLogSample, LOG_TIME_SAMPLE);
    }

    private void testGetLogTime(String logString, String expectedLogTime) {
        final String actualLogTime = LogInfoFinder.getLogTime(logString);
        assertEquals("Bad log time", expectedLogTime, actualLogTime);
    }

    @Test
    public void testGetLogLevelFromMonitLog() {
        def templateForLogLevelTest = logTemplate.replaceFirst("nodeName", NODE_NAME_SAMPLE).replaceFirst("logTime",LOG_TIME_SAMPLE).replaceFirst("loggerName", LOGGER_NAME_SAMPLE).replaceFirst("logData", LOG_DATA_SAMPLE);
        def logLevels = [
            "MONIT",
            "SEVERE",
            "WARNING",
            "INFO",
            "CONFIG",
            "FINE",
            "FINER",
            "FINEST",
            ""
        ];
        logLevels.each {
            def parametrizedMonitLog = templateForLogLevelTest.replaceFirst("logLevel", it);testGetLogLevel(parametrizedMonitLog, it)
        }
    }

    @Test
    public void testGetLogLevelFromProblematicLog() {
        testGetLogLevel(problematicLogSample, "INFO");
    }

    private void testGetLogLevel(String logString, String expectedLogLevel) {
        final String actualLogLevel = LogInfoFinder.getLogLevel(logString);
        assertEquals("Bad log level", expectedLogLevel, actualLogLevel);
    }

    @Test
    public void testGetLoggerNameFromMonitLog() {
        def templateForLoggerNameTest = logTemplate.replaceFirst("nodeName", NODE_NAME_SAMPLE).replaceFirst("logTime",LOG_TIME_SAMPLE).replaceFirst("logLevel", MONIT_LOG_LEVEL).replaceFirst("logData", LOG_DATA_SAMPLE);
        def loggerNames = [
            "Petals.Container.Components.petals-se-bpel",
            "Petals.Container.Components",
            "",
            "Petals"
        ];
        loggerNames.each {
            def parametrizedMonitLog = templateForLoggerNameTest.replaceFirst("loggerName", it);testGetLoggerName(parametrizedMonitLog, it)
        }
    }


    @Test
    public void testGetLoggerNameFromProblematicLog() {
        testGetLoggerName(problematicLogSample, LOGGER_NAME_SAMPLE);
    }

    private void testGetLoggerName(String logString, String expectedLoggerName) {
        final String actualLoggerName = LogInfoFinder.getLoggerName(logString);
        assertEquals("Bad logger name", expectedLoggerName, actualLoggerName);
    }

    @Test
    public void testGetMonitLogDataFromMonitLog() {
        def expectedMonitLogData = [traceCode:'provideFlowStepBegin', flowInstanceId:'62cdea40-cf20-11e0-bdfa-000c29523cc7', flowStepId:'64083f00-cf20-11e0-87fc-000c29523cc7', flowStepInterfaceName:'{http://usecase1/bpel-P12/}interfaceName1', flowStepServiceName:'{http://usecase1/bpel-P12/}serviceName1', flowStepOperationName:'{http://usecase1/bpel-P12/}process', flowStepEndpointName:'endpointName1', flowPreviousStepId:'62ce3860-cf20-11e0-bdfa-000c29523cc7'];
        testGetMonitLogData(monitLogSample, expectedMonitLogData);
    }

    @Test
    public void testGetMonitLogDataReturnOrderedMap() {
        def LinkedHashMap<String, String> expectedMonitLogData = [traceCode:'provideFlowStepBegin', flowInstanceId:'62cdea40-cf20-11e0-bdfa-000c29523cc7', flowStepId:'64083f00-cf20-11e0-87fc-000c29523cc7', flowStepInterfaceName:'{http://usecase1/bpel-P12/}interfaceName1', flowStepServiceName:'{http://usecase1/bpel-P12/}serviceName1', flowStepOperationName:'{http://usecase1/bpel-P12/}process', flowStepEndpointName:'endpointName1', flowPreviousStepId:'62ce3860-cf20-11e0-bdfa-000c29523cc7'];
        LinkedHashMap<String, String> actualLogData = LogInfoFinder.getMonitLogData(monitLogSample);
        actualLogData.newKeyIterator().eachWithIndex { obj, i ->
            assertTrue(expectedMonitLogData.newKeyIterator().getAt(i).equals(obj))
        }
    }

    @Test
    public void testGetMonitLogDataFromOtherLevelLog() {
        testGetMonitLogData(otherLevelLogSample, emptyMap);
    }

    @Test
    public void testGetMonitLogDataFromEmptyLog() {
        testGetMonitLogData(emptyLog, emptyMap);
    }

    @Test
    public void testGetMonitLogDataFromNullLog() {
        testGetMonitLogData(null, emptyMap);
    }

    private void testGetMonitLogData(String logString, Map<String,String> expectedMonitLogData) {
        final Map<String, String> actualMonitLogData = LogInfoFinder.getMonitLogData(logString);
        assertEquals(expectedMonitLogData, actualMonitLogData);
    }

    @Test
    public void testGetLogDataFromMonitLog() {
        testGetLogData(monitLogSample, LOG_DATA_SAMPLE);
    }

    @Test
    public void testGetLogDataFromOtherLevelLog() {
        testGetLogData(otherLevelLogSample, "BPEL-Processes deployment on Core server has been performed");
    }

    @Test
    public void testGetLogDataFromProblematicLog() {
        testGetLogData(problematicLogSample, "node2 2012/08/25 16:44:44,532 GMT-0300 MONIT [Petals.Container.Components.petals-se-quartz]");
    }

    private void testGetLogData(String logString, String expectedLogData) {
        final String actualLogData = LogInfoFinder.getLogData(logString);
        assertEquals("Bad log data", expectedLogData, actualLogData);
    }
}
