package org.ow2.petals.components.steps;

import static org.junit.Assert.assertFalse

import java.io.ByteArrayInputStream
import java.io.InputStream
import java.util.List
import java.util.Map
import java.util.logging.ConsoleHandler
import java.util.logging.Handler
import java.util.logging.Level
import java.util.logging.LogManager
import java.util.logging.Logger

import javax.xml.namespace.QName

import org.hamcrest.Matchers
import org.jbehave.core.annotations.AfterScenario
import org.jbehave.core.annotations.Alias
import org.jbehave.core.annotations.BeforeScenario
import org.jbehave.core.annotations.Given
import org.jbehave.core.annotations.ScenarioType
import org.jbehave.core.annotations.Then
import org.jbehave.core.failures.UUIDExceptionWrapper
import org.jbehave.core.model.ExamplesTable
import org.jbehave.core.model.OutcomesTable
import org.jbehave.core.model.OutcomesTable.OutcomesFailed
import org.ow2.petals.commons.logger.AbstractFlowLogData
import org.ow2.petals.component.api.Component
import org.ow2.petals.component.api.ComponentConfiguration
import org.ow2.petals.components.stories.util.LogInfoFinder
import org.ow2.petals.components.stories.util.RegexMatcher
import org.ow2.petals.components.stories.util.StoryContext

import com.ebmwebsourcing.easycommons.lang.UncheckedException
import com.ebmwebsourcing.easycommons.logger.BasicLogDataAppender
import com.ebmwebsourcing.easycommons.logger.LogDataFormatter
import com.ebmwebsourcing.easycommons.logger.handler.TestFileHandler
import com.ebmwebsourcing.jbi.adaptor.impl.ComponentType
import com.ebmwebsourcing.jbi.adaptor.impl.WrappedComponent


public class AbstractComponentSteps extends StoryContext {

    static int count = 0;

    public static final String CDK_NAMESPACE_URI = "http://petals.ow2.org/components/extensions/version-5";
    public static final String OPERATION_PARAM_NAME = new QName(CDK_NAMESPACE_URI, "operation").toString();
    public static final String MEP_PARAM_NAME = new QName(CDK_NAMESPACE_URI, "mep").toString();

    @Given("the ESB only composed of <nodeHost> known as <nodeName>")
    def esbOnlyComposedOf(String nodeHost, String nodeName) {
        this.nodeHost = nodeHost;
        this.nodeName = nodeName;
    }

    @Given("MONIT enabled on <nodeName>")
    def monitLogLevelIsEnabledOn(String nodeName) {
        this.logLevel = com.ebmwebsourcing.easycommons.logger.Level.MONIT;
        initLogger(nodeName);
    }

    @Given("INFO enabled on <nodeName>")
    def infoLogLevelIsEnabledOn(String nodeName) {
        this.logLevel = Level.INFO;
        initLogger(nodeName);
    }

    @Then("no MONIT record is logged")
    def noMonitRecordLogged() {
        testFileHandler.getLogEntries().each { assertFalse(LogInfoFinder.isValidMonitLogTrace(it)) };
    }

    @AfterScenario(uponType=ScenarioType.EXAMPLE)
    def stop() {
        if(component != null && component.isStarted()) {
			component.uninstallAllServices();
            component.stop();
        }
    }

    @BeforeScenario(uponType=ScenarioType.EXAMPLE)
    def beforeScenario() {
        LogManager.logManager.reset();
    }

    def private initLogger(nodeName) {
        Logger rootLogger = Logger.getLogger("");

        LogDataFormatter formatter = new LogDataFormatter();
        formatter.setExtraParametersDelimiter("", "");
        formatter.setPrefix(nodeName);
        formatter.addLogDataAppender(new BasicLogDataAppender());
        testFileHandler = new TestFileHandler(getClass().getName() + "_" + count++);

        rootLogger.addHandler(testFileHandler);
        rootLogger.addHandler(new ConsoleHandler());
        Handler[] handlers = rootLogger.getHandlers();

        for(Handler handler : handlers) {
            handler.setFormatter(formatter);
            handler.setLevel(logLevel);
        }
    }


    protected ComponentConfiguration createComponentConfiguration(
    String componentName = "",
    List<URL> resources = new ArrayList<URL>()) {
        componentLogger = Logger.getLogger(componentName);
        componentLogger.setLevel(logLevel);
        ComponentConfiguration componentConfiguration = new ComponentConfiguration(componentLogger);
        if(resources != null && ! resources.isEmpty()) {
            for(URL resource : resources) {
                componentConfiguration.addResource(resource);
            }
        }
        return componentConfiguration;
    }


    protected Component createAndStartComponent(ComponentType componentType,
    ComponentConfiguration componentConfiguration) {

        /*if (componentType.equals(ComponentType.SE_BPEL)) {
         componentConfiguration.setParameter("{http://petals.ow2.org/components/petals-bpel-engine/version-1.0}internal-logger", "true")
         }*/
        Component component = new WrappedComponent(componentType, componentConfiguration);
        component.start();
        return component;
    }


    InputStream createPayloadInputStream(String payloadAsString) {
        return new ByteArrayInputStream(payloadAsString.getBytes());
    }

    protected boolean isOutcomesMapVerified(Map<String, String> actual, ExamplesTable expected) {
        OutcomesTable outcomes = new OutcomesTable();
        for (int i = 0 ; i < expected.getRowCount() ; ++i) {
            Map<String, String> row = expected.getRow(i);
            String matcherName = row.get("matcher");
            def actualGet = actual.get(row.get("key"));
            def expectedGetRowAsParameters = expected.getRowAsParameters(i, true);
            if ("equals".equals(matcherName)) {
                outcomes.addOutcome(row.get("key"), actualGet,
                        Matchers.equalTo(expectedGetRowAsParameters.valueAs("value", String.class)));
            } else if ("regex".equals(matcherName)) {
                outcomes.addOutcome(row.get("key"), actualGet,
                        new RegexMatcher(expectedGetRowAsParameters.valueAs("value", String.class)));
            } else {
                throw new UncheckedException("Unknown matcher '${matcherName}'");
            }
        }
        try {
            outcomes.verify();
        } catch (UUIDExceptionWrapper uew) {
            if (uew.getCause() instanceof OutcomesFailed) {
                return false;
            } else {
                throw uew;
            }
        }
        return true;
    }

    protected List<String> matchLogEntries(String nodeName, String componentName, ExamplesTable logDataTable) {
        List<String> matchedLogEntries = new ArrayList<String>();
        int expectedLogEntryCount = 0;
        for (String logEntry : testFileHandler.getLogEntries()) {
            if (isExpectedLogEntry(logEntry, nodeName, componentName, logDataTable)) {
                matchedLogEntries.add(logEntry);
                ++expectedLogEntryCount;
            }
        }
        return matchedLogEntries;
    }

    protected boolean isExpectedLogEntry(String logEntry,
    String expectedNodeName, String expectedComponentName, ExamplesTable expectedLogDataTable) {
        if (!LogInfoFinder.isValidLogTrace(logEntry)) return false;
        if (!LogInfoFinder.getNodeName(logEntry)) return false;
        if (!com.ebmwebsourcing.easycommons.logger.Level.MONIT.toString().equals(LogInfoFinder.getLogLevel(logEntry))) return false;
        if (!expectedComponentName.equals(LogInfoFinder.getLoggerName(logEntry))) return false;
        return isOutcomesMapVerified(LogInfoFinder.getMonitLogData(logEntry), expectedLogDataTable);
    }

    @Then("""after <processingDelay> second(s),
	a MONIT record is logged on <nodeName> by <componentName>
	containing strictly following info in this order:\$logDataTable""")
    @Alias("after <processingDelay> second(s), a first MONIT record is logged on <nodeName> by <componentName> containing strictly following info in this order:\$logDataTable")
    def afterDelayAMonitRecordIsLoggedContainingFollowingData(
    int processingDelay,
    String nodeName,
    String componentName,
    ExamplesTable logDataTable) {
        sleep(processingDelay * 1000);
        this.aMonitRecordIsLoggedContainingFollowingInfo(nodeName, componentName, logDataTable);
    }

    @Then("""a first MONIT record is logged on <nodeName> by <componentName>
	containing strictly following info in this order:\$logDataTable""")
    @Alias("""a MONIT record is logged on <nodeName> by <componentName>
	containing strictly following info in this order:\$logDataTable""")
    public void aMonitRecordIsLoggedContainingFollowingInfo(
    String nodeName,
    String componentName,
    ExamplesTable logDataTable) {
        List<String> matchedLogEntries = matchLogEntries(nodeName, componentName, logDataTable)
        assert matchedLogEntries.size() == 1;
        firstLogEntry = matchedLogEntries.get(0);
    }

    @Then("""later,
        an other MONIT record is logged on <nodeName> by <componentName>
        containing strictly following info in this order:\$logDataTable""")
    public void laterAnOtherOrSecondMONITRecordIsLoggedContainingFollowingInfo(
    String nodeName,
    String componentName,
    ExamplesTable logDataTable) {
        this.anotherLogEntry = checkCurrentLogEntry(nodeName, componentName, logDataTable, firstLogEntry);
    }

    protected String checkCurrentLogEntry(String nodeName, String componentName, ExamplesTable logDataTable, previousLogEntry) {
        List<String> matchedLogEntries = matchLogEntries(nodeName, componentName, logDataTable)
        assert matchedLogEntries.size() == 1;
        String currentLogEntry = matchedLogEntries.get(0);
        boolean laterInTime = LogDataFormatter.DATE_FORMAT.parse(LogInfoFinder.getLogTime(previousLogEntry)).getTime() < LogDataFormatter.DATE_FORMAT.parse(LogInfoFinder.getLogTime(currentLogEntry)).getTime();
        boolean equalInTimeAndLaterInList = LogDataFormatter.DATE_FORMAT.parse(LogInfoFinder.getLogTime(previousLogEntry)).getTime() == LogDataFormatter.DATE_FORMAT.parse(LogInfoFinder.getLogTime(currentLogEntry)).getTime() && matchedLogEntries.indexOf(previousLogEntry) < matchedLogEntries.indexOf(currentLogEntry)
        assert (laterInTime || equalInTimeAndLaterInList)
        return currentLogEntry
    }

    @Then("flowStepId of first record must be equal to flowStepId of the other record")
    public void thenFlowStepIdOfFirstRecordMustBeEqualToFlowStepIdOfTheOtherRecord() {
        String flowStepIdOfFirstRecord =
                LogInfoFinder.getMonitLogData(firstLogEntry).get(AbstractFlowLogData.FLOW_STEP_ID_PROPERTY_NAME);
        String flowStepIdOfOtherRecord =
                LogInfoFinder.getMonitLogData(anotherLogEntry).get(AbstractFlowLogData.FLOW_STEP_ID_PROPERTY_NAME);
        assert flowStepIdOfFirstRecord.equals(flowStepIdOfOtherRecord);
    }

    @Then("flowInstanceId of first record must be equal to flowInstanceId of the other record")
    public void thenFlowInstanceIdOfFirstRecordMustBeEqualToFlowInstanceIdOfTheOtherRecord() {
        String flowInstanceIdOfFirstRecord =
                LogInfoFinder.getMonitLogData(firstLogEntry).get(AbstractFlowLogData.FLOW_INSTANCE_ID_PROPERTY_NAME);
        String flowInstanceIdOfOtherRecord =
                LogInfoFinder.getMonitLogData(anotherLogEntry).get(AbstractFlowLogData.FLOW_INSTANCE_ID_PROPERTY_NAME);
        assert flowInstanceIdOfFirstRecord.equals(flowInstanceIdOfOtherRecord);
    }
}

