/******************************************************************************
 * Copyright (c) 2008, EBM WebSourcing
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     EBM WebSourcing - initial API and implementation
 *******************************************************************************/

package com.ebmwebsourcing.commons.jbi.sugenerator.utils;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Import one or several files into a folder from their URLs.
 * @author Vincent Zurczak - EBM WebSourcing
 */
public class FileImporter {
	/** The unique instance of this class. */
	private static FileImporter instance = new FileImporter();
	/** */
	private static int renamedCpt = 0;
	
	/** Private constructor (singleton pattern). */
	private FileImporter() {};
	
	/**
	 * @return the unique instance of this class.
	 */
	public static FileImporter getInstance() {
		return instance;
	}
	
	/**
	 * Import a file into a folder.
	 * If a file to import has the same name than a file in the collecting folder,
	 * here is the way it works:
	 * <ul>
	 * <li>If <b>overwrite</b> is true, then the file is overwritten.</li>
	 * <li>Otherwise, if <b>rename</b> is true, the imported file will be renamed (by appending the date at the end).</li>
	 * <li>Otherwise, an exception will be raised stating that this file could not be imported.</li>
	 * </ul>
	 * 
	 * @param url the URL of a file to import.
	 * @param fileRelativePath the relative path of the file in <b>folder</b>.
	 * It can be null, but in any case, it should not begin with a slash.
	 * If it is the same thing than the file name found in the URL, then
	 * no intermediate folder is created.
	 * 
	 * @param folder the folder where these files have to be imported.
	 * @param overwrite true if a file should be overwritten in case of name conflict.
	 * @param rename true if a file should be renamed in case of name conflict.
	 * @param showHiddenFiles true if hidden files should be shown.
	 * Typically, if a file is hidden, its name starts with a dot. Set this parameter to true
	 * will remove this dot.
	 * Example: ".toto.txt" becomes "toto.txt".
	 * 
	 * @return the name of the imported file (possibly renamed).
	 * @throws FileImportsException if one or several exceptions occurred while importing files.
	 * @throws IllegalArgumentException if the folder is not a directory.
	 */
	public String importFile( 
			URL url, String fileRelativePath, File folder, 
			boolean overwrite, boolean rename, boolean showHiddenFiles ) 				
		throws FileImportsException, IllegalArgumentException {
		
		Map<URL, String> urlsAndRelativePath = new HashMap<URL, String> ();
		fileRelativePath = fileRelativePath == null ? "" : fileRelativePath;
		urlsAndRelativePath.put( url, fileRelativePath );

		Map<URL, String> importedFiles = importFiles( urlsAndRelativePath, folder, overwrite, rename, showHiddenFiles );
		return importedFiles.get( url );
	}
	
	/**
	 * Import a list of files into a folder.
	 * If an exception is thrown while importing a file, it does not prevent this method
	 * from importing the other files in the list. This exception is stored into a FileImportsException.
	 * Once all the files have been imported (or tried to), if the FileImportsException has stored
	 * any exception, it is thrown. This way, the user can determine which files could not be imported and why. 
	 * 
	 * If a file to import has the same name than a file in the collecting folder, here is the way it works:
	 * <ul>
	 * <li>If <b>overwrite</b> is true, then the file is overwritten.</li>
	 * <li>Otherwise, if <b>rename</b> is true, the imported file will be renamed (by appending the date at the end 
	 * 		- millisecond precision). If the newly renamed file also exists, an exception will be thrown.</li>
	 * <li>Otherwise, an exception will be raised stating that this file could not be imported.</li>
	 * </ul>
	 * 
	 * @param urlsAndRelativePath a map of file URLs to import. 
	 * Key = file URL. 
	 * Value = relative path with respect to folder.
	 * Let the value empty ("") to import the file at the root of the folder.
	 * If this value is the same than the file name found in the URL, then no
	 * intermediate folder is created.
	 * If the relative path leads the file to be outside the destination folder, then this file
	 * is placed at the root of the folder.
	 * In any case, the relative path should not begin with a slash and should not be null.
	 * 
	 * @param folder the folder where these files have to be imported.
	 * @param overwrite true if a file should be overwritten in case of name conflict.
	 * @param rename true if a file should be renamed in case of name conflict.
	 * @param showHiddenFiles true if hidden files should be shown.
	 * Typically, if a file is hidden, its name starts with a dot. Set this parameter to true
	 * will remove this dot.
	 * Example: ".toto.txt" becomes "toto.txt".
	 * 
	 * TODO: Typically, if a file is hidden, its name starts with a dot. Well, not on Windows.
	 * 
	 * @return a map containing the URLs with the file names (possibly renamed).
	 * The key is the URL and the value is the file name. Never null.
	 * @throws FileImportsException if one or several exceptions occurred while importing files.
	 * @throws IllegalArgumentException if the folder is not a directory.
	 */
	public Map<URL, String> importFiles( 
			Map<URL, String> urlsAndRelativePath, File folder, 
			boolean overwrite, boolean rename, boolean showHiddenFiles ) 
		throws FileImportsException, IllegalArgumentException {
		
		if( !folder.isDirectory())
			throw new IllegalArgumentException( folder.getAbsolutePath() + " is not a directory." );
		
		GregorianCalendar calendar = new GregorianCalendar();
		SimpleDateFormat sdf = new SimpleDateFormat( "dd_MM_yyyy__'at'__HH_mm_ss" ); 
		
		Map<URL, String> fileNames = new HashMap<URL, String> ();
		FileImportsException finalException = new FileImportsException();
		
		// Import files.
		URL lastUrl = null;
		for( Map.Entry<URL, String> importedUrl : urlsAndRelativePath.entrySet()) {
			boolean importFailed = false;
			try {
				lastUrl = importedUrl.getKey();
				
				// Replace all "\" by "/".
				String urlPath = importedUrl.getKey().toExternalForm();
				urlPath = urlPath.replaceAll( "\\\\", "/" );
				
				// Get file name.
				String[] parts = urlPath.split( "/" );
				String fileName = ( parts.length > 1 ) ? parts[ parts.length - 1 ] : "";
				String originalFileName = fileName;
				
				// Hidden files.
				if( fileName.startsWith( "." ) && showHiddenFiles )
					fileName = fileName.substring( 1 );
				
				// No file name.
				if( "".equals( fileName ))
					fileName = "importedFile__" + sdf.format( calendar.getTime());
								
				// Replace strange characters in file name.
				// In fact, replace every character except numbers, letters, '.', '_' and '-'.
				fileName = fileName.replaceAll( "[^-._\\w]", "_" );
								
				// Clean the relative path.
				String relativePath = importedUrl.getValue();
				relativePath = relativePath == null ? "" : relativePath;
				relativePath = relativePath.replaceAll( "\\\\", "/" );
				
				// No relative path if it ends with the file name.
				relativePath = 	relativePath.endsWith( originalFileName ) 
								? relativePath.substring( 0, relativePath.length() - originalFileName.length()) 
								: relativePath;
				relativePath = 	relativePath.endsWith( "/" ) 
								? relativePath.substring( 0, relativePath.length() - 1 ) 
								: relativePath;
				
				// Make sure this relative path does not save files outside the destination folder.
				File destinationFolder = new File( folder, relativePath );
				String destinationFolderPath = destinationFolder.getAbsolutePath().toLowerCase();
				String folderPath = folder.getAbsolutePath().toLowerCase();
				if( !destinationFolderPath.startsWith( folderPath ))
					relativePath = "";
								
				// Create intermediate folders.
				if( !"".equals( relativePath )) {					
					folder = new File( folder, relativePath );
					if( !folder.exists())
						folder.mkdirs();
				}
				
				// Prepare files.
				InputStream source = importedUrl.getKey().openStream();
				File newFile = new File( folder, fileName );
				if( !newFile.exists()) {
					newFile.createNewFile();
				}
				else if( overwrite ) { 
					PrintWriter printwriter = new PrintWriter( new FileOutputStream( newFile ));
					printwriter.println( "" );				    
					printwriter.close();
				}
				else if( rename ) {
					int dotPosition = fileName.lastIndexOf( "." );
					if( dotPosition <= 0 )
						fileName += "_renamed_" + renamedCpt++;
					else
						fileName = 
							JbiNameFormatter.insertSuffixBeforeFileExtension( 
									fileName, "_renamed_" + renamedCpt++ );
					
					// There are few chances for this file to already exist.
					// Otherwise, it will throw an exception.
					newFile = new File( folder, fileName );
					newFile.createNewFile();	
				}
				else 
					importFailed = true;
				
				// Register file name
				relativePath += "".equals( relativePath ) ? "" : "/";
				relativePath += fileName;
				fileNames.put( importedUrl.getKey(), relativePath );
				
				// Exception logged if import fails.
				if( importFailed ){
					Exception e = new IOException( "File " + fileName + " already exists. Import failed." );
					finalException.failedImportUrls.put( lastUrl, e );
					continue;
				}
				
				// Write into the file.
				OutputStream out = new FileOutputStream( newFile );
			    byte buf[] = new byte[ 1024 ];
			    int len;
			    while ((len = source.read( buf )) > 0) {
	                out.write( buf, 0, len);
	            }
			    out.close ();
			    source.close ();
			    
			} catch( Exception e ) {
				finalException.failedImportUrls.put( lastUrl, e );
			}
		}
		
		// Throw exception ?
		if( finalException.hasExceptions())
			throw finalException;
				
		return fileNames;
	}
	
	/**
	 * List exception which occurred while importing a list of files.
	 */
	public class FileImportsException extends Exception {
		/** */
		private static final long serialVersionUID = 3562089833670361410L;
		/** URL that failed to be imported and the associated exceptions. */
		public Map<URL, Exception> failedImportUrls = new HashMap<URL, Exception> ();

		
		/**
		 * @see List#isEmpty()
		 * @return true if the class has at least one exception.
		 */
		protected boolean hasExceptions() {
			return failedImportUrls.size() >= 1;
		}
		
		/*
		 * (non-Javadoc)
		 * @see java.lang.Throwable#toString()
		 */
		public String toString() {
			StringWriter stringWriter = new StringWriter();
			PrintWriter result = new PrintWriter( stringWriter );
			
			// Basic message.
			if( failedImportUrls.isEmpty())
				result = result.append( 
						"FileImportsException: no error was found while importing files.\n" );
			else if( failedImportUrls.size() > 1 )
				result = result.append( 
						"FileImportsException: " + failedImportUrls.size() + " errors were found while importing files.\n" );
			else
				result = result.append( 
						"FileImportsException: 1 error was found while importing files.\n" );
			
			// Detailed stack trace.
			for( Map.Entry<URL, Exception> e : failedImportUrls.entrySet()) {
				result.print( "URL " + e.getKey() + " failed to be imported." );
				result.print( e.getValue().getMessage());
				e.getValue().printStackTrace( result );
			}
			
			result.flush();
			return stringWriter.toString();
		}
	}
}
