/**
 * easySchema - SOA Tools Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * $id.java
 * -------------------------------------------------------------------------
 */
package com.ebmwebsourcing.commons.schema;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Map;
import java.util.logging.Logger;

import javax.xml.transform.Source;
import javax.xml.transform.dom.DOMSource;

import org.xml.sax.InputSource;

import com.ebmwebsourcing.commons.schema.api.Schema;
import com.ebmwebsourcing.commons.schema.api.SchemaException;
import com.ebmwebsourcing.commons.schema.api.SchemaReader;
import com.ebmwebsourcing.commons.schema.api.SchemaWriter;
import com.ebmwebsourcing.commons.schema.api.XmlException;
import com.ebmwebsourcing.commons.schema.api.SchemaReader.FeatureConstants;
import com.ebmwebsourcing.commons.schema.impl.Constants;
import com.ebmwebsourcing.commons.schema.impl.SchemaJAXBContext;
import com.ebmwebsourcing.commons.schema.util.SourceHelper;

/**
 * This abstract class defines a factory API that enables applications
 * to obtain a SchemaFactory capable of producing new Definitions, new
 * SchemaReaders, and new SchemaWriters.
 *
 * Some ideas used here have been shamelessly copied from the
 * wonderful JAXP and Xerces work.
 *
 * @author Nicolas Salatge - eBM WebSourcing
 */
public abstract class SchemaFactory
{
	static private Schema defaultSchema;

	private static Logger log = Logger.getLogger(SchemaFactory.class.getName());


	static {
		final URL schemaUrl4DOM = SchemaJAXBContext.class.getResource("/" +  Constants.XSD_SCHEMA4DOM);
		final URL schemaUrl = SchemaJAXBContext.class.getResource("/" +  Constants.XSD_SCHEMA);
		try {
			DOMSource source = SourceHelper.convertInputSource2DOMSource(new InputSource(schemaUrl4DOM.openStream()));
			SchemaFactory.defaultSchema = new com.ebmwebsourcing.commons.schema.impl.SchemaReader().readSchema(schemaUrl.toURI(), source);
		} catch (IOException e) {
			e.printStackTrace();
			log.warning("Error to read default schema => " + e.getMessage());
		} catch (XmlException e) {
			e.printStackTrace();
			log.warning("Error to read default schema => " + e.getMessage());
		} catch (URISyntaxException e) {
			e.printStackTrace();
			log.warning("Error to read default schema => " + e.getMessage());
		}
	}

	/**
	 * Get a new instance of a SchemaFactory. This method
	 * follows (almost) the same basic sequence of steps that JAXP
	 * follows to determine the fully-qualified class name of the
	 * class which implements SchemaFactory.
	 * <p>
	 * The steps in order are:
	 * <ol>
	 *  <li>Check the property file META-INF/services/javax.wsdl.factory.SchemaFactory.</li>
	 *  <li>Check the javax.wsdl.factory.SchemaFactory system property.</li>
	 *  <li>Check the lib/wsdl.properties file in the JRE directory. The key
	 *  will have the same name as the above system property.</li>
	 *  <li>Use the default class name provided by the implementation.</li>
	 * </ol>
	 * <p>
	 * Once an instance of a SchemaFactory is obtained, invoke
	 * newDefinition(), newSchemaReader(), or newSchemaWriter(), to create
	 * the desired instances.
	 */
	public static SchemaFactory newInstance() throws SchemaException
	{
		return new SchemaFactoryImpl();
	}

	/**
	 * Create a new instance of a Definition.
	 */
	public abstract Schema newSchema();

	/**
	 * Create a new instance of a SchemaReader.
	 */
	public abstract SchemaReader newSchemaReader();

	/**
	 * Create a new instance of a SchemaReader.
	 */
	public abstract SchemaReader newSchemaReader(Map<FeatureConstants, Object> features);

	/**
	 * Create a new instance of a SchemaWriter.
	 */
	public abstract SchemaWriter newSchemaWriter();

	/**
	 * Get the default schema
	 * @return the default schema
	 */
	public static Schema getDefaultSchema() {
		return SchemaFactory.defaultSchema;
	}
}