package com.ebmwebsourcing.commons.schema.util;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import com.ebmwebsourcing.commons.schema.api.XmlException;

public class SourceHelper {

	private final static ThreadLocal<DocumentBuilder> documentBuilderThreadLocal = new ThreadLocal<DocumentBuilder>() {

		@Override
		protected DocumentBuilder initialValue() {
			try {
				final DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory
				.newInstance();
				documentBuilderFactory.setNamespaceAware(true);
				final DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
				return documentBuilder;
			} catch (ParserConfigurationException e) {
				throw new RuntimeException("Failed to create DocumentBuilder", e);
			}
		}
	};

	private final static ThreadLocal<Transformer> transformerWithoutXmlDeclarationThreadLocal = new ThreadLocal<Transformer>() {

		@Override
		protected Transformer initialValue() {
			try {
				final Transformer transformer = TransformerFactory.newInstance().newTransformer();
								
				Properties props = new Properties();
				props.put(OutputKeys.OMIT_XML_DECLARATION, "yes");
				transformer.setOutputProperties(props);
				return transformer;

			} catch (TransformerConfigurationException e) {
				throw new RuntimeException("Failed to create Transformer", e);
			}
		}
	};

	/**
	 * Creates a new instance of {@link SourceHelper}
	 * 
	 */
	private SourceHelper() {
		super();
	}


	/**
	 * Create a Source from a DOM document.
	 * 
	 * @param document
	 *      A DOM document
	 * @return the Source built from the DOM document
	 */
	public static DOMSource createSource(final Document document) {
		// normalize the document to assure the resolution of each node
		// namespace
		document.normalizeDocument();
		return new DOMSource(document);
	}

	/**
	 * Create a DOM Document from a Source.
	 * 
	 * @param s
	 *      the ${link {@link Source}
	 * @return the DOM document
	 * @throws PEtALSCDKException
	 */
	public static Document createDocument(final Source s) throws XmlException {

		try {
			final Document document = documentBuilderThreadLocal.get().newDocument();
			final DOMResult domResult = new DOMResult(document);
			SourceHelper.transformerWithoutXmlDeclarationThreadLocal.get().transform(s, domResult);
			return document;
		} catch (TransformerException e) {
			throw new XmlException(e);
		}
	}


	public static DOMSource convertInputSource2DOMSource(InputSource inputSource) throws XmlException {
		DOMSource res = null;
		try {
			DocumentBuilder db = DocumentBuilderFactory.newInstance().newDocumentBuilder();
			Document document = db.parse(inputSource);
			res = new DOMSource(document);
		} catch (ParserConfigurationException e) {
			throw new XmlException(e);
		} catch (SAXException e) {
			throw new XmlException(e);
		} catch (IOException e) {
			throw new XmlException(e);
		}
		return res;
	}

	public static InputSource convertDOMSource2InputSource(DOMSource inputSource) throws XmlException {
		InputSource res = null;
		try {
			ByteArrayOutputStream os = new ByteArrayOutputStream();
			StreamResult streamResult = new StreamResult(os);
			Transformer transformer = 
				TransformerFactory.newInstance().newTransformer();
			transformer.transform(inputSource, streamResult);
			InputStream is = new java.io.ByteArrayInputStream(os.toByteArray());
			res = new InputSource(is);
		} catch (TransformerException e) {
			throw new XmlException(e);
		}
		return res;
	}





	public static DOMSource cloneSource(DOMSource input) throws XmlException {
		Document doc = createDocument(input);
		return createSource(doc);
	}

}
