/**
 * easyWSDL - SOA Tools Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * $id.java
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.commons.wsdl.util;

import java.net.URI;
import java.net.URISyntaxException;

/**
 * @author Nicolas Salatge - eBM WebSourcing
 */
public class Constants {

    public static final String WSDL11_PACKAGE = "com.ebmwebsourcing.commons.wsdl.impl.wsdl11";

    public static final String WSDL20_PACKAGE = "com.ebmwebsourcing.commons.wsdl.impl.wsdl20";

    /**
     * Constants for WSDL version.
     * 
     */
    public enum WSDLVersionConstants {
        WSDL11(com.ebmwebsourcing.commons.wsdl.impl.wsdl11.Constants.WSDL_11_NAMESPACE), WSDL20(
                com.ebmwebsourcing.commons.wsdl.impl.wsdl20.Constants.WSDL_20_NAMESPACE);

        /**
         * 
         * @param pattern
         * @return
         */
        public static WSDLVersionConstants valueOf(final URI pattern) {
            WSDLVersionConstants result = null;
            if (pattern != null) {
                for (final WSDLVersionConstants version : WSDLVersionConstants.values()) {
                    if (version.nameSpace.equals(pattern.toString())) {
                        result = version;
                    }
                }
            }
            return result;
        }

        private final String nameSpace;

        private final URI version;

        /**
         * Creates a new instance of {@link WSDLVersionConstants}
         * 
         * @param nameSpace
         */
        private WSDLVersionConstants(final String nameSpace) {
            this.nameSpace = nameSpace;
            try {
                this.version = new URI(nameSpace);
            } catch (final URISyntaxException e) {
                throw new Error("Unexpected Error in URI namespace syntax", e); // NOPMD
            }
        }

        /**
         * 
         * @return
         */
        public URI value() {
            return this.version;
        }

        /**
         * Please use this equals method instead of using :<code>
         * value().equals(mep)
         * </code> which is
         * almost 10 times slower...
         * 
         * @param mep
         * @return
         */
        public boolean equals(final URI version) {
            return this.toString().equals(version.toString());
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.lang.Enum#toString()
         */
        @Override
        public String toString() {
            return this.nameSpace;
        }
    }

    /**
     * Constants for the Message Exchange Patterns.
     * 
     */
    public enum SOAPMEPConstants {
        ONE_WAY("http://www.w3.org/2006/08/soap/mep/one-way/"), REQUEST_RESPONSE(
                "http://www.w3.org/2003/05/soap/mep/request-response"), SOAP_RESPONSE(
                "http://www.w3.org/2003/05/soap/mep/soap-response/");

        /**
         * 
         * @param pattern
         * @return
         */
        public static SOAPMEPConstants valueOf(final URI pattern) {
            SOAPMEPConstants result = null;
            if (pattern != null) {
                for (final SOAPMEPConstants mep : SOAPMEPConstants.values()) {
                    if (mep.nameSpace.equals(pattern.toString())) {
                        result = mep;
                    }
                }
            }
            return result;
        }

        private final String nameSpace;

        private final URI mepURI;

        /**
         * Creates a new instance of {@link SOAPMEPConstants}
         * 
         * @param nameSpace
         */
        private SOAPMEPConstants(final String nameSpace) {
            this.nameSpace = nameSpace;
            try {
                this.mepURI = new URI(nameSpace);
            } catch (final URISyntaxException e) {
                throw new Error("Unexpected Error in URI namespace syntax", e); // NOPMD
            }
        }

        /**
         * 
         * @return
         */
        public URI value() {
            return this.mepURI;
        }

        /**
         * Please use this equals method instead of using :<code>
         * value().equals(mep)
         * </code> which is
         * almost 10 times slower...
         * 
         * @param mep
         * @return
         */
        public boolean equals(final URI mep) {
            return this.toString().equals(mep.toString());
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.lang.Enum#toString()
         */
        @Override
        public String toString() {
            return this.nameSpace;
        }
    }

    /**
     * Constants for the Patterns.
     * 
     */
    public enum MEPPatternConstants {
        IN_ONLY("http://www.w3.org/ns/wsdl/in-only"), ROBUST_IN_ONLY(
                "http://www.w3.org/ns/wsdl/robust-in-only"), IN_OUT(
                "http://www.w3.org/ns/wsdl/in-out"), IN_OPTIONAL_OUT(
                "http://www.w3.org/ns/wsdl/in-optional-out"), OUT_ONLY(
                "http://www.w3.org/ns/wsdl/out-only"), ROBUST_OUT_ONLY(
                "http://www.w3.org/ns/wsdl/robust-out-only"), OUT_IN(
                "http://www.w3.org/ns/wsdl/out-in"), OUT_OPTIONAL_IN(
                "http://www.w3.org/ns/wsdl/out-optional-in");

        /**
         * 
         * @param pattern
         * @return
         */
        public static MEPPatternConstants valueOf(final URI pattern) {
            MEPPatternConstants result = null;
            if (pattern != null) {
                for (final MEPPatternConstants p : MEPPatternConstants.values()) {
                    if (p.nameSpace.equals(pattern.toString())) {
                        result = p;
                    }
                }
            }
            return result;
        }

        private final String nameSpace;

        private final URI patternURI;

        /**
         * Creates a new instance of {@link MEPPatternConstants}
         * 
         * @param nameSpace
         */
        private MEPPatternConstants(final String nameSpace) {
            this.nameSpace = nameSpace;
            try {
                this.patternURI = new URI(nameSpace);
            } catch (final URISyntaxException e) {
                throw new Error("Unexpected Error in URI namespace syntax", e); // NOPMD
            }
        }

        /**
         * 
         * @return
         */
        public URI value() {
            return this.patternURI;
        }

        /**
         * Please use this equals method instead of using :<code>
         * value().equals(pattern)
         * </code> which is
         * almost 10 times slower...
         * 
         * @param mep
         * @return
         */
        public boolean equals(final URI pattern) {
            return this.toString().equals(pattern.toString());
        }

        /**
         * (non-Javadoc)
         * 
         * @see java.lang.Enum#toString()
         */
        @Override
        public String toString() {
            return this.nameSpace;
        }
    }
}
