/**
 * easy VIPER software - Copyright (c) 2009 PetalsLink, 
 * http://www.petalslink.com/ 
 *  
 * This library is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2.1 of the License, or (at your option) 
 * any later version. This library is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
 * General Public License for more details. 
 *  
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, Inc., 
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 *  
 * ------------------------------------------------------------------------- 
 * $Id$ 
 * ------------------------------------------------------------------------- 
 */ 
package com.ebmwebsourcing.easyviper.core.impl.engine;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map.Entry;
import java.util.logging.Logger;

import org.oasisopen.sca.ServiceReference;
import org.oasisopen.sca.annotation.PolicySets;
import org.oasisopen.sca.annotation.Reference;
import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.Interface;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.fraclet.annotations.Lifecycle;
import org.objectweb.fractal.fraclet.types.Step;
import org.objectweb.fractal.julia.ComponentInterface;
import org.objectweb.fractal.julia.Interceptor;
import org.ow2.frascati.tinfi.oasis.ServiceReferenceImpl;

import com.ebmwebsourcing.easycommons.sca.helper.api.SCAException;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAComponentImpl;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAHelper;
import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.Node;
import com.ebmwebsourcing.easyviper.core.api.engine.Scope;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.Behaviour;
import com.ebmwebsourcing.easyviper.core.api.engine.fault.Fault;

/**
 * @author Nicolas Salatge - eBM WebSourcing
 */
@org.oasisopen.sca.annotation.Scope("COMPOSITE")
@org.oasisopen.sca.annotation.Service(value=Node.class,names="service")
@PolicySets("frascati:scaEasyPrimitive")
public class NodeImpl extends SCAComponentImpl implements Node {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private Logger log = Logger.getLogger(NodeImpl.class.getName());

	/**
	 * child childNodes
	 */
	@Reference(name="childNodes",required=false)
	protected List<Node> childNodes = Collections.synchronizedList(new ArrayList<Node>());

	@Reference(name="incomingNodes",required=false)
	protected List<Node> incomingNodes = Collections.synchronizedList(new ArrayList<Node>());

	@Reference(name="outgoingNodes",required=false)
	protected List<Node> outgoingNodes = Collections.synchronizedList(new ArrayList<Node>());

	@Reference(name="behaviour",required=false)
	protected Behaviour behaviour;

	@Reference(name="parentNode",required=false)
	protected Node parentNode;

	@Reference(name="execution",required=false)
	protected Execution execution;


	/**
	 * Default constructor
	 * 
	 * @throws CoreException
	 */
	public NodeImpl() {
		super();
	}


	/**
	 * Stop the NodeImpl behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = Step.STOP)
	public void stopSCAComponent() throws SCAException {
		this.log.finest("node " + this.getName() + " is stopping... ");
		if ((this.childNodes != null) && (this.childNodes.size() > 0)) {
			final Component parentComp = SCAHelper.getSCAHelper().getParent(
					this.getComponent());
			for (final Node child : this.getChildNodes()) {
				this.log.finest("stop child node: " + this.getName());

				final Component nodeComp = SCAHelper.getSCAHelper()
				.getComponentByInterface(parentComp,
						(ServiceReference<?>) child,
				"service");
				if ((nodeComp != null)
						&& (SCAHelper.getSCAHelper()
								.isStarted(nodeComp))) {
					SCAHelper.getSCAHelper().stopComponent(nodeComp);
				}
			}
		}
		this.log.finest("node stopped: " + this.getName());
	}

	public void setActivity(final Behaviour behaviour) {

		this.behaviour = behaviour;
	}

	public Behaviour getActivity() {
		return this.behaviour;
	}

	public void execute() throws CoreException {
		try {

			// execute the behaviour of the node
			if (this.behaviour != null) {

				// start the behaviour
				Component activityComp = null;
				if( this.behaviour instanceof ServiceReference<?> ) {
					activityComp = SCAHelper
					.getSCAHelper()
					.getComponentByInterface(
							this.execution.getParentScope().getComponent(),
							(ServiceReference<?>) this.behaviour,
					"service");
				}
				else {
					activityComp = SCAHelper
					.getSCAHelper()
					.getComponentByInterface(
							this.execution.getParentScope().getComponent(),
							(org.objectweb.fractal.api.Interface) this.behaviour,
					"service");
				}
				if ((activityComp != null)
						&& (!SCAHelper.getSCAHelper().isStarted(
								activityComp))) {
					SCAHelper.getSCAHelper().startComponent(
							activityComp);
				}


				try {
					this.behaviour.execute();

				} catch (CoreException e) {

					Scope currentScope = this.getExecution().getParentScope();

					Scope faultScope = findFaultScope(currentScope, e);
					if(faultScope == null || faultScope.isFaultScope() != null) {
						// fault not catched
						log.finest("fault not catched: " + e.getClass().getName());
						this.execution.setState(Execution.State.ENDED);
						throw e;
					} else {
						// fault catched
						log.finest("fault catched by scope: " + faultScope.getName());
						faultScope.setFaultScope(e);

						if(faultScope.getParentScope().getName().equals(this.execution.getParentScope().getName())) {
							getExecution().setNextExecutableElements(this, (Node)faultScope.getComponent().
									getFcInterface("service"));

						} else {

							this.execution.setState(Execution.State.ENDED);

							faultScope.getParentScope().linkedExecution2ExecutableElement(faultScope.getParentScope().getExecution(), faultScope);
							faultScope.getParentScope().linkedExecutableElement2Execution(faultScope, faultScope.getParentScope().getExecution());

							faultScope.execute();
						}
					}
				}

				log.finest("State of behaviour's node: " + this.getName() + " => " + this.getBehaviour().getState());
			}


		} catch (NoSuchInterfaceException e1) {
			throw new CoreException(e1);
		} catch (SCAException e1) {
			throw new CoreException(e1);
		} 


	}

	private Scope findFaultScope(Scope currentScope, Exception e) throws CoreException {
		Scope faultScope = null;

		// test if currentScope is not already a fault scope
		Exception exTest = null;
		Scope current = currentScope;
		while(exTest==null && current != null) {
			exTest = current.isFaultScope();
			current = current.getParentScope();
		}
		if(exTest == null) {
			for(Entry<Fault, Scope> entry: currentScope.getExceptions().entrySet()) {
				if(entry.getKey().getFaultHandler() != null && entry.getKey().getFaultHandler().match(currentScope, entry.getKey(), e, entry.getValue())) {
					faultScope = entry.getValue();
					break;
				}
			}

			if(faultScope == null && currentScope.getParentScope() != null) {
				faultScope = this.findFaultScope(currentScope.getParentScope(), e);
			}
		}
		return faultScope;
	}

	public void selectDefaultNextTarget()
	throws CoreException {
		this.log.finest("select default next target");

		Execution execution = getExecution();
		boolean b = execution.hasNextExecutableElement();
		if (!b) {
			List<Node> out = this.getOutgoingNodes();
			this.log.finest("list of following transition: " + out);
			this.log.finest("this.getParentNode(): " + this.getParentNode());
			this.log.finest("this.getExecution(): " + this.getExecution());

			// choose next element
			if (out.size() == 1) {
				// select outgoing transition
				getExecution().setNextExecutableElements(this, out);
			} else if (this.getParentNode() != null) {
				this.log.finest("this.getParentNode().getExecution(): " + this.getParentNode().getExecution());
				if(this.getExecution().getParentExecution() != null) {
					//FIXME using 'name' ain't pretty but is better than using references
					// because Fractal sometimes wraps the same content with different objects
					//previously was : if (this.getParentNode().getExecution() == this.getExecution()) {
					if (this.getParentNode().getExecution().getName().equals(this.getExecution().getName())) {
						this.log.finest("parent node is next target: " + this.getParentNode().getName());
						// select parent node
						getExecution().setNextExecutableElements(this, this.getParentNode());
					} else {
						getExecution().setNextExecutableElements(this, Collections.EMPTY_LIST);
					}
				} else {
					this.log.finest("parent node is next target: " + this.getParentNode().getName());
					// select parent node
					getExecution().setNextExecutableElements(this, this.getParentNode());
				}
			} else {
				getExecution().setNextExecutableElements(this, Collections.EMPTY_LIST);
			}
		}
	}

	public Node getParentNode() {
		return this.parentNode;
	}

	public void setParentNode(Node parentNode) {
		this.parentNode = parentNode;
	}

	public Execution getExecution() {
		return this.execution;
	}

	public List<Node> getChildNodes() {
		return new ArrayList<Node>(this.childNodes);
	}

	public Behaviour getBehaviour() {

		if( this.behaviour instanceof Interface ) {
			return this.behaviour;
		}

		/*
		 * The solution implemented below is linked to a difference between
		 * Fraclet/Julia and Tinfi.
		 * 
		 * With Fraclet/Julia, when injecting a reference in a @Required
		 * annotated field, the reference of the target bound interface is
		 * injected.
		 * 
		 * With Tinfi, the ServiceReference of the source component is injected.
		 * 
		 * Let getBehaviour return the target bound interface.
		 */
		if(this.behaviour != null){
			ServiceReference<?> sr = (ServiceReference<?>) this.behaviour;
			ServiceReferenceImpl<?> cri = (ServiceReferenceImpl<?>) sr;
			Object service = cri._getDelegate();
			ComponentInterface ci = (ComponentInterface) service;
			Interceptor intercep = (Interceptor) ci.getFcItfImpl();
			Behaviour itf = (Behaviour) intercep.getFcItfDelegate();
			return itf;
		}else{
			return null;
		}
	}

	@Override
	public String toString() {
		String res = "";
		res = "node " + this.getName();
		return res;
	}

	public List<Node> getIncomingNodes() {
		return this.incomingNodes;
	}

	public List<Node> getOutgoingNodes() {
		return this.outgoingNodes;
	}

	public void setLog(final Logger logger) {
		this.log = logger;
	}

	public Logger getLogger() {
		return this.log;
	}

	public void setExecution(Execution exec) {
		this.execution = exec;
	}
}
