/**
* easy VIPER software - Copyright (c) 2009 PetalsLink, 
* http://www.petalslink.com/ 
*  
* This library is free software; you can redistribute it and/or modify it under 
* the terms of the GNU Lesser General Public License as published by the Free 
* Software Foundation; either version 2.1 of the License, or (at your option) 
* any later version. This library is distributed in the hope that it will be 
* useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
* General Public License for more details. 
*  
* You should have received a copy of the GNU Lesser General Public License 
* along with this library; if not, write to the Free Software Foundation, Inc., 
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 *  
 * ------------------------------------------------------------------------- 
 * $Id$ 
 * ------------------------------------------------------------------------- 
 */ 
package com.ebmwebsourcing.easyviper.core.impl.engine;

import static org.objectweb.fractal.fraclet.types.Step.CREATE;
import static org.objectweb.fractal.fraclet.types.Step.DESTROY;
import static org.objectweb.fractal.fraclet.types.Step.START;
import static org.objectweb.fractal.fraclet.types.Step.STOP;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.logging.Logger;

import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.fraclet.annotations.Controller;
import org.objectweb.fractal.fraclet.annotations.Interface;
import org.objectweb.fractal.fraclet.annotations.Lifecycle;
import org.objectweb.fractal.fraclet.annotations.Requires;
import org.objectweb.fractal.fraclet.extensions.Membrane;
import org.objectweb.fractal.fraclet.types.Cardinality;
import org.objectweb.fractal.fraclet.types.Contingency;

import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.ExecutableElement;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.Node;
import com.ebmwebsourcing.easyviper.core.api.engine.Process;
import com.ebmwebsourcing.easyviper.core.api.engine.Scope;
import com.ebmwebsourcing.easyviper.core.fractal.Binding;
import com.ebmwebsourcing.easyviper.core.fractal.FractalHelper;
import com.ebmwebsourcing.easyviper.core.impl.engine.thread.ThreadExecution;
import com.ebmwebsourcing.easyviper.core.impl.engine.thread.ThreadSignal;

/**
 * @author Nicolas Salatge - eBM WebSourcing
 */
@org.objectweb.fractal.fraclet.annotations.Component(provides = @Interface(name = "service", signature = Execution.class))
@Membrane(controller = "primitive")
public class ExecutionImpl implements Execution {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private Logger log = Logger.getLogger(ExecutionImpl.class.getName());

	/**
	 * variable for debugging
	 */
	private boolean stepByStep = false;

	/**
	 * {@link Execution#STATE_ACTIVE}, {@link Execution#STATE_CANCELLED},
	 * {@link Execution#STATE_ENDED}, {@link Execution#STATE_INACTIVE},
	 * {@link Execution#STATE_SUSPENDED} or any other state defined by a
	 * specific processDefinition language.
	 */
	private State currentState;

	@Requires(name = "currentTarget", contingency = Contingency.OPTIONAL)
	private ExecutableElement currentTarget;

	private ThreadGroup group = null;

	public ExecutableElement getCurrentTarget() {
		return this.currentTarget;
	}

	@Requires(name = "parentExec", contingency = Contingency.OPTIONAL)
	private Execution parent = null;

	/** are concurrent childExecutions that related to this execution. */
	@Requires(name = "childs", cardinality = Cardinality.COLLECTION, contingency = Contingency.OPTIONAL)
	private final Map<String, Execution> childExecutions = new TreeMap<String, Execution>();

	private long speedTime;

	private ThreadSignal threadSignal;

	/**
	 * The Fractal execution component
	 */
	@Controller
	private Component component;

	// private int cpt_childs;

	public Component getComponent() {
		return this.component;
	}

	/**
	 * Create the scope behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = CREATE)
	public void create() throws CoreException {
		this.log.fine("Fractal execution created: "
				+ FractalHelper.getFractalHelper().getName(this.component));
	}

	public void init(final Component fractalNode) throws CoreException {
		this.component = fractalNode;
		this.log.fine("Fractal execution initiated: "
				+ FractalHelper.getFractalHelper().getName(this.component));
	}

	/**
	 * Start the ExecutionImpl behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = START)
	public void start() throws CoreException {
		this.log.fine("start execution: " + this.getName());
		this.setState(Execution.State.ACTIVE);
		this.group = new ThreadGroup(this.getName());
	}

	/**
	 * Stop the ExecutionImpl behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = STOP)
	public void stop() throws CoreException {
		this.log.finest("stop execution: " + this.getName());
		this.setState(Execution.State.SUSPENDED);
	}

	/**
	 * Destroy the ExecutionImpl behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = DESTROY)
	public void destroy() throws CoreException {
		this.log.fine("Fractal execution destroyed: " + this.getName());
		this.setState(Execution.State.CANCELLED);
	}

	public Scope getParentScope() throws CoreException {
		Scope res = null;
		final Component scopeComponent = FractalHelper.getFractalHelper().getParent(
				this.component);
		try {
			res = (Scope) scopeComponent.getFcInterface("scope");
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
		return res;
	}

	public synchronized void run() throws CoreException {

		// start the current target component
		final ExecutableElement target = this.currentTarget;
		final Component targetComp = FractalHelper
		.getFractalHelper()
		.getComponentByInterface(
				FractalHelper.getFractalHelper().getParent(
						this.getComponent()),
						(org.objectweb.fractal.api.Interface) target, "service");
		if ((targetComp != null)
				&& (!FractalHelper.getFractalHelper().isStarted(targetComp))) {
			FractalHelper.getFractalHelper().startComponent(targetComp);
		}

		if(this.currentTarget != null && this.currentTarget.getNextSelectedExecutableElements() != null) {
			this.currentTarget.getNextSelectedExecutableElements().clear();
			this.currentTarget.getNextSelectedExecutableElements().add(
					this.currentTarget);

			if(!this.stepByStep) {
				this.next();
			}
		} else {
			throw new CoreException("Error initialization exception");
		}
	}

	public void removeChildExecution(final Execution execution)
	throws CoreException {
		if (execution != null) {
			final Component exeComp = FractalHelper.getFractalHelper()
			.getComponentByInterface(
					this.getParentScope().getComponent(),
					(org.objectweb.fractal.api.Interface) execution,
			"service");
			if (exeComp != null) {
				FractalHelper.getFractalHelper().deleteComponent(exeComp);
			}
		}
	}

	/**
	 * @throws CoreException
	 * @see Execution#end(String, boolean)
	 */
	public void end() throws CoreException {
		try {
			if (FractalHelper.getFractalHelper().isBinded(this.component,
			"currentTarget")) {
				// delete binding between node and execution
				final List<Binding> listOfBindings = new ArrayList<Binding>();
				listOfBindings
				.add(new Binding(
						"currentTarget",
						(org.objectweb.fractal.api.Interface) ((Node) this.currentTarget)
						.getComponent().getFcInterface(
						"service")));
				FractalHelper.getFractalHelper().deleteBindings(this.component,
						listOfBindings);
			}
			this.group = new ThreadGroup(this.getName());
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
	}

	public Execution getChildExecution(final String name) {
		return this.childExecutions.get(name);
	}

	public List<Execution> getChildExecutions() {
		final List<Execution> res = new ArrayList<Execution>();
		res.addAll(this.childExecutions.values());
		return res;
	}

	public String getName() throws CoreException {
		return FractalHelper.getFractalHelper().getName(this.component);
	}

	public Execution getParentExecution() {
		return this.parent;
	}

	public State getState() {
		return this.currentState;
	}

	public void setState(final State state) {
		this.currentState = state;
	}

	public void setName(final String name) throws CoreException {
		if (name != null) {
			FractalHelper.getFractalHelper().changeName(this.component, name);
		}
	}

	/**
	 * @throws CoreException
	 * @see Execution#createChildExecution(String)
	 */
	public Execution createChildExecution(final String name) throws CoreException {
		// cpt_childs++;

		// create child execution
		final Execution childExecution = this.getParentScope()
		.createExecution(
				this.getName() + "_child_"
				+ this.childExecutions.size());

		final Component exeChildComp = FractalHelper.getFractalHelper()
		.getComponentByInterface(this.getParentScope().getComponent(),
				(org.objectweb.fractal.api.Interface) childExecution,
		"service");

		FractalHelper.getFractalHelper().startComponent(exeChildComp);

		childExecution.setStepByStep(this.stepByStep);
		childExecution.setSpeedTime(this.speedTime);

		this.log.finest("creating " + childExecution.getName());

		// add it to this execution
		this.addExecution(childExecution);

		return childExecution;
	}

	public void addExecution(final Execution childExecution) throws CoreException {
		try {

			// link child to execution
			List<Binding> listOfBindings = new ArrayList<Binding>();
			listOfBindings
			.add(new Binding(
					"childs_"
					+ String.valueOf(this.childExecutions
							.size()),
							(org.objectweb.fractal.api.Interface) (childExecution)
							.getComponent().getFcInterface("service")));
			FractalHelper.getFractalHelper().addBindings(this.getComponent(),
					listOfBindings);

			// link execution to child
			listOfBindings = new ArrayList<Binding>();
			listOfBindings.add(new Binding("parentExec",
					(org.objectweb.fractal.api.Interface) this.getComponent()
					.getFcInterface("service")));
			FractalHelper.getFractalHelper().addBindings(
					childExecution.getComponent(), listOfBindings);

		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
	}

	public void removeChildExecutions() throws CoreException {
		Execution child = null;
		while ((this.childExecutions.values() != null)
				&& (this.childExecutions.values().size() > 0)) {
			child = this.childExecutions.values().iterator().next();
			log.finest("remove execution: " + child.getName());
			this.removeChildExecution(child);
		}
	}

	public void setParent(final Execution parent) {
		this.parent = parent;
	}

	public synchronized void next() throws CoreException {
		try {

			State actualState = this.currentState;
			final ExecutableElement actualElmt = this.currentTarget;

			// for slowly
			if (this.speedTime > 0) {
				try {
					Thread.sleep(this.speedTime);
				} catch (final InterruptedException e) {
					throw new CoreException(e);
				}
			}

			// for debug
			if ((actualState == State.ACTIVE) && (this.stepByStep == true)) {
				this.currentState = State.SUSPENDED;
			}

			if (actualState == State.ACTIVE) {

				// run next activities
				if ((actualElmt != null)
						&& (actualElmt.getNextSelectedExecutableElements()
								.size() == 1)) {
					// run the next elmt
					this.execute(actualElmt.getNextSelectedExecutableElements()
							.get(0));
				} else if ((actualElmt != null)
						&& (actualElmt.getNextSelectedExecutableElements()
								.size() > 1)) {
					// run multithread activities
					this.log.finest("start multithread executions for execution: "
							+ this.getName());

					// copy the target elmt list
					final List<ExecutableElement> nextElmts = new ArrayList<ExecutableElement>();
					nextElmts.addAll(actualElmt
							.getNextSelectedExecutableElements());

					// remove this current execution of the parent list
					actualElmt.getNextSelectedExecutableElements().clear();

					// ThreadExecution list
					final List<ThreadExecution> childThreadList = new ArrayList<ThreadExecution>();

					final Iterator<ExecutableElement> itNextElmts = nextElmts
					.iterator();
					ExecutableElement nextElmt = null;
					while (itNextElmts.hasNext()) {
						nextElmt = itNextElmts.next();
						final Execution childExecution = this
						.createChildExecution(this.getName()
								+ "_child_"
								+ this.childExecutions.size());

						// get next component
						final ExecutableElement target = nextElmt;
						final Component targetComp = FractalHelper
						.getFractalHelper()
						.getComponentByInterface(
								FractalHelper.getFractalHelper()
								.getParent(this.getComponent()),
								(org.objectweb.fractal.api.Interface) target,
						"service");

						// link the execution to the currentTarget
						if (!FractalHelper.getFractalHelper().isAlreadyBind(
								childExecution.getComponent(), "currentTarget",
								(org.objectweb.fractal.api.Interface) target)) {
							this.getParentScope()
							.linkedExecution2ExecutableElement(
									(Execution) childExecution
									.getComponent()
									.getFcInterface("service"),
									target);
						}

						// link the currentTarget to the child execution
						if (!FractalHelper
								.getFractalHelper()
								.isAlreadyBind(
										targetComp,
										"execution",
										(org.objectweb.fractal.api.Interface) (Execution) childExecution
										.getComponent().getFcInterface(
										"service"))) {
							this.getParentScope()
							.linkedExecutableElement2Execution(
									(ExecutableElement) targetComp
									.getFcInterface("service"),
									(Execution) childExecution
									.getComponent()
									.getFcInterface("service"));
						}

						// run the child execution
						if (!this.isStepByStep()) {
							final ThreadExecution threadChildExecution = new ThreadExecution(
									this.group, childExecution);
							threadChildExecution.start();
							childThreadList.add(threadChildExecution);
						} else {
							childExecution.setStepByStep(true);
							childExecution.run();
						}
					}

					if (!this.isStepByStep()) {
						// wait the end of all childs
						try {
							for (final ThreadExecution thread : childThreadList) {
								thread.join();
								if (thread.getException() != null) {
									this.log.finest("error in child execution: "
											+ thread.getChildExecution()
											.getName());
									throw new CoreException(thread
											.getException());
								} else {
									this.log.finest("child execution "
											+ thread.getChildExecution()
											.getName() + " is " + thread.getChildExecution().getState());
								}
							}
						} catch (final InterruptedException e) {
							throw new CoreException(e);
						}

						// delete ended execution
						this.deleteEndedChildExecution();

						if (this.childExecutions.values().size() == 0) {
							// restart the parent execution
							this.log
							.finest("All childs have finished => restart the parent execution");
							// this.removeChildExecutions();
							if (actualElmt != null) {
								this.currentState = State.ACTIVE;
								actualState = State.ACTIVE;
								this.execute(actualElmt);
							}
						} else {
							this.log.finest("Some executions must be suspended");
							this.log.finest("So, execution suspended");
							this.currentState = State.SUSPENDED;
							actualState = State.SUSPENDED;
						}
					}
				}

				if ((actualState == State.ACTIVE)
						&& (this.currentTarget != null)
						&& (this.currentTarget
								.getNextSelectedExecutableElements().size() == 0)) {
					if (this.childExecutions.size() > 0) {
						this.log.finest("wait the end of child node");
						// if(this.isStepByStep()) {
						Iterator<Execution> it = this.childExecutions.values()
						.iterator();
						while (it.hasNext()) {
							final Execution childExecution = it.next();
							if (childExecution.getState() == State.ENDED) {
								this.signal(childExecution);
								it = this.childExecutions.values().iterator();
							}
						}
						// }
					} else if (this.currentTarget.selectDefaultNextTarget()
							.size() == 0) {

						if (this.currentState != State.ENDED) {
							this.log.finest("Execution ended: " + this.getName());
						}
						// End of process
						this.currentState = State.ENDED;
						actualState = State.ENDED;

						if (this.getParentExecution() == null) {
							// link the currentTarget to the execution
							if (!FractalHelper
									.getFractalHelper()
									.isAlreadyBind(
											actualElmt.getComponent(),
											"execution",
											(org.objectweb.fractal.api.Interface) (Execution) this
											.getComponent()
											.getFcInterface("service"))) {
								this.getParentScope()
								.linkedExecutableElement2Execution(
										(ExecutableElement) actualElmt
										.getComponent()
										.getFcInterface(
										"service"),
										(Execution) this.getComponent()
										.getFcInterface(
										"service"));
							}

							Process p = null;
							try {
								p = (Process) this.getParentScope().getComponent().getFcInterface("process");
							} catch(Exception e) {
								// do nothing
							}
							if((p != null)&&(p.getState() == Process.State.PROCESS_ACTIVE)) {
								p.end(false);
							} 


						} else {
							if (this.threadSignal == null) {
								this.threadSignal = new ThreadSignal(this
										.getParentExecution(), this, Thread
										.currentThread());
								this.threadSignal.start();
							}
						}
					}
				}

			}

		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
	}

	/**
	 * @throws CoreException
	 * @see Execution#execute(Node)
	 */
	public synchronized void execute(final ExecutableElement execElmt)
	throws CoreException {
		this.log.finest("Entering in execute method");

		try {
			if (execElmt != null) {
				final ExecutableElement target = execElmt;

				final Component targetComp = FractalHelper.getFractalHelper()
				.getComponentByInterface(
						FractalHelper.getFractalHelper().getParent(
								this.getComponent()),
								(org.objectweb.fractal.api.Interface) target,
				"service");

				// unbind execution
				FractalHelper.getFractalHelper()
				.deleteLinkWithAnItfClientOfComponent(
						this.getComponent(), "currentTarget");

				// start the target component
				if ((targetComp != null)
						&& (!FractalHelper.getFractalHelper().isStarted(
								targetComp))) {
					FractalHelper.getFractalHelper().startComponent(targetComp);
				}

				this.log.finest("run the next elmt for execution: " + this.getName()
						+ " => " + target.getName());

				// link the execution to the currentTarget
				if (!FractalHelper.getFractalHelper().isAlreadyBind(
						this.getComponent(), "currentTarget",
						(org.objectweb.fractal.api.Interface) target)) {
					this.getParentScope().linkedExecution2ExecutableElement(
							(Execution) this.getComponent().getFcInterface(
							"service"), target);
				}

				// link the currentTarget to the execution
				if (!FractalHelper.getFractalHelper().isAlreadyBind(
						targetComp,
						"execution",
						(org.objectweb.fractal.api.Interface) (Execution) this
						.getComponent().getFcInterface("service"))) {
					this.getParentScope().linkedExecutableElement2Execution(
							(ExecutableElement) targetComp
							.getFcInterface("service"),
							(Execution) this.getComponent().getFcInterface(
							"service"));
				}
				// TODO: Not necessary => Fix fractal bug 
				try {
					if(targetComp.getFcInterface("scope") != null) {
						Node scope = (Node) targetComp.getFcInterface("service");
						scope.setExecution(this);
					}
				} catch(NoSuchInterfaceException e) {
					// do nothing
				}

				// clear next elemt list
				execElmt.getNextSelectedExecutableElements().clear();

				execElmt.execute();
			}
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}

		this.log.finest("Exit of execute method");
	}

	public long getSpeedTime() {
		return this.speedTime;
	}

	public void setSpeedTime(final long time) {
		this.speedTime = time;
	}

	public void runSlowly(final long time) throws CoreException {
		this.setSpeedTime(time);
		this.run();
	}

	public void runStepByStep() throws CoreException {
		this.stepByStep = true;
		this.run();
	}

	public synchronized void signal() throws CoreException {
		this.signal(null);
	}

	public synchronized void signal(final Execution childExecution)
	throws CoreException {
		if (childExecution != null) {
			this.log.finest("wake up of execution: " + this.getName() + " by "
					+ childExecution.getName());
		} else {
			this.log.finest("wake up of execution: " + this.getName()
					+ " by external user");
		}

		// delete ended execution
		this.deleteEndedChildExecution();

		if (childExecution != null) {
			if (this.currentTarget != null) {
				this.currentTarget.getNextSelectedExecutableElements().clear();
				this.currentTarget.getNextSelectedExecutableElements().add(
						this.currentTarget);
			}

		}
		if (this.currentState != State.ENDED) {
			if (childExecution != null) {
				this.currentState = State.SUSPENDED;
			} else {
				this.currentState = State.ACTIVE;
				if (this.getParentExecution() == null) {
					this.log.finest("call next method for execution: "
							+ this.getName());
					this.next();
				} else {
					if(!this.stepByStep) {
						this.log.finest("Start child execution in thread");
						final ThreadExecution threadChildExecution = new ThreadExecution(
								this.group, this);
						threadChildExecution.start();
					} else {
						this.next();
					}
				}
			}
		}

	}

	private void deleteEndedChildExecution() throws CoreException {
		Iterator<Execution> it = this.childExecutions.values().iterator();
		while (it.hasNext()) {
			final Execution childExec = it.next();
			if (childExec.getState() == State.ENDED) {
				this.log.finest("remove the child execution : "
						+ childExec.getName());
				this.removeChildExecution(childExec);
				it = this.childExecutions.values().iterator();
			}
		}
	}

	public boolean isStepByStep() {
		return this.stepByStep;
	}

	public void setStepByStep(final boolean stepByStep) {
		this.stepByStep = stepByStep;
	}

	public void setLog(final Logger logger) {
		this.log = logger;
	}

	@Override
	public String toString() {
		String res = super.toString();
		try {
			if(this.getName() != null) {
				res = this.getName();
			}
		} catch (CoreException e) {
			// do nothing
		}
		return res;
	}

}
