/**
* easy VIPER software - Copyright (c) 2009 PetalsLink, 
* http://www.petalslink.com/ 
*  
* This library is free software; you can redistribute it and/or modify it under 
* the terms of the GNU Lesser General Public License as published by the Free 
* Software Foundation; either version 2.1 of the License, or (at your option) 
* any later version. This library is distributed in the hope that it will be 
* useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
* General Public License for more details. 
*  
* You should have received a copy of the GNU Lesser General Public License 
* along with this library; if not, write to the Free Software Foundation, Inc., 
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 *  
 * ------------------------------------------------------------------------- 
 * $Id$ 
 * ------------------------------------------------------------------------- 
 */ 
package com.ebmwebsourcing.easyviper.core.impl.engine;

import static org.objectweb.fractal.fraclet.types.Step.CREATE;
import static org.objectweb.fractal.fraclet.types.Step.DESTROY;
import static org.objectweb.fractal.fraclet.types.Step.START;
import static org.objectweb.fractal.fraclet.types.Step.STOP;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.Map.Entry;
import java.util.logging.Logger;

import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.fraclet.annotations.Controller;
import org.objectweb.fractal.fraclet.annotations.Interface;
import org.objectweb.fractal.fraclet.annotations.Lifecycle;
import org.objectweb.fractal.fraclet.annotations.Requires;
import org.objectweb.fractal.fraclet.extensions.Membrane;
import org.objectweb.fractal.fraclet.types.Cardinality;
import org.objectweb.fractal.fraclet.types.Contingency;

import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.ExecutableElement;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.Node;
import com.ebmwebsourcing.easyviper.core.api.engine.Scope;
import com.ebmwebsourcing.easyviper.core.api.engine.Transition;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.Behaviour;
import com.ebmwebsourcing.easyviper.core.api.engine.fault.Fault;
import com.ebmwebsourcing.easyviper.core.fractal.FractalHelper;

/**
 * @author Nicolas Salatge - eBM WebSourcing
 */
@org.objectweb.fractal.fraclet.annotations.Component(provides = @Interface(name = "service", signature = Node.class))
@Membrane(controller = "primitive")
public class NodeImpl extends ExecutableElementImpl implements Node {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private Logger log = Logger.getLogger(NodeImpl.class.getName());

	/**
	 * child childNodes
	 */
	@Requires(name = "childNodes", cardinality = Cardinality.COLLECTION, contingency = Contingency.OPTIONAL)
	protected Map<String, Node> childNodes = Collections.synchronizedMap(new TreeMap<String, Node>());

	@Requires(name = "incomingTransitions", cardinality = Cardinality.COLLECTION, contingency = Contingency.OPTIONAL)
	protected Map<String, Transition> incomingTransitions = Collections.synchronizedMap(new TreeMap<String, Transition>());

	@Requires(name = "outgoingTransitions", cardinality = Cardinality.COLLECTION, contingency = Contingency.OPTIONAL)
	protected Map<String, Transition> outgoingTransitions = Collections.synchronizedMap(new TreeMap<String, Transition>());

	@Requires(name = "behaviour", contingency = Contingency.OPTIONAL)
	protected Behaviour behaviour;

	@Requires(name = "parentNode", contingency = Contingency.OPTIONAL)
	private Node parentNode;

	@Requires(name = "execution", contingency = Contingency.OPTIONAL)
	protected Execution execution;

	/**
	 * The Fractal scope component
	 */
	@Controller
	private Component component;

	/**
	 * Default constructor
	 * 
	 * @throws CoreException
	 */
	public NodeImpl() {
		super();
	}

	public Component getComponent() {
		return this.component;
	}

	/**
	 * Create the scope behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = CREATE)
	public void create() throws CoreException {
		this.log.fine("Fractal node created: "
				+ FractalHelper.getFractalHelper().getName(this.component));
	}

	public void init(final Component fractalNode) throws CoreException {
		this.component = fractalNode;
		this.log.fine("Fractal node initiated: "
				+ FractalHelper.getFractalHelper().getName(this.component));
	}

	/**
	 * Start the NodeImpl behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = START)
	public void start() throws CoreException {
		this.log.fine("Fractal node started: " + this.getName());
	}

	/**
	 * Stop the NodeImpl behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = STOP)
	public void stop() throws CoreException {
		this.log.finest("node " + this.getName() + " is stopping... ");

		if ((this.childNodes != null) && (this.childNodes.size() > 0)) {
			final Component parentComp = FractalHelper.getFractalHelper().getParent(
					this.getComponent());
			for (final Node child : this.getChildNodes()) {
				this.log.finest("stop child node: " + this.getName());

				final Component nodeComp = FractalHelper.getFractalHelper()
				.getComponentByInterface(parentComp,
						(org.objectweb.fractal.api.Interface) child,
				"service");
				if ((nodeComp != null)
						&& (FractalHelper.getFractalHelper()
								.isStarted(nodeComp))) {
					FractalHelper.getFractalHelper().stopComponent(nodeComp);
				}
			}
		}

		this.log.finest("node stopped: " + this.getName());
	}

	/**
	 * Destroy the NodeImpl behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = DESTROY)
	public void destroy() throws CoreException {
		this.log.fine("Fractal node destroyed: " + this.getName());

	}

	public String getName() throws CoreException {
		return FractalHelper.getFractalHelper().getName(this.component);
	}

	public void setName(final String name) throws CoreException {
		if (name != null) {
			FractalHelper.getFractalHelper().changeName(this.component, name);
		}
	}

	public void setActivity(final Behaviour behaviour) {
		this.behaviour = behaviour;

	}

	public Behaviour getActivity() {
		return this.behaviour;
	}

	public void execute() throws CoreException {
		try {
			// execute the behaviour of the node
			if (this.behaviour != null) {

				// start the behaviour
				final Component activityComp = FractalHelper
				.getFractalHelper()
				.getComponentByInterface(
						this.execution.getParentScope().getComponent(),
						(org.objectweb.fractal.api.Interface) this.behaviour,
				"service");
				if ((activityComp != null)
						&& (!FractalHelper.getFractalHelper().isStarted(
								activityComp))) {
					FractalHelper.getFractalHelper().startComponent(
							activityComp);
				}


				try {

					// run the behaviour
					this.behaviour.execute();
					//Test if the process is active for the exit activity
					if((this.getExecution().getState() != Execution.State.CANCELLED)&&(this.getExecution().getState() != Execution.State.SUSPENDED)) {
						this.selectDefaultNextTarget();
						// signal the execution
						if (this.execution != null) {
							this.execution.next();
						}
					} else {
						log.finest("this node " + this.getName() + " is " + this.getBehaviour().getState());
						log.finest("this execution " + this.getExecution().getName() + " is " + this.getExecution().getState());
					}

				} catch (CoreException e) {

					Scope currentScope = this.getExecution().getParentScope();

					Scope faultScope = findFaultScope(currentScope, e);
					if(faultScope == null || faultScope.isFaultScope() != null) {
						// fault not catched
						log.finest("fault not catched: " + e.getClass().getName());
						this.execution.setState(Execution.State.ENDED);
						throw e;
					} else {
						// fault catched
						log.finest("fault catched by scope: " + faultScope.getName());
						faultScope.setFaultScope(e);

						if(faultScope.getParentScope().getName().equals(this.execution.getParentScope().getName())) {
							this.nextSelectedExecElmts.clear();

							this.nextSelectedExecElmts.add((Node)faultScope.getComponent().getFcInterface("service"));

							this.execution.next();
						} else {

							this.execution.setState(Execution.State.ENDED);

							faultScope.getParentScope().linkedExecution2ExecutableElement(faultScope.getParentScope().getExecution(), faultScope);
							faultScope.getParentScope().linkedExecutableElement2Execution(faultScope, faultScope.getParentScope().getExecution());

							faultScope.execute();
						}
					}
				}

				log.finest("State of behaviour's node: " + this.getName() + " => " + this.getBehaviour().getState());
			}


		} catch (NoSuchInterfaceException e1) {
			throw new CoreException(e1);
		}


	}

	private Scope findFaultScope(Scope currentScope, Exception e) throws CoreException {
		Scope faultScope = null;

		// test if currentScope is not already a fault scope
		Exception exTest = null;
		Scope current = currentScope;
		while(exTest==null && current != null) {
			exTest = current.isFaultScope();
			current = current.getParentScope();
		}
		if(exTest == null) {
			for(Entry<Fault, Scope> entry: currentScope.getExceptions().entrySet()) {
				if(entry.getKey().getFaultHandler() != null && entry.getKey().getFaultHandler().match(currentScope, entry.getKey(), e, entry.getValue())) {
					faultScope = entry.getValue();
					break;
				}
			}

			if(faultScope == null && currentScope.getParentScope() != null) {
				faultScope = this.findFaultScope(currentScope.getParentScope(), e);
			}
		}
		return faultScope;
	}

	public List<ExecutableElement> selectDefaultNextTarget()
	throws CoreException {
		this.log.finest("select default next target");
		if (this.nextSelectedExecElmts.size() == 0) {
			List<Transition> out = this.getOutgoingTransitions();
			this.log.finest("list of following transition: " + out);
			this.log.finest("this.getParentNode(): " + this.getParentNode());
			this.log.finest("this.getExecution(): " + this.getExecution());

			// choose next element
			if (out.size() == 1) {
				// select outgoing transition
				this.nextSelectedExecElmts.addAll(out);
			} else if (this.getParentNode() != null) {
				this.log.finest("this.getParentNode().getExecution(): " + this.getParentNode().getExecution());
				if(this.getExecution().getParentExecution() != null) {
					if (this.getParentNode().getExecution() == this.getExecution()) {
						this.log.finest("parent node is next target: " + this.getParentNode().getName());
						// select parent node
						this.nextSelectedExecElmts.add(this.getParentNode());
					}
				} else {
					this.log.finest("parent node is next target: " + this.getParentNode().getName());
					// select parent node
					this.nextSelectedExecElmts.add(this.getParentNode());
				}
			}

			// choose default target element
			this.log.finest("choose default target element after " + this.getName()
					+ " for: " + this.getExecution().getName() + " => "
					+ this.nextSelectedExecElmts.size());
		}
		return this.nextSelectedExecElmts;
	}

	public Node getParentNode() {
		return this.parentNode;
	}

	public void setParentNode(Node parentNode) {
		this.parentNode = parentNode;
	}

	public Execution getExecution() {
		return this.execution;
	}

	public List<Node> getChildNodes() {
		return new ArrayList<Node>(this.childNodes.values());
	}

	public Behaviour getBehaviour() {
		return this.behaviour;
	}

	@Override
	public String toString() {
		String res = "";
		try {
			res = "node " + this.getName();
		} catch (final CoreException e) {
			res = e.getMessage();
		}
		return res;
	}

	public List<Transition> getIncomingTransitions() {
		return new ArrayList<Transition>(this.incomingTransitions.values());
	}

	public List<Transition> getOutgoingTransitions() {
		return new ArrayList<Transition>(this.outgoingTransitions.values());
	}

	public Map<String, Transition> getMapOutgoingTransitions() {
		return this.outgoingTransitions;
	}

	public Map<String, Transition> getMapIncomingTransitions() {
		return this.incomingTransitions;
	}

	public void setLog(final Logger logger) {
		this.log = logger;
	}

	public Logger getLogger() {
		return this.log;
	}

	public void setExecution(Execution exec) {
		this.execution = exec;
	}
}
