/**
* easy VIPER software - Copyright (c) 2009 PetalsLink, 
* http://www.petalslink.com/ 
*  
* This library is free software; you can redistribute it and/or modify it under 
* the terms of the GNU Lesser General Public License as published by the Free 
* Software Foundation; either version 2.1 of the License, or (at your option) 
* any later version. This library is distributed in the hope that it will be 
* useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
* General Public License for more details. 
*  
* You should have received a copy of the GNU Lesser General Public License 
* along with this library; if not, write to the Free Software Foundation, Inc., 
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 *  
 * ------------------------------------------------------------------------- 
 * $Id$ 
 * ------------------------------------------------------------------------- 
 */ 
package com.ebmwebsourcing.easyviper.core.impl.engine.thread.service;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Engine;
import com.ebmwebsourcing.easyviper.core.api.engine.thread.service.Service;
import com.ebmwebsourcing.easyviper.core.api.engine.thread.service.ServiceManager;

public class ServiceManagerImpl implements ServiceManager {

	private Logger log = Logger.getLogger(ServiceManagerImpl.class.getName());


	private List<Service> services = new ArrayList<Service>();


	private ScheduledThreadPoolExecutor scheduledThreadPoolExecutor;

	private Engine engine = null;

	public ServiceManagerImpl(Engine engine) throws CoreException {
		this.engine = engine;
		this.addService(AutoFlushMessageServiceImpl.class);
		this.addService(AutoTrashProcessServiceImpl.class);
		
		// Start the listeners in the thread worker
		scheduledThreadPoolExecutor = new ScheduledThreadPoolExecutor(services.size());
	}

	public void shutdownAllServices() {
		scheduledThreadPoolExecutor.shutdown();
		try {
			scheduledThreadPoolExecutor.awaitTermination(5, TimeUnit.SECONDS);
		} catch (InterruptedException e1) {
			// do nothing
		}
	}

	public void restartAllServices() {
		// Start the listeners in the thread worker
		scheduledThreadPoolExecutor = new ScheduledThreadPoolExecutor(services.size());

		for (Service service : services) {
			scheduledThreadPoolExecutor.scheduleWithFixedDelay(service, 0, 1,
					TimeUnit.MILLISECONDS);
		}
	}

	public List<Service> getServices() {
		return this.services;
	}

	public void addService(Class<? extends Service> serviceClass) throws CoreException {
		Service s = findService(serviceClass);
		if(s != null) {
			log.warning("This service already exist: " + s);
		} else if(serviceClass != null) {
			log.fine("creation and adding of service: " + serviceClass);
			Constructor<?> constructor = serviceClass.getConstructors()[0];
			try {
				Service service = (Service) constructor.newInstance(this.engine);
				this.services.add(service);
			} catch (IllegalArgumentException e) {
				throw new CoreException(e);
			} catch (InstantiationException e) {
				throw new CoreException(e);
			} catch (IllegalAccessException e) {
				throw new CoreException(e);
			} catch (InvocationTargetException e) {
				throw new CoreException(e);
			} 
		}
	}

	private Service findService(Class<? extends Service> serviceClass) {
		Service serviceFound = null;
		
		for(Service service: this.services) {
			
			if(service.getClass().equals(serviceClass)) {
				serviceFound = service;
				break;
			}
		}
		return serviceFound;
	}

	public Service removeService(Class<? extends Service> serviceClass) {
		Service serviceToRemove = findService(serviceClass);


		if(serviceToRemove != null) {
			scheduledThreadPoolExecutor.remove(serviceToRemove);
			scheduledThreadPoolExecutor.setCorePoolSize(this.services.size());
			this.services.remove(serviceToRemove);
		}
		return serviceToRemove;
	}

	public void activateService(
			Class<? extends Service> serviceClass)
	throws CoreException {
		Service service = findService(serviceClass);
		if(service != null && service.getRefreshFrequency() > 0) {
			log.finest("this service is activated: " + serviceClass.getSimpleName());
			scheduledThreadPoolExecutor.setCorePoolSize(this.services.size());
			scheduledThreadPoolExecutor.scheduleWithFixedDelay(service, 0, 1,
					TimeUnit.MILLISECONDS);
		} else {
			log.warning("Impossible to find this service: " + serviceClass.getSimpleName());
		}
	}

	public <S extends Service> S getService(Class<S> serviceClass) {
		S service = (S) this.findService(serviceClass);
		return service;
	}


}
