/**
 * BPMN Diagram - SVG/VML web based editor for BPMN Standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.bpmndiagram.business.domain.util;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;

import com.ebmwebsourcing.bpmndiagram.business.domain.to.CPat;
import com.ebmwebsourcing.bpmndiagram.business.domain.to.FlowNode;
import com.ebmwebsourcing.bpmndiagram.business.domain.to.Lane;
import com.ebmwebsourcing.bpmndiagram.business.domain.to.Pool;
import com.ebmwebsourcing.bpmndiagram.business.domain.to.Constants.BPMNObjectGroup;
import com.ebmwebsourcing.bpmndiagram.business.domain.to.Constants.BPMNObjectType;

/**
 * 
 * @author nfleury
 *
 */
public class CPatHelper {

	private CPat cpat;
	
	private FlowNode proxyFlowNode;
	
	private ArrayList<FlowNode> proxyFlowNodes;
	
	public CPatHelper(CPat cpat) {
		this.cpat = cpat;
	}
	
	
	public FlowNode getFlowNodeById(final String id){
		
		@SuppressWarnings("unused")
		CPatVisitor cpv = new CPatVisitor(cpat){
			
	
			@Override
			public void visitFlowNode(FlowNode flowNode) {
				
				if (flowNode.getId().equals(id)){
						setProxyFlowNode(flowNode);
				}
			
			}
			

			
		};
		
		
	return getProxyFlowNode();	
	}
	
	/**
	 * Gets the availables types in current process
	 * @return
	 */
	public ArrayList<BPMNObjectType> getAvailableFlowNodeTypes(){
		
		ArrayList<BPMNObjectType> result = new ArrayList<BPMNObjectType>();
		
		for(FlowNode n:this.retrieveAllFlowNodes()){
			result.add(n.getType());
		}
		
		return result;
	}
	
	/**
	 * Check if cpat contains only the specified types otherwise return false
	 * @param types
	 * @return
	 */
	public boolean cpatContainsOnly(ArrayList<BPMNObjectType> types){
		
		for(BPMNObjectType t : this.getAvailableFlowNodeTypes()){
			
			if (!types.contains(t)) return false;
			
		}
		
		return true;
	}
	
	
	
	
	
	public ArrayList<FlowNode> retrieveAllFlowNodes(){
		
		proxyFlowNodes = new ArrayList<FlowNode>();
		
		@SuppressWarnings("unused")
		CPatVisitor cpv = new CPatVisitor(cpat){
			
			@Override
			public void visitFlowNode(FlowNode flowNode) {
					
				proxyFlowNodes.add(flowNode);
				
			}
			
		};
		
		return getProxyFlowNodes();
	}
	
	
    public Set<FlowNode> retrieveStartEvents(){
        
        Set<FlowNode> result = new HashSet<FlowNode>();
        
        for(FlowNode element:this.retrieveAllFlowNodes()){
            
            //if its an event and if
            //there is no incoming sequence flow
            if (element.getType().getGroup() == BPMNObjectGroup.Event
                && element.getInSeqFlow().size()==0){
                
                result.add(element);
                
            }
            
        }
        
        return result;
        
    }
	
    
    public Set<FlowNode> retrieveEndEvents(){
        
        Set<FlowNode> result = new HashSet<FlowNode>();
        
        for(FlowNode element:this.retrieveAllFlowNodes()){
            
            //if its an event and if
            //there is no incoming sequence flow
            if (element.getType().getGroup() == BPMNObjectGroup.Event
                && element.getOutgoingSeqFlow().size()==0){
                
                result.add(element);
                
            }
            
        }
        
        return result;
    
    }
    
	
	public void setProxyFlowNodes(ArrayList<FlowNode> proxyFlowNodes) {
		this.proxyFlowNodes = proxyFlowNodes;
	}
	
	public ArrayList<FlowNode> getProxyFlowNodes() {
		return proxyFlowNodes;
	}
	
	
	public void setProxyFlowNode(FlowNode proxyFlowNode) {
		this.proxyFlowNode = proxyFlowNode;
	}
	
	public FlowNode getProxyFlowNode() {
		return proxyFlowNode;
	}
	
	
	
	public Pool getPoolById(String id){
		
		for(Pool pool:cpat.getParticipant()){
			if (pool.getId().equals(id)){
				return pool;
			}
		}
		
		return null;
	}
	
	
	public Lane getLaneFromCPatRolesById(String id){

		for(Lane lane:cpat.getRole()){
			
			if (lane.getId().equals(id)){
				return lane;
			}
			
		}

		
		return null;
	}
	
	
	public FlowNode getFlowNodeFromCPatSolutionsById(String id){
		
		for(FlowNode flowNode:cpat.getSolution()){
			
			if (flowNode.getId().equals(id)){
				
				return flowNode;
				
			}
			
		}
		
		return null;
	}
	

	
	/**
	 * Returns the relative path where all cpat data (wsdl,bpel etc...) <br>
	 * should be uploaded/downloaded based on the path of the uploaded wsdl for current CPat
	 * @return
	 */
	public String determineCPatRelativeUploadPath(){
		
		String result = null;
		
		for(FlowNode activity : this.retrieveAllFlowNodes()){
			
			if (activity.getWSDLUrl()!=null){
				
				FlowNodeUtil fu = new FlowNodeUtil(activity);
				
				result = activity.getWSDLUrl().replace(fu.retrieveWSDLFileName(), "");
				
			}
			
		}
		
		
		return result;
	}
	
	
	
	public ArrayList<FlowNode> getFlowNodesByType(BPMNObjectType type){
		
		ArrayList<FlowNode> result = new ArrayList<FlowNode>();
		
		for(FlowNode n:this.retrieveAllFlowNodes()){
			if (n.getType()==type){
				result.add(n);
			}
		}
		
		
		return result;
	}
	
	public ArrayList<FlowNode> getFlowNodesByGroup(BPMNObjectGroup typeGroup){
		
		ArrayList<FlowNode> result = new ArrayList<FlowNode>();
		
		for(FlowNode n:this.retrieveAllFlowNodes()){
			if (n.getType().getGroup()==typeGroup){
				result.add(n);
			}
		}
		
		
		return result;
	}
	
	public boolean everyTaskHasAWSDL(){
		
		for(FlowNode f:this.getFlowNodesByType(BPMNObjectType.Task)){
			
			if (f.getWSDLUrl()==null || f.getWSDLUrl().contains("none")){
				return false;
			}
		
		}
		
		return true;
	}
	
	
	
	public boolean everyEventHasProperties(){
		
		for(FlowNode f:this.getFlowNodesByGroup(BPMNObjectGroup.Event)){
			
			if (f.getAttributes()==null || f.getAttributes().length==0){
				return false;
			}
		
		}
		
		return true;
		
	}
	
	
	
	public void printCPat(){
		
		System.out.println("-------------[CPAT:"+cpat.getName()+"]--------------");
		System.out.println("id:"+cpat.getId());
		System.out.println("category:"+cpat.getCategory());
		System.out.println("problem:"+cpat.getProblem());
		System.out.println("phase:"+cpat.getPhase());
		System.out.println("precondition:"+cpat.getPreCondition());
		System.out.println("postcondition:"+cpat.getPostCondition());
		
		
		new CPatVisitor(cpat){
			@Override
			public void visitPool(Pool pool) {
				
				System.out.println("------Pool:"+pool.getName());
				System.out.println("nb of lanes:"+pool.getLanes().size());
				System.out.println("x:"+pool.getPositionX());
				System.out.println("y:"+pool.getPositionY());
				
			}
			
			@Override
			public void visitLane(Lane lane) {
			
				
				System.out.println("----Lane:"+lane.getName());
				System.out.println("x:"+lane.getPositionX());
				System.out.println("y:"+lane.getPositionY());

			
			}
			
			@Override
			public void visitTask(FlowNode task) {
			
				System.out.println("--Task:"+task.getTitle());
				System.out.println("x:"+task.getPositionX());
				System.out.println("y:"+task.getPositionY());
				System.out.println("out seq flow:"+task.getOutgoingSeqFlow().size());
				System.out.println("in seq flow:"+task.getInSeqFlow().size());
				System.out.println("wsdl url:"+task.getWSDLUrl());
				
			}
			
			@Override
			public void visitDataBasedExclusiveGateway(FlowNode dataBasedExclusiveGateway) {

				System.out.println("--ExclusiveGateway:"+dataBasedExclusiveGateway.getTitle());
				System.out.println("x:"+dataBasedExclusiveGateway.getPositionX());
				System.out.println("y:"+dataBasedExclusiveGateway.getPositionY());
				System.out.println("out seq flow:"+dataBasedExclusiveGateway.getOutgoingSeqFlow().size());
				System.out.println("in seq flow:"+dataBasedExclusiveGateway.getInSeqFlow().size());
				
			}
			
			@Override
			public void visitMessageEvent(FlowNode messageEvent) {
				
				System.out.println("--Message event:"+messageEvent.getTitle());
				System.out.println("x:"+messageEvent.getPositionX());
				System.out.println("y:"+messageEvent.getPositionY());
				System.out.println("out seq flow:"+messageEvent.getOutgoingSeqFlow().size());
				System.out.println("in seq flow:"+messageEvent.getInSeqFlow().size());
				System.out.println("properties:"+messageEvent.getAttributes());
				
			}
			
			@Override
			public void visitEmptyEndEvent(FlowNode emptyEntEvent) {
				
				System.out.println("--Empty end Message event:"+emptyEntEvent.getTitle());
				System.out.println("x:"+emptyEntEvent.getPositionX());
				System.out.println("y:"+emptyEntEvent.getPositionY());
				System.out.println("out seq flow:"+emptyEntEvent.getOutgoingSeqFlow().size());
				System.out.println("in seq flow:"+emptyEntEvent.getInSeqFlow().size());
				System.out.println("properties:"+emptyEntEvent.getAttributes());
			}
			
			
			
		};
		
		System.out.println("-------------[END CPAT]--------------");
	}
	
	
	
}
