/**
 * BPMN Diagram - SVG/VML web based editor for BPMN Standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.activity;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.event.end.EndMessageElementType;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.event.end.EndNoneElementType;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.event.intermediate.IntermediateThrowingMessageElementType;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.event.intermediate.IntermediateEmptyElementType;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.event.intermediate.IntermediateCatchingMessageElementType;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.event.start.StartTopLevelConditionalElementType;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.event.start.StartTopLevelMessageElementType;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.event.start.StartTopLevelNoneType;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.foundation.FlowNodeElement;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.foundation.eventlogic.ActivityMarkerGraphicsEvent;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.foundation.eventlogic.ActivityMarkerHandler;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.foundation.eventlogic.HasTaskMarkersHandlers;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.foundation.eventlogic.TaskMarkerEvent;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.foundation.eventlogic.TaskMarkerHandler;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.gateway.ExclusiveGatewayType;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.gateway.InclusiveGatewayType;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.gateway.ParallelGatewayType;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.widget.ContextMenu;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.widget.ContextMenuBar;
import com.ebmwebsourcing.gwt.raphael.client.core.Path;
import com.ebmwebsourcing.gwt.raphael.client.core.Rectangle;
import com.ebmwebsourcing.gwt.raphael.client.core.Text;
import com.ebmwebsourcing.gwt.raphael.client.diagram.connector.Connectable;
import com.ebmwebsourcing.gwt.raphael.client.diagram.element.DiagramComplexElement;
import com.ebmwebsourcing.gwt.raphael.client.diagram.element.DiagramElement;
import com.ebmwebsourcing.gwt.raphael.client.diagram.element.DiagramElementChildData;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.ComplexElementEvent;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.ComplexElementHandler;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementDragListener;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementDropListener;
import com.ebmwebsourcing.gwt.raphael.client.diagram.type.DiagramElementType;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.MouseOutHandler;
import com.google.gwt.event.dom.client.MouseOverEvent;
import com.google.gwt.event.dom.client.MouseOverHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.json.client.JSONNumber;
import com.google.gwt.json.client.JSONObject;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.MenuItem;
import com.google.gwt.user.client.ui.MenuItemSeparator;
import com.google.gwt.user.client.ui.PopupPanel;

/**
 * A TaskElement instance represents a Task in BPMN Notation. <br/>
 * A TaskElement instance just hold the graphic behavior of a task. Basically,
 * you can add markers to a task, or turn it into a subprocess...
 * 
 * @author nfleury
 * 
 */
public class TaskElement extends FlowNodeElement implements Activity,
		ComplexElementHandler, ClickHandler, HasTaskMarkersHandlers {

	private static final int X_OFFSET_FOR_TASK_MARKERS = 5; // offset from left
															// border
	private static final int Y_OFFSET_FOR_TASK_MARKERS = 5; // offset from top
															// border
	private static final int X_MARGIN = 5; // distance between markers
	private static final int Y_OFFSET_FOR_ACTIVITY_MARKERS = 8; // offset from
																// bottom border
	private static final int INNER_MARGIN = 30;
	private static final int MIN_WIDTH = 100;
	private static final int MIN_HEIGHT = 50;
	
	private static final String GRADIENT = "90-#facdb0-#ffffff";

	// markers

	/**
	 * This enum provides an easy way to add markers to a TaskElement
	 */
	public enum TaskMarkerGraphic {
		NONE, SEND, RECEIVE, USER, MANUAL, BUSINESS_RULE, SERVICE, SCRIPT
	}

	/*
	 * State machine enums
	 */

	/**
	 * General state (subprocess or not)
	 */
	private enum MainState {
		NOT_SUBPROCESS, SUBPROCESS
	}

	/**
	 * Sub-state. Used to determine whether the task element is expanded or
	 * collapsed
	 * 
	 * @author enhan
	 * 
	 */
	private enum SubProcessState {
		EXPANDED, COLLAPSED
	}

	/* Markers management */
	private List<TaskMarkerElement> taskMarkersDisplayed; // displayed task
															// markers
	// displayed activity markers
	private List<ActivityMarkerElement> activityMarkersDisplayed; 

	private String initialLabel;

	// context menu implementation
	// dictionary to lookup for menu items
	private Map<TaskMarkerGraphic, MenuItem> taskMarkerItems; 
	// dictionary to lookup for menu items
	private Map<ActivityMarkerGraphic, MenuItem> activityMarkerItems; 
	private MenuItemSeparator markersExpandSeparator;
	private MenuItem expandCollapseItem;

	/* Drag and drop related stuff */
	private DragNDropController dndController;
	private List<DiagramElementType> acceptedElementTypes;
	private List<DiagramElementChildData> children;

	/* Subprocess */
	private HandlerRegistration registration; // contains registration for a
												// marker (expand or collapse)

	private int widthCollapsed;
	private int heightCollapsed;
	private int widthExpanded;
	private int heightExpanded;

	private MainState mainState;
	private SubProcessState subState;

	/* Contextual menu */
	private PopupPanel popupPanel;

	/**
	 * Constructs a TaskElement
	 * 
	 * @param id
	 *            is the id attribute value of the underlying HTML element
	 * @param elementLabel
	 *            is the label of this task element. It is in fact, the name of
	 *            this task
	 * @param x
	 *            x position of the task. x is in canvas (DiagramPanel) relative
	 *            coordinates.
	 * @param y
	 *            y position of the task. y is in canvas (DiagramPanel) relative
	 *            coordinates.
	 */
	public TaskElement(String id, String elementLabel, int x, int y) {
		super(new Rectangle(id, x, y, 80, 50, 8), new TaskType());

		this.label = new DiagramElement(new Text(id + "-label", elementLabel,
				0, 0));

		int labelX = this.getWidth()/2;
		int labelY = this.getHeight()/2;
		this.addDiagramElement(label, labelX, labelY, true);

		this.isDraggable(true);
		this.isResizable(true);

		// markers init :
		taskMarkersDisplayed = new ArrayList<TaskMarkerElement>();
		activityMarkersDisplayed = new ArrayList<ActivityMarkerElement>();

		dndController = new DragNDropController();

		// addDiagramElementListener(myListener);
		initialLabel = elementLabel;

		acceptedElementTypes = new ArrayList<DiagramElementType>();
		children = new ArrayList<DiagramElementChildData>();

		// DND config :
//		acceptedElementTypes.add(new TaskType());
//		acceptedElementTypes.add(new StartTopLevelNoneType());
//		acceptedElementTypes.add(new StartTopLevelMessageElementType());
//		acceptedElementTypes.add(new StartTopLevelConditionalElementType());
//		acceptedElementTypes.add(new IntermediateEmptyElementType());
//		acceptedElementTypes.add(new IntermediateCatchingMessageElementType());
//		acceptedElementTypes.add(new EndMessageElementType());
//		acceptedElementTypes.add(new EndNoneElementType());
//		acceptedElementTypes.add(new IntermediateThrowingMessageElementType());
//		acceptedElementTypes.add(new ExclusiveGatewayType());
//		acceptedElementTypes.add(new InclusiveGatewayType());
//		acceptedElementTypes.add(new ParallelGatewayType());

		this.connectable = new Connectable(this);
		
		
				// state machine
		mainState = MainState.NOT_SUBPROCESS;

		sinkEvents(Event.MOUSEEVENTS);

		heightCollapsed = 0;
		heightExpanded = 0;
		widthCollapsed = 0;
		widthExpanded = 0;

		popupPanel = new PopupPanel(true);
		// popping over all other zIndexes...
		popupPanel.getElement().getStyle().setZIndex(30000);
		

	}
	
	
	//-------------------------------------------------------------------------
	// 					PUBLIC API
	//-------------------------------------------------------------------------
	
	/**
	 * Activate the marker m
	 * 
	 * @param m
	 */
	public void setMarkerActivated(TaskMarkerGraphic m) {
		if (!taskMarkersDisplayed.isEmpty() && taskMarkersDisplayed.get(0).getMarker()!=m) {
			// remove all markers (the marker actually) in the list
//			if (m != TaskMarkerGraphic.NONE) {
//				setMarkerActivated(m, true);
//			}
			setMarkerActivated(taskMarkersDisplayed.get(0).getMarker(), false);
		}
		if (m != TaskMarkerGraphic.NONE) {
			setMarkerActivated(m, true);
		}
	}
	
	
	
	/**
	 * this method offers a convenient way to determine if a marker is enabled
	 * or not
	 * 
	 * @param m
	 *            the marker which you want to control presence
	 * @return true if the marker is activated, false otherwise
	 */
	public boolean isTaskMarkerActivated(TaskMarkerGraphic m) {
		for (TaskMarkerElement elem : taskMarkersDisplayed) {
			if (elem.getMarker() == m) {
				return true;
			}
		}
		return false;
	}

	/**
	 * This method is provided for convenience. It allows to retrieve a
	 * collection containing the task markers displayed. It might be useful if
	 * you want to check compatibilities between markers before adding a new
	 * one.
	 * 
	 * @return
	 */
	public List<TaskMarkerGraphic> getVisibleTaskMarkers() {
		List<TaskMarkerGraphic> markers = new ArrayList<TaskMarkerGraphic>();
		for (TaskMarkerElement elem : taskMarkersDisplayed) {
			markers.add(elem.getMarker());
		}
		return markers;
	}

	/**
	 * This method is provided for convenience. It allows to retrieve a
	 * collection containing the activity markers displayed.
	 */
	public List<ActivityMarkerGraphic> getActivityMarkersActivated() {
		List<ActivityMarkerGraphic> markers = new ArrayList<ActivityMarkerGraphic>();
		for (ActivityMarkerElement marker : activityMarkersDisplayed) {
			markers.add(marker.getMarker());
		}
		return markers;
	}

	/**
	 * This method returns <code>true</code> if the marker m is activated.
	 * Otherwise, it returns <code>false</code>.
	 */
	public boolean isActivityMarkerActivated(ActivityMarkerGraphic m) {
		for (ActivityMarkerElement marker : activityMarkersDisplayed) {
			if (marker.getMarker() == m) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Activate the specified marker, assuming it does not conflict with
	 * existing markers
	 */
	public void setActivityMarkerActivated(ActivityMarkerGraphic m,
			boolean enable) {
		boolean isActivated = isActivityMarkerActivated(m);

		if (enable) {
			// activate
			if (!isActivated) {

				// we should first check compatibility with the already
				// displayed markers. this is however done by the underlying
				// model

				// layout algorithm
				ActivityMarkerElement newMarker = new ActivityMarkerElement(m,
						getX(), getY());
				if (m == ActivityMarkerGraphic.SUBPROCESS_COLLAPSE
						|| m == ActivityMarkerGraphic.SUBPROCESS_EXPAND) {
					registration = newMarker.addClickHandler(this);
				}

				// trick to get marker width and height before displaying it.
				// this is done to avoid ugly jumping from initial position to
				// the real position calculated further.
				this.getDiagramPanel().add(newMarker, this.getX(), this.getY());
				addDiagramElement(newMarker, getX(), getY(), false);
				newMarker.setVisible(false);
				newMarker.setOpacity("0.0");

				// add this element to the list
				activityMarkersDisplayed.add(newMarker);

				// the marker is now attached. we can get its width and height

				updateActivityMarkersPosition();

				// finally set the new marker visible
				newMarker.setVisible(true);
				newMarker.setOpacity("1.0");

				// notify
				if (m != ActivityMarkerGraphic.SUBPROCESS_COLLAPSE
						&& m != ActivityMarkerGraphic.SUBPROCESS_EXPAND) {
					activityMarkerItems.get(m).setText(
							"Remove " + m.toString() + " marker");
					ActivityMarkerGraphicsEvent.fireAddActivityMarkerGraphic(
							this, m);
				}
			}
		} else {
			// remove
			if (isActivated) {
				ActivityMarkerElement elem = null;
				for (ActivityMarkerElement marker : activityMarkersDisplayed) {
					if (marker.getMarker() == m) {
						elem = marker;
					}
				}

				assert (elem != null);

				activityMarkersDisplayed.remove(elem);
				elem.remove();

				updateActivityMarkersPosition();

				// notify
				if (m != ActivityMarkerGraphic.SUBPROCESS_COLLAPSE
						&& m != ActivityMarkerGraphic.SUBPROCESS_EXPAND) {
					activityMarkerItems.get(m).setText(
							"Add " + m.toString() + " marker");
					ActivityMarkerGraphicsEvent
							.fireRemoveActivityMarkerGraphic(this, m);
				}
			}
		}
	}
	
	/**
	 * Listen to SVG refreshments.
	 */
	public void onSvgRefresh(ComplexElementEvent event) {
		updateActivityMarkersPosition();

	}
	
	/**
	 * add a FlowNodeElement to this. The element becomes a child of this.
	 * @param flowNodeElement
	 * @param x
	 * @param y
	 */
	public void addFlowNodeElement(FlowNodeElement flowNodeElement, int x, int y) {
		int absolutePositionX = getX() + x;
		int absolutePositionY = getY() + y;

		getDiagramPanel().add(flowNodeElement, absolutePositionX,
				absolutePositionY);

		diagramElementDrop(flowNodeElement);

	}
	
	/**
	 * Returns true if this is a subprocess
	 * @return
	 */
	public boolean isSubprocess() {
		return !children.isEmpty();
	}
	
	
	public void onClick(ClickEvent arg0) {
		// click on a marker element
		if (isActivityMarkerActivated(ActivityMarkerGraphic.SUBPROCESS_COLLAPSE)) {

			expandLeave();
			collapseEnter();

		} else {

			collapseLeave();
			expandEnter();

		}
	}

	@Override
	public void refreshSVGPosition() {

		super.refreshSVGPosition();


	}
	
	/**
	 * Retrieve inner subprocess content. Useful to determine what is included
	 * inside this subprocess
	 * 
	 * @return
	 */
	public List<DiagramElement> getSubprocessContent() {
		List<DiagramElement> subprocess = new ArrayList<DiagramElement>();
		for (DiagramElementChildData child : children) {
			subprocess.add(child.getDiagramElement());
		}
		return subprocess;
	}

	

	@Override
	public void setHeight(final int height) {
		super.setHeight(height);
	}


	@Override
	public void setWidth(final int width) {
		super.setWidth(width);
	}
	
	@Override
	public void refresh() {
		super.refresh();
	}
	
	
	
	
	//----------------- registration methods ----------------------------------

	public HandlerRegistration addClickHandler(ClickHandler arg0) {
		return addHandler(arg0, ClickEvent.getType());
	}

	public HandlerRegistration addActivityMarkersHandler(
			ActivityMarkerHandler handler) {
		return addHandler(handler, ActivityMarkerGraphicsEvent.getType());
	}

	public HandlerRegistration addTaskMarkerHandler(TaskMarkerHandler handler) {
		return addHandler(handler, TaskMarkerEvent.getType());
	}


	
	//-------------------------------------------------------------------------
	// 					PROTECTED/PRIVATE API
	//-------------------------------------------------------------------------
	
	

	@Override
	protected void onLoad() {
		super.onLoad();
		isDroppable(false);
		this.getJqueryObject().resizableMinHeight(MIN_HEIGHT);
		this.getJqueryObject().resizableMinWisth(MIN_WIDTH);
		this.getSvgElement().attr("stroke-width", "2");
		this.label.getSvgElement().attr("font-size", "12");
		this.label.getSvgElement().attr("font-weight", "5");
		getSvgElement().attr("fill",GRADIENT);
		setLabel(initialLabel);
		widthCollapsed = getOffsetWidth();
		heightCollapsed = getOffsetHeight();

		// label is centered
//		label.setY(getY() + getOffsetHeight() / 2);
//		label.setX(getX() + getOffsetWidth() / 2);
//		setWidgetPosition(label, label.getX() - getX(), getOffsetWidth() / 2);
	}

	

	/**
	 * Activate a marker inside this task item. Since a task has only one
	 * marker, this algorithm is now private. The code here is kept however
	 * because it might be useful someday
	 * 
	 * @param m
	 *            the marker to unable or disable
	 * @param activated
	 */
	private void setMarkerActivated(TaskMarkerGraphic m, boolean activated) {

		boolean isActivated = isTaskMarkerActivated(m);

		if (activated) {
			// try to enable element
			if (!isActivated) {

				// layout algorithm
				int xPos = X_OFFSET_FOR_TASK_MARKERS;
				for (TaskMarkerElement elem : taskMarkersDisplayed) {
					xPos += elem.getSvgElement().getWidth();
					xPos += X_MARGIN;
				}

				
				//remove all the previous markers
//				for(TaskMarkerElement e:taskMarkersDisplayed){
//					
//					if (!e.getMarker().equals(m)) e.remove();
////					taskMarkersDisplayed.remove(e);
//					
//				}
//				
				
				TaskMarkerElement elem = new TaskMarkerElement(m, xPos,
						Y_OFFSET_FOR_TASK_MARKERS);
				taskMarkersDisplayed.add(elem);
				this.getDiagramPanel().add(elem, this.getX() + xPos,
						this.getY() + Y_OFFSET_FOR_TASK_MARKERS);
				addDiagramElement(elem, xPos, Y_OFFSET_FOR_TASK_MARKERS, false);


				
				
//				if (m != TaskMarkerGraphic.NONE && taskMarkerItems.get(m)!=null) {
//					taskMarkerItems.get(m).setText(
//							"Remove " + m.toString() + " marker");
//				}

				// notify
//				TaskMarkerEvent.fireAddTaskMarker(this, m);
			}
		} else {
			if (isActivated) {
				// remove the marker
				int lastXValid = X_OFFSET_FOR_TASK_MARKERS;
				TaskMarkerElement toRemove = null;
				boolean shift = false;

				for (TaskMarkerElement elem : taskMarkersDisplayed) {
					if (!shift) {
						if (elem.marker == m) {
							toRemove = elem;
							shift = true;
						} else {
							lastXValid += elem.getSvgElement().getWidth()
									+ X_MARGIN;
						}
					} else {
						// caution it appears that setX and getX are in absolute
						// coord
						elem.setX(lastXValid + getX());
						lastXValid += elem.getSvgElement().getWidth()
								+ X_MARGIN;
					}
				}
				assert (toRemove != null);
				// cleaning
				taskMarkersDisplayed.remove(toRemove);
				toRemove.remove();

				
				
//				if (m != TaskMarkerGraphic.NONE) {
//					taskMarkerItems.get(m).setText(
//							"Add " + m.toString() + " marker");
//				}

//				TaskMarkerEvent.fireRemoveMarker(this, m);

			}
		}
	}

	
	private void updateActivityMarkersPosition() {

		// tricky part : recompute positions
		int totalWidth = 0;
		for (int i = 0; i < activityMarkersDisplayed.size(); i++) {
			totalWidth += activityMarkersDisplayed.get(i).getSvgElement()
					.getWidth();
			if (i < activityMarkersDisplayed.size() - 1) {
				totalWidth += X_MARGIN;
			}

		}

		// calculate the first x :
		int xPos = getX() + getOffsetWidth() / 2 - totalWidth / 2;
		for (ActivityMarkerElement elem : activityMarkersDisplayed) {
			int yPos = getY() + getOffsetHeight()
					- elem.getSvgElement().getHeight()
					- Y_OFFSET_FOR_ACTIVITY_MARKERS;

			elem.setY(yPos);
			elem.setX(xPos);
			setWidgetPosition(elem, xPos - getX(), yPos - getY());
			xPos += elem.getSvgElement().getWidth() + X_MARGIN;
		}

	}

	
	/**
	 * 
	 */
	@Override
	protected void setupContextMenu(ContextMenu menu) {
//		menu.addSeparator();
//		taskMarkerItems = new HashMap<TaskMarkerGraphic, MenuItem>();
//
//		ContextMenuBar taskMarkers = new ContextMenuBar(true);
//		taskMarkers.setAutoOpen(true);
//		taskMarkers.setAnimationEnabled(true);
//		// task markers:
//		for (TaskMarkerGraphic marker : TaskMarkerGraphic.values()) {
//			if (marker != TaskMarkerGraphic.NONE) {
//				MenuItem item = new MenuItem("Add " + marker.toString()
//						+ " marker", new TaskMarkerCommand(marker));
//				taskMarkers.addItem(item);
//				taskMarkerItems.put(marker, item);
//			}
//		}
//		menu.addItem("Task markers", taskMarkers);
//
//		activityMarkerItems = new HashMap<ActivityMarkerGraphic, MenuItem>();
//
//		ContextMenuBar activityMarkers = new ContextMenuBar(true);
//		activityMarkers.setAutoOpen(true);
//		activityMarkers.setAnimationEnabled(true);
//		for (ActivityMarkerGraphic marker : ActivityMarkerGraphic.values()) {
//			if (marker != ActivityMarkerGraphic.SUBPROCESS_COLLAPSE
//					&& marker != ActivityMarkerGraphic.SUBPROCESS_EXPAND) {
//				MenuItem item = new MenuItem("Add " + marker.toString()
//						+ " marker", new ActivityMarkerCommand(marker));
//				activityMarkers.addItem(item);
//				activityMarkerItems.put(marker, item);
//			}
//		}
//		menu.addItem("Activity markers", activityMarkers);
//
//		markersExpandSeparator = menu.addSeparator();
//
//		expandCollapseItem = new MenuItem("Expand", new Command() {
//
//			public void execute() {
//				if (mainState == MainState.SUBPROCESS) {
//					if (subState == SubProcessState.COLLAPSED) {
//						collapseLeave();
//						expandEnter();
//					} else {
//						expandLeave();
//						collapseEnter();
//					}
//				}
//			}
//		});
//
//		markersExpandSeparator.setVisible(false);
//		expandCollapseItem.setVisible(false);
//		menu.addItem(expandCollapseItem);
//		taskMarkers.getElement().getStyle().setZIndex(30000);
	}

	/*  ************************ DND related stuffs ***************************** */

	private boolean isAcceptableElement(DiagramElement elem) {
		return acceptedElementTypes.contains(elem.getDiagramElementType());
	}

	private boolean isChild(DiagramElement elem) {
		for (DiagramElementChildData child : children) {
			if (child.getDiagramElement() == elem) {
				return true;
			}
		}
		return false;
	}

	private void diagramElementHover(DiagramElement elem) {
		elem.getJqueryObject().draggableRevert(!isAcceptableElement(elem));
		if (isAcceptableElement(elem)) {
			// paint in blue or something :)
			setBackgroundColor("blue");
			setOpacity("0.5");

		} else {
			// paint in red :)
			setBackgroundColor("red");
			setOpacity("0.5");
		}
	}

	private void diagramElementOut() {
//		setBackgroundColor("none");
		getSvgElement().attr("fill",GRADIENT);
		setOpacity("1");
	}

	

	private void diagramElementDrop(DiagramElement elem) {
		diagramElementOut();

		if (isAcceptableElement(elem)) {
			boolean stateChange = children.isEmpty();

			
			// attach it
			children.add(new DiagramElementChildData(elem,
					elem.getX() - getX(), elem.getY() - getY(), false));
			// enlarge me if needed !
			if (!stateChange && subState == SubProcessState.COLLAPSED){
				elem.setVisible(false);
			}
			
			if (stateChange) {
				notSubprocessLeave();
				subprocessEnter();
			}

			
			if (getX() + getOffsetWidth() < elem.getX() + elem.getOffsetWidth()
					+ INNER_MARGIN) {
				int newWidth = getOffsetWidth()
						+ (elem.getX() + elem.getOffsetWidth() - (getX() + getOffsetWidth()))
						+ INNER_MARGIN;
				setWidth(newWidth);
//				refreshSVGsize();
			}

			if (getY() + getOffsetHeight() < elem.getY()
					+ elem.getOffsetHeight() + INNER_MARGIN) {
				int newHeight = getOffsetHeight()
						+ (elem.getY() + elem.getOffsetHeight() - (getY() + getOffsetHeight()))
						+ INNER_MARGIN;
				setHeight(newHeight);
//				refreshSVGsize();
			}

			elem.addDragListener(dndController);

			// check zIndex
			if (elem.getZIndex() <= this.getZIndex()) {
				elem.setZIndex(getZIndex() + 1);
			}
			

			

		} else {
			// go away !!
			JSONObject json = new JSONObject();
			json.put("y", new JSONNumber(elem.getContext().getLastYPosition()));
			json.put("x", new JSONNumber(elem.getContext().getLastXPosition()));
			elem.getSvgElement().animate(json.getJavaScriptObject(), 400);

		}

	}

	private void childDroppedInsideMe(DiagramElement droppedElement) {
		diagramElementOut();

		// check margins

		updateChildPosition(droppedElement, droppedElement.getX() - getX(),
				droppedElement.getY() - getY());

	}

	private void childDroppedOutsideMe(DiagramElement elem) {
		// remove from me - no longer a child :
		if (!children.isEmpty()) {

			int i = 0;
			while (i < children.size()
					&& children.get(i).getDiagramElement() != elem) {
				i++;
			}

			// unlink the element
			// remove listener
			children.get(i).getDiagramElement().removeDiagramDragListener(
					dndController);
			// remove from my references
			children.remove(i);

			// check if we're still having children
			if (children.isEmpty()) {
				subprocessLeave();
				notSubrocessEnter();
			}
		}
	}

	

	@Override
	protected void attachDefaultListeners() {
		super.attachDefaultListeners();
		//addDropListener(dndController);
	}


	@Override
	protected void refreshSVGsize() {
		// first of all, check boundaries.
		super.refreshSVGsize();
		// update
		updateLabelPosition();
		updateActivityMarkersPosition();

	}

	private void updateChildPosition(DiagramElement child, int x, int y) {
		for (DiagramElementChildData cdata : children) {

			if (cdata.getDiagramElement() == child) {
				cdata.setX(x);
				cdata.setY(y);
			}

		}
	}

	private boolean droppedInsideMe(DiagramElement elem) {
		return (elem.getX() <= getX() + getOffsetWidth()
				&& elem.getX() >= getX()
				&& elem.getY() <= getY() + getOffsetHeight() && elem.getY() >= getY());
	}

	// -------------------------------------------------------------------------
	// STATE MACHINE IMPLEMENTATION
	// -------------------------------------------------------------------------

	// main state
	private void subprocessEnter() {
		mainState = MainState.SUBPROCESS;
		// first time we are going to be subprocess
		if (heightExpanded == 0) {
			heightExpanded = heightCollapsed;
			widthExpanded = widthCollapsed;
		}
		expandEnter();
		refresh();
	}

	private void subprocessLeave() {
		switch (subState) {
		case COLLAPSED:
			collapseLeave();
			break;
		case EXPANDED:
			expandLeave();
		}
		expandCollapseItem.setVisible(false);
		markersExpandSeparator.setVisible(false);
	}

	private void notSubrocessEnter() {
		mainState = MainState.NOT_SUBPROCESS;
		// restore "collapsed size"
		setWidth(widthCollapsed);
		setHeight(heightCollapsed);
//		refreshSVGsize();
		// label is centered
		label.setY(getY() + getOffsetHeight() / 2);
		label.setX(getX() + getOffsetWidth() / 2);
		setWidgetPosition(label, label.getX() - getX(), getOffsetWidth() / 2);
		// min size update
		this.getJqueryObject().resizableMinHeight(MIN_HEIGHT);
		this.getJqueryObject().resizableMinWisth(MIN_WIDTH);
	}

	private void notSubprocessLeave() {
		// backup size
		widthCollapsed = getOffsetWidth();
		heightCollapsed = getOffsetHeight();
	}

	private void expandEnter() {
		subState = SubProcessState.EXPANDED;

		setWidth(widthExpanded);
		setHeight(heightExpanded);

		refreshSVGsize();
		// do expend
		for (DiagramElementChildData child : children) {
			child.getDiagramElement().show();
			child.getDiagramElement().refreshSVGPosition();
		}
		setActivityMarkerActivated(ActivityMarkerGraphic.SUBPROCESS_COLLAPSE,
				true);
		// label position on top
		label.setY(getY() + 10);
		label.setX(getX() + getOffsetWidth() / 2);
		setWidgetPosition(label, label.getX() - getX(), 10);

		// min size update
		this.getJqueryObject().resizableMinHeight(
				calculateSubprocessMinHeight());
		this.getJqueryObject().resizableMinWisth(calculateSubprocessMinWidth());

		expandCollapseItem.setVisible(true);
		expandCollapseItem.setText("Collapse");
		markersExpandSeparator.setVisible(true);

	}

	private void expandLeave() {
		widthExpanded = getOffsetWidth();
		heightExpanded = getOffsetHeight();
		// clean handler reference
		registration.removeHandler();
		setActivityMarkerActivated(ActivityMarkerGraphic.SUBPROCESS_COLLAPSE,
				false);
	}

	private void collapseEnter() {
		subState = SubProcessState.COLLAPSED;
		setWidth(widthCollapsed);
		setHeight(heightCollapsed);
//		refreshSVGsize();
		// do collapse
		for (DiagramElementChildData child : children) {
			child.getDiagramElement().hide();
		}
		setActivityMarkerActivated(ActivityMarkerGraphic.SUBPROCESS_EXPAND,
				true);
		// label is centered
		label.setY(getY() + getOffsetHeight() / 2);
		setWidgetPosition(label, label.getX() - getX(), getOffsetWidth() / 2);

		// min size update
		this.getJqueryObject().resizableMinHeight(MIN_HEIGHT);
		this.getJqueryObject().resizableMinWisth(MIN_WIDTH);
		expandCollapseItem.setVisible(true);
		expandCollapseItem.setText("Expand");
		markersExpandSeparator.setVisible(true);
	}

	private void collapseLeave() {
		widthCollapsed = getOffsetWidth();
		heightCollapsed = getOffsetHeight();
		registration.removeHandler();
		setActivityMarkerActivated(ActivityMarkerGraphic.SUBPROCESS_EXPAND,
				false);
	}

	private void updateLabelPosition() {
		switch (mainState) {
		case NOT_SUBPROCESS:
			// label is centered
			if (label.isLoaded()){
				
				int newLX = (getWidth()/2);  
				int newLY = (getHeight()/2);
				
				label.setY(newLY);
				label.setX(newLX);
			
//			setWidgetPosition(label, label.getX() - getX(),
//					getOffsetWidth() / 2);
			}
			
			break;
		case SUBPROCESS:
			switch (subState) {
			case COLLAPSED:
				// label is centered
				label.setY(getY() + getOffsetHeight() / 2);
				label.setX(getX() + getOffsetWidth() / 2);
				setWidgetPosition(label, label.getX() - getX(),
						getOffsetWidth() / 2);

				widthCollapsed = getOffsetWidth();
				heightCollapsed = getOffsetHeight();
				break;
			case EXPANDED:
				// label position on top
				label.setY(getY() + 10);
				label.setX(getX() + getOffsetWidth() / 2);
				setWidgetPosition(label, label.getX() - getX(), 10);

				heightExpanded = getOffsetHeight();
				widthExpanded = getOffsetWidth();
				break;
			}
		}

	}

	

	// min height & width calculation

	private int calculateSubprocessMinWidth() {
		// some fun algo :)
		int minSize = 0;
		for (DiagramElementChildData child : children) {
			DiagramElement elem = child.getDiagramElement();
			if (elem.getX() + getOffsetWidth() - getX() > minSize) {
				minSize = elem.getX() + elem.getOffsetWidth() - getX();
			}
		}
		// add a margin
		// hack ;)
		minSize += INNER_MARGIN;
		// min size can't be less than initial min
		if (minSize < MIN_WIDTH) {
			minSize = MIN_WIDTH;
		}

		return minSize;
	}

	private int calculateSubprocessMinHeight() {
		// some fun algo :)
		int minSize = 0;
		for (DiagramElementChildData child : children) {
			DiagramElement elem = child.getDiagramElement();
			if (elem.getY() + getOffsetHeight() - getY() > minSize) {
				minSize = elem.getY() + elem.getOffsetHeight() - getY();
			}
		}
		// add a margin
		minSize += INNER_MARGIN;
		// min size can't be less than initial min
		if (minSize < MIN_HEIGHT) {
			minSize = MIN_HEIGHT;
		}

		return minSize;
	}

	@Override
	public void remove() {
		super.remove();
		for (DiagramElementChildData cd : children) {
			cd.getDiagramElement().remove();
		}
	}

	@Override
	protected boolean isInsideLabelElement(int x, int y) {
		int labelX = label.getSvgElement().getX();
		int labelY = label.getSvgElement().getY();
		int labelW = label.getSvgElement().getWidth();
		int labelH = label.getSvgElement().getHeight();
		labelX -= labelW / 2;
		labelY -= labelH / 2;

		boolean xOk = x >= labelX && x <= labelX + labelW;
		boolean yOk = y >= labelY && y <= labelY + labelH;

		return yOk && xOk;
	}
	
	@Override
	public boolean preferDeffered() {
		return false;
	}

	// -------------------------------------------------------------------------
	// INNER CLASSES
	// -------------------------------------------------------------------------

	/**
	 * A TaskMarkerElement is an element added to a TaskElement
	 * 
	 * @author enhan
	 * 
	 */
	private class TaskMarkerElement extends DiagramElement {

		private TaskMarkerGraphic marker;

		public TaskMarkerElement(TaskMarkerGraphic m, int x, int y) {
			super(new Path(DOM.createUniqueId(), PathHelper
					.pathForTaskMarker(m), x, y));
			marker = m;

		}

		public TaskMarkerGraphic getMarker() {
			return marker;
		}

		@Override
		protected void onLoad() {
			super.onLoad();
			this.getSvgElement().attr("fill", "black");
			this.getSvgElement().attr("stroke-width", "0.5");
		}

	}

	private class DragNDropController implements DiagramElementDropListener,
			DiagramElementDragListener {

		public void onActivate(DiagramElement diagramElement) {
		}

		public void onDeactivate(DiagramElement diagramElement) {
		}

		public void onDrop(DiagramElement targetElement,
				DiagramElement droppedElement) {
			if (isChild(droppedElement)) {

				if (targetElement == TaskElement.this) {
					childDroppedInsideMe(droppedElement);
				} else {
					// this will never happen
					childDroppedOutsideMe(droppedElement);
				}

			} else if (targetElement == TaskElement.this) {
				// new child ?
				diagramElementDrop(droppedElement);
			}

		}

		public void onOut(DiagramElement targetElement,
				DiagramElement outElement) {
			diagramElementOut();
		}

		public void onOver(DiagramElement diagramElement,
				DiagramElement overElement) {
			if (diagramElement == TaskElement.this && isChild(overElement))
				;
			diagramElementHover(overElement);
		}

		public void onDrag(DiagramElement diagramElement) {

		}

		public void onStart(DiagramElement diagramElement) {

		}

		public void onStop(DiagramElement diagramElement) {
			// a child element stopped somewhere !
			// check if it is inside me
			if (isChild(diagramElement) && !droppedInsideMe(diagramElement)) {
				childDroppedOutsideMe(diagramElement);
			}
		}

	}

	private class TaskMarkerCommand implements Command {

		private TaskMarkerGraphic associatedMarker;

		public TaskMarkerCommand(TaskMarkerGraphic marker) {
			associatedMarker = marker;
		}

		public void execute() {
			if (isTaskMarkerActivated(associatedMarker)) {
				setMarkerActivated(TaskMarkerGraphic.NONE);
			} else {
				setMarkerActivated(associatedMarker);
			}
		}
	}

	private class ActivityMarkerCommand implements Command {

		private ActivityMarkerGraphic associatedMarker;

		public ActivityMarkerCommand(ActivityMarkerGraphic marker) {
			associatedMarker = marker;
		}

		public void execute() {
			setActivityMarkerActivated(associatedMarker,
					!isActivityMarkerActivated(associatedMarker));
		}
		
	}

	
	
}
