/**
 * BPMN Diagram - SVG/VML web based editor for BPMN Standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.swimlane;

import java.util.ArrayList;

import com.ebmwebsourcing.bpmndiagram.business.domain.to.Lane;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.foundation.eventlogic.HasNameHandlers;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.foundation.eventlogic.NameEvent;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.bpmn1.foundation.eventlogic.NameHandler;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.widget.PopupEditor;
import com.ebmwebsourcing.bpmndiagram.presentation.gwt.client.widget.PopupTextLineEditor;
import com.ebmwebsourcing.gwt.raphael.client.core.Rectangle;
import com.ebmwebsourcing.gwt.raphael.client.core.Text;
import com.ebmwebsourcing.gwt.raphael.client.diagram.element.DiagramComplexElement;
import com.ebmwebsourcing.gwt.raphael.client.diagram.element.DiagramElement;
import com.ebmwebsourcing.gwt.raphael.client.diagram.element.DiagramElementChildData;
import com.ebmwebsourcing.gwt.raphael.client.diagram.element.DiagramSortableElement;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementListenerAdapter;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementResizableListenerAdapter;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementSortableListenerAdapter;
import com.ebmwebsourcing.gwt.raphael.client.diagram.semantic.Selectable;
import com.google.gwt.dom.client.Element;
import com.google.gwt.dom.client.NativeEvent;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.DoubleClickEvent;
import com.google.gwt.event.dom.client.DoubleClickHandler;
import com.google.gwt.event.logical.shared.CloseEvent;
import com.google.gwt.event.logical.shared.CloseHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.DeferredCommand;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.gwt.user.client.ui.PopupPanel.PositionCallback;


/**
 * 
 * @author nfleury
 *
 */
public class PoolElement extends DiagramComplexElement implements Selectable, HasNameHandlers{
	
	//Graphical components
	private DiagramSortableElement dropZone;
	private DiagramComplexElement labelZone;
	private PopupEditor<String> labelEditor;
	private LabelClickHandler labelHandler;
	
	private ArrayList<LaneElement> lanes = new ArrayList<LaneElement>();

	private DiagramElement label;
	private String labelTxt;
	
	private ArrayList<PoolListener> listeners = new ArrayList<PoolListener>();

	
	public PoolElement(String id,String elementLabel, int x, int y) {
	
		super(new Rectangle(id, x, y, 500, 200, 0),new PoolType());
		
		this.dropZone 	= new DiagramSortableElement(new Rectangle(id+"dropZone",0,0,480,200,0));

		
		this.labelZone 	= new DiagramComplexElement(new Rectangle(id+"LaneLabelZone", 0, 0, 20, 200, 0));
		
		this.label = new DiagramElement(new Text(id+"-label",elementLabel,50,35));
		
		this.labelZone.addDiagramElement(label, 10, 90, false);
		
		
		this.addDiagramElement(labelZone, 0, 0, true);
		this.addDiagramElement(dropZone, 20, 0, true);
		
		this.isDraggable(true);
		this.isResizable(true);


		this.dropZone.addStyleName("dropZone");
		
		labelHandler = new LabelClickHandler();
		labelEditor = new PopupTextLineEditor();
		labelEditor.addCloseHandler(labelHandler);
		addClickHandler(labelHandler);
		labelZone.addDoubleClickHandler(labelHandler);
		
		
   }
	

	
	public void setLabel(String label){
		this.labelTxt = label;
		
		this.label.getSvgElement().attr("text", label);
	}
	
	public String getLabel() {
		return this.labelTxt;
	}
	
	
	public void addLane(final LaneElement lane){
		

			PoolElement.this.dropZone.addDiagramElement(lane);
			PoolElement.this.lanes.add(lane);
	
			lane.setParentPool(PoolElement.this);
					
			resize();
		

			
			//fire event
			for(PoolListener listener:listeners){
				listener.onReceiveLane(lane);
			}
	
	}
	
	public void removeLane(LaneElement lane){

		resize();
		
		//fire event
		for(PoolListener listener:listeners){
			listener.onRemoveLane(lane);
		}
		
		this.lanes.remove(lane);
	}
	

	
	
	
	@Override
	protected void onLoad() {		
		super.onLoad();
		
		this.dropZone.getSvgElement().attr("stroke-width", "2");
//		this.dropZone.getSvgElement().attr("stroke", "blue");
		this.labelZone.getSvgElement().attr("stroke-width", "2");
		this.labelZone.getSvgElement().attr("fill", "0-#e1e1e1-#ffffff");
		this.getSvgElement().attr("stroke-width", "2");
		
		this.getJqueryObject().css("z-index","5000");
		
		this.dropZone.connectWith(".dropZone");
		
		//Label
		this.label.getSvgElement().attr("font-size", "14");
		this.label.getSvgElement().attr("font-weight", "5");
		this.label.getSvgElement().rotate(-90, true);
		

	}
	
	
	public int getLanesTotalHeight(){
		
		int total = 0;
		for(LaneElement l:lanes){
			
			total = l.getHeight() + total;
			
		}
		
		return total;
	}
	
	public void resize(){

		int height = getLanesTotalHeight();
		int width  = getLongestLaneWidth();
		
		if (height>0){
			
			this.setHeight(height);
			
		}
		
		
		if (width > 0){
			int newPoolWidth = width + labelZone.getOffsetWidth();
			this.setWidth(newPoolWidth);
		}
		
		// center label
		
		

		
	}
	
	
	private int getLongestLaneWidth(){
		
		int longest = -1;
		
		for(LaneElement lane : lanes){
			
			if (lane.getOffsetWidth()>longest){
				longest = lane.getOffsetWidth();
			}
			
		}
		
		return longest;
	}
	
	
	public ArrayList<LaneElement> getLanes(){
		return lanes;
	}
	
	
	public void addListener(PoolListener listener){
		listeners.add(listener);
	}
	
	
	@Override
	protected void refreshSVGsize() {
		super.refreshSVGsize();
		
		dropZone.setWidth(this.getWidth()-labelZone.getWidth());
		dropZone.setHeight(this.getHeight());
		labelZone.setHeight(this.getHeight()-1);
		
		this.label.setY(this.getHeight()/2);
		
	}
	
	
	@Override
	protected void attachDefaultListeners() {
		super.attachDefaultListeners();

		/////////////
		//Auto resize the pool when it receives a lane
		/////////
		this.addListener(new PoolListenerAdapter(){
			@Override
			public void onReceiveLane(LaneElement receivedlane) {
				
				PoolElement.this.resize();
			
			}
		});
		
		this.dropZone.addSortableListerner(new DiagramElementSortableListenerAdapter(){
			@Override
			public void onReceive(DiagramElement sortableElement, DiagramElement receivedElement) {
				
				addLane((LaneElement)receivedElement);
//				PoolElement.this.resize();
				
				receivedElement.getJqueryObject().css("left","0px");
				((LaneElement)receivedElement).getDropZone().getJqueryObject().css("left","20px");
				((LaneElement)receivedElement).getDropZone().getJqueryObject().css("position","relative");
				PoolElement.this.getJqueryObject().resizableMinHeight(PoolElement.this.getLanesTotalHeight());
			}
			
			@Override
			public void onRemove(DiagramElement sortableElement, DiagramElement removedElement) {

				removeLane((LaneElement)removedElement);
				PoolElement.this.resize();
				
			}
			
			@Override
			public void onOver(DiagramElement sortableElement, DiagramElement overElement) {
				
//				DiagramSortableElement sortable = (DiagramSortableElement) sortableElement;
//				
//				int height = PoolElement.this.dropZone.getChildrenTotalHeight() + overElement.getOffsetHeight();
//				
//				if (height > 0 && sortable.containsChild(overElement)==false){
//					
//					PoolElement.this.setHeight(""+height);
//					PoolElement.this.refreshSVGsize();
//				
//				}
				
			}
			
			@Override
			public void onOut(DiagramElement sortableElement, DiagramElement outElement) {
				
					
				PoolElement.this.resize();
				
			
			}
			
			
			
			
			
		});
		
		
		/////////
		//Pool resizing handling
		//////////////
//		this.addResizableListener(new DiagramElementResizableListenerAdapter(){
//			
//			@Override
//			public void onResize() {
//				
//				//Limit the resizing height to the sum of the height of all childs 
//				PoolElement.this.getJqueryObject().resizableMinHeight(PoolElement.this.dropZone.getChildrenTotalHeight());
//				
//				
//			}
//			
//		});
		
		
		
		
	}



	public void isDeselected() {
		this.setBorderColor("black");
		for(DiagramElementChildData cdata:this.getDiagramElementChildren()){
			
			cdata.getDiagramElement().setBorderColor("black");
		
	}
		
	}



	public void isSelected() {
		this.setBorderColor("red");
		for(DiagramElementChildData cdata:this.getDiagramElementChildren()){
			
				cdata.getDiagramElement().setBorderColor("red");
			
		}
		
		this.toFront();
		
	}

	public HandlerRegistration addNameHandler(NameHandler handler) {
		return addHandler(handler, NameEvent.getType());
	}

	
	
	public ArrayList<PoolListener> getListeners() {
		return listeners;
	}
	
	public HandlerRegistration addClickHandler(ClickHandler handler) {
		return addHandler(handler, ClickEvent.getType());
	}
	
	private void showLabelEditor(){
		labelEditor.setValue(getLabel());
		// show label editor in center
		labelEditor.setPopupPositionAndShow(new PositionCallback() {
		
			
			public void setPosition(int width, int height) {
				
				int xPos = label.getAbsoluteLeft()-width/2;
				int yPos = label.getAbsoluteTop()-height/2;
				labelEditor.setPopupPosition(xPos, yPos);
			}
		});
		labelEditor.focusOn();
	}
	
	private void updateName(String name){
		setLabel(name);
		NameEvent.fire(this, name);
	}
	
	//******************************
	// Inner classes
	//******************************
	
	private class LabelClickHandler implements ClickHandler, CloseHandler<PopupPanel>, DoubleClickHandler{

	public void onClick(ClickEvent event) {
			int relativeX = getRelativeX(event.getNativeEvent(),getDiagramPanel().getElement());
			int relativeY = getRelativeY(event.getNativeEvent(), getDiagramPanel().getElement());
			
			int labelX = label.getSvgElement().getX();
			int labelY = label.getSvgElement().getY();
			int labelW = label.getSvgElement().getWidth();
			int labelH = label.getSvgElement().getHeight();
			labelX -= labelH/2;
			labelY -= labelW/2;
			
			boolean xOk = relativeX >= labelX && relativeX <= labelX + labelH;
			boolean yOk = relativeY >= labelY && relativeY <= labelY + labelW;
			if(xOk && yOk){
				showLabelEditor();
			}
		}
		
		private int getRelativeX(NativeEvent e, Element target){
			return e.getClientX() - target.getAbsoluteLeft() + target.getScrollLeft() +
		      target.getOwnerDocument().getScrollLeft();

		}
		
		private int getRelativeY(NativeEvent e, Element target){
			return e.getClientY() - target.getAbsoluteTop() + target.getScrollTop() + target.getOwnerDocument().getScrollTop();
		}

		public void onClose(CloseEvent<PopupPanel> event) {
			updateName(labelEditor.getValue());
		}

		public void onDoubleClick(DoubleClickEvent arg0) {
			showLabelEditor();
			
		}
		
	}
}
