/**
 * geasy-diagram-editor - A project for editing diagrams based on OMG Diagram Definition standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.geasytools.diagrameditor;


import com.ebmwebsourcing.geasytools.diagrameditor.api.events.loader.IDiagramViewLoaderHandler;
import com.ebmwebsourcing.geasytools.diagrameditor.api.events.validation.IRuleNotificationHandler;
import com.ebmwebsourcing.geasytools.diagrameditor.api.events.validation.IValidationHandler;
import com.ebmwebsourcing.geasytools.diagrameditor.api.graphic.IDiagramView;
import com.ebmwebsourcing.geasytools.diagrameditor.api.modeleditor.IEditorView;
import com.ebmwebsourcing.geasytools.diagrameditor.api.validation.IConformityRule;
import com.ebmwebsourcing.geasytools.diagrameditor.api.validation.IRuleNotificationRegistry;
import com.ebmwebsourcing.geasytools.diagrameditor.api.validation.IRuleNotificationView;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IDiagram;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IMainModelElement;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IModelElement;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.loader.ElementLoadedEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.loader.LoadErrorEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.loader.LoadSuccessEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.loader.LoadingStateChangeEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.HideContextualNotificationsEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.HideRulesNotificationWithLevelEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.NotificationSelectedEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.RuleResolvedEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.ShowContextualNotificationsEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.ShowRulesNotificationWithLevelEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.ValidationFailureEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.ValidationStartEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.ValidationSuccessEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.ValidationWarningEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.validation.DiagramValidationException;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.validation.RuleNotificationRegistry;
import com.google.gwt.event.shared.GwtEvent;
import com.google.gwt.event.shared.HandlerManager;

public class DiagramController implements  IDiagramController {

	private HandlerManager handlerManager;
	private DiagramInterchangeModelBuilder diBuilder;
	private DiagramEditorModelBuilder editorModelBuilder;
	
	private DiagramElementViewStateHandler graphicStateHandler;
	private DiagramElementViewSyntaxModelBinderHandler syntaxModelBinderHandler;
	private DiagramValidationHandler diagramViewsValidationHandler;
	
	private DiagramViewLoader diagramViewLoader;
	
	private IEditorView editorView;
	private IDiagramView diagramGraphic;
	
	private IMainModelElement mainModelElement;
	
	private IRuleNotificationRegistry ruleNotificationRegistry;
	private IRuleNotificationView ruleNotificationView;
	
	public DiagramController(IDiagramView diagramUI,IMainModelElement mainModel) {
		
		this.diagramGraphic			= diagramUI;
		this.diBuilder		 		= new DiagramInterchangeModelBuilder(diagramUI,this);
		
		this.mainModelElement				= mainModel;
		
		this.graphicStateHandler 			= new DiagramElementViewStateHandler(this);
		this.diagramViewLoader				= new DiagramViewLoader(this);
		this.handlerManager 				= new HandlerManager(this);
		this.syntaxModelBinderHandler 		= new DiagramElementViewSyntaxModelBinderHandler(this);
		this.diagramViewsValidationHandler 	= new DiagramValidationHandler(this,diagramUI);
		this.ruleNotificationRegistry		= new RuleNotificationRegistry(this);
		
		DiagramViewListener listener = new DiagramViewListener(this);
		listener.listen(diagramUI);
		
		//initialize main model on uiPanel
		diagramUI.setMainModel(mainModel);		
	}
	
	public DiagramController(IDiagramView diagramUI,IEditorView editorView,IMainModelElement mainModel) {
		this(diagramUI,mainModel);
		this.editorView = editorView;
		this.editorModelBuilder 	= new DiagramEditorModelBuilder(diagramUI,this);
		

	}
	
	protected DiagramElementViewStateHandler getGraphicStateHandler() {
		return graphicStateHandler;
	}
	
	protected DiagramElementViewSyntaxModelBinderHandler getSyntaxModelBinderHandler() {
		return syntaxModelBinderHandler;
	}
	
	public void setNotificationView(IRuleNotificationView view){
		this.ruleNotificationView = view;
	}
	

	
	public void loadDiagram(IDiagram diagramInterchange,boolean getWithRelativeCoordinates){
	
		this.diagramViewLoader.loadDiagram(diagramGraphic,diagramInterchange,getWithRelativeCoordinates);
	}
	
	public IDiagram getDiagramInterchangeModel(boolean getWithRelativeCoordinates){
		IDiagram diagram = this.diBuilder.getDiagramInterchangeModel();
		try{
			diagram.getRootElement().setModelElement(getDiagramSyntaxModel());
		}
		catch(DiagramValidationException dve){
			diagram.getRootElement().setModelElement(null);
		}
		return diagram;
	}
	
	/**
	 * Returns actual diagram Syntax Model if validation pass 
	 * otherwise returns null
	 * @return
	 */
	public IModelElement getDiagramSyntaxModel() throws DiagramValidationException{
		
		//first validate actual diagram
		try{
			
			this.validate();	
		
		}catch (DiagramValidationException e) {
			
			throw new DiagramValidationException(e.getMessage());
			
		}finally{
			
			//finally bind the diagram syntax model
			return this.diagramGraphic.getSyntaxModelBuilder().getSyntaxModel();
			
		}

		
		
	}
	
	public void setEditorView(IEditorView editorView){
		this.editorView = editorView;
	}
	
	public IEditorView getEditorView() {
		return editorView;
	}
	
	
	public void addLoaderHandler(IDiagramViewLoaderHandler handler) {
		handlerManager.addHandler(LoadingStateChangeEvent.TYPE, handler);
		handlerManager.addHandler(LoadSuccessEvent.TYPE, handler);
		handlerManager.addHandler(ElementLoadedEvent.TYPE, handler);
		handlerManager.addHandler(LoadErrorEvent.TYPE, handler);
	}
	
	public void addValidationHandler(IValidationHandler handler) {
		handlerManager.addHandler(ValidationSuccessEvent.TYPE, handler);
		handlerManager.addHandler(ValidationFailureEvent.TYPE, handler);
		handlerManager.addHandler(ValidationWarningEvent.TYPE, handler);
		handlerManager.addHandler(ValidationStartEvent.TYPE, handler);
	}
	
	public void addRuleNotificationHandler(IRuleNotificationHandler handler) {
		handlerManager.addHandler(NotificationSelectedEvent.TYPE, handler);
		handlerManager.addHandler(RuleResolvedEvent.TYPE, handler);
		handlerManager.addHandler(ShowRulesNotificationWithLevelEvent.TYPE, handler);
		handlerManager.addHandler(HideRulesNotificationWithLevelEvent.TYPE, handler);
		handlerManager.addHandler(ShowContextualNotificationsEvent.TYPE, handler);
		handlerManager.addHandler(HideContextualNotificationsEvent.TYPE, handler);
	}
	
	public void undo(){
		
	}
	
	public void redo(){
		
	}

	public void validate() throws DiagramValidationException{
		
		this.diagramViewsValidationHandler.validate();
	
		if (this.getNotificationRegistry().getRulesInFailure().size()>0) {
			
			String msg = "";
			
			for(IConformityRule r : this.getNotificationRegistry().getRulesInFailure()){
				msg = msg + "</br> - " + r.getRuleName() + "</br>";
			}
			
			throw new DiagramValidationException(msg);
			
		}
	
	}
	
	@Override
	public void fireEvent(GwtEvent<?> event) {
		handlerManager.fireEvent(event);
	}

	@Override
	public IRuleNotificationRegistry getNotificationRegistry() {
		return ruleNotificationRegistry;
	}

	@Override
	public IRuleNotificationView getRuleNotificationView() {
		return ruleNotificationView;
	}

	@Override
	public IMainModelElement getMainModelElement() throws DiagramValidationException{
		
		getDiagramSyntaxModel();
		
		return this.mainModelElement;
	}
	
	protected IMainModelElement getMainModelElementWithoutValidation(){
		return this.mainModelElement;
	}

	@Override
	public void removeSelectedElements() {
		this.diagramGraphic.removeSelectedElements();
	}
	
	
	
}
