
package com.ebmwebsourcing.petalsview.service.dataloader;

import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.springframework.stereotype.Service;

import com.ebmwebsourcing.petalsview.service.flowref.FlowRefManager;
import com.ebmwebsourcing.petalsview.service.flowref.FlowStepRefManager;
import com.ebmwebsourcing.webcommons.util.InputStreamUtil;

import com.petalsview.dataset.Dataset;

import javax.annotation.Resource;
import javax.xml.transform.stream.StreamSource;

@Service("dataloader")
public class DataLoaderImpl implements Dataloader {
    private final Logger logger = Logger.getLogger(this.getClass());

    @Resource(name = "flowRefManager")
    private FlowRefManager flowRefManager;
    
    @Resource(name="flowStepRefManager")
	private FlowStepRefManager flowStepRefManager;

    public void loadDataSet(String xmlDataSetLocation) throws DataLoaderException {

        logger.info("Load a petals view dataset from location: " + xmlDataSetLocation);
        InputStream datasetStream;
        try {
            datasetStream = InputStreamUtil.getInputStream(xmlDataSetLocation);
        } catch (Exception e) {
            throw new DataLoaderException("Can't load Petals View dataset", e);
        }
        loadDataset(datasetStream);
    }

    public void loadDataset(InputStream datasetStream) throws DataLoaderException {
        Dataset dataset = null;
        try {
            dataset = ViewDatasetHandler.getInstance().convertStreamSource2ViewDataset(
                    new StreamSource(datasetStream));
        } catch (Exception e) {
            throw new DataLoaderException("Can't load Petals View dataset", e);
        }

        // Create flowRef
        Map<String, String> flowRefIndexToFlowRefId = new HashMap<String, String>();
        loadFlowRef(dataset, flowRefIndexToFlowRefId);

        // Create flowStepRef
        Map<String, String> flowStepRefIndexToFlowStepRefId = new HashMap<String, String>();
        loadFlowStepRef(dataset, flowStepRefIndexToFlowStepRefId, flowRefIndexToFlowRefId);

        // Create flowStepRefParameter
        loadFlowStepParamRef(dataset, flowStepRefIndexToFlowStepRefId);

        // Create flowStepRefError
        Map<String, String> flowStepErrorRefId = new HashMap<String, String>();
        loadFlowStepErrorRef(dataset, flowStepErrorRefId, flowStepRefIndexToFlowStepRefId);

    }

    private void loadFlowRef(Dataset dataset, Map<String, String> flowRefIndexToFlowRefId) throws DataLoaderException {

        List<com.petalsview.dataset.FlowRef> flowR = dataset.getFlowRef();

        for (com.petalsview.dataset.FlowRef fR : flowR) {
            try {
                String id = flowRefManager.createFlowRef(fR.getType(), fR.getName());

                flowRefIndexToFlowRefId.put(fR.getIndex(), id);
            } catch (Exception e) {
            	 throw new DataLoaderException("A Referential with that informations already exists in database", e);
            }
        }
    }

    private void loadFlowStepRef(Dataset dataset, Map<String, String> flowStepRefIndexToFlowStepRefId,
            Map<String, String> flowRefIndexToFlowRefId) throws DataLoaderException {

        List<com.petalsview.dataset.FlowStepRef> flowStepR = dataset.getFlowStepRef();

        for (com.petalsview.dataset.FlowStepRef fSR : flowStepR) {
            com.ebmwebsourcing.petalsview.persistence.model.flowref.FlowStepRef flowStepRef = new com.ebmwebsourcing.petalsview.persistence.model.flowref.FlowStepRef();
            flowStepRef.setDefaultErrorMessage(fSR.getDefaultErrorMessage());
            flowStepRef.setFlowEndStep(fSR.isFlowEndStep());
            flowStepRef.setFlowStartStep(fSR.isFlowStartStep());
            flowStepRef.setInterfaceName(fSR.getInterfaceName());
            flowStepRef.setServiceName(fSR.getServiceName());
            flowStepRef.setName(fSR.getName());
            flowStepRef.setSuccessMessage(fSR.getSucessMessage());

            String idStepRef = fSR.getFlowRefIndex();
            if (idStepRef != null) {
                com.ebmwebsourcing.petalsview.persistence.model.flowref.FlowRef flowRef = null;
                try {
                    flowRef = flowRefManager.loadFlowRef(flowRefIndexToFlowRefId.get(idStepRef));
                } catch (Exception e) {
                	 throw new DataLoaderException("A Referential with that informations already exists in database", e);
                }
                flowStepRef.setFlowref(flowRef);
            }
            try {
                String id = flowStepRefManager.saveOrUpdateStepRef(flowStepRef);
                flowStepRefIndexToFlowStepRefId.put(fSR.getIndex(), id);

            } catch (Exception e) {
            	 throw new DataLoaderException("A Referential with that informations already exists in database", e);
            }
        }
    }

    private void loadFlowStepParamRef(Dataset dataset,
            Map<String, String> flowStepRefIndexToFlowStepRefId) throws DataLoaderException {

        List<com.petalsview.dataset.FlowStepParameterRef> flowStepParamRef = dataset
                .getFlowStepParameterRef();
        
            for (com.petalsview.dataset.FlowStepParameterRef fSPR : flowStepParamRef) {
                String idStepRef = fSPR.getFlowStepRefIndex();
                if (idStepRef != null) {
                    try {
                    	flowStepRefManager.addParameterName(flowStepRefIndexToFlowStepRefId.get(idStepRef), fSPR.getIdx(), fSPR.getName(),
                                fSPR.isGlobal());
                    } catch (Exception e) {
                        e.printStackTrace();
                        throw new DataLoaderException("Can't load step with id : " + idStepRef, e);
                    }
                } else {
                    throw new DataLoaderException("Can't load step because its id is null");
                }
            }
    }

    private void loadFlowStepErrorRef(Dataset dataset, Map<String, String> flowStepErrorRefId,
            Map<String, String> flowStepRefIndexToFlowStepRefId) throws DataLoaderException {

        List<com.petalsview.dataset.FlowStepErrorRef> flowStepError = dataset
                .getFlowStepErrorRef();

        for (com.petalsview.dataset.FlowStepErrorRef fSER : flowStepError) {
            String idStepRef = fSER.getFlowStepRefIndex();
            if (idStepRef != null) {
                try {
                    com.ebmwebsourcing.petalsview.persistence.model.flowref.FlowStepRef flowStepRef = flowStepRefManager
                            .loadStepRef(flowStepRefIndexToFlowStepRefId.get(idStepRef));
                    com.ebmwebsourcing.petalsview.persistence.model.flowref.FlowStepErrorRef flowStepErrorRef = new com.ebmwebsourcing.petalsview.persistence.model.flowref.FlowStepErrorRef();
                    flowStepErrorRef.setErrorCode(fSER.getErrorCode());
                    flowStepErrorRef.setFlowstepref(flowStepRef);
                    flowStepErrorRef.setMessage(fSER.getMessage());

                    String id = flowStepRefManager.saveOrUpdateErrorRef(flowStepErrorRef);

                    flowStepErrorRefId.put(fSER.getIndex(), id);

                } catch (Exception e) {
                    e.printStackTrace();
                    throw new DataLoaderException("Can't load step with id : " + idStepRef, e);
                }
            }
        }
    }

}
