/**
 * Petals View - Functional Supervision.
 * Copyright (c) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * FlowManagerImpl.java
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.petalsview.service.flow;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.springframework.stereotype.Service;

import com.ebmwebsourcing.petalsview.persistence.dao.flow.FlowDAO;
import com.ebmwebsourcing.petalsview.persistence.dao.flow.FlowStepDAO;
import com.ebmwebsourcing.petalsview.persistence.dao.flowref.FlowRefDAO;
import com.ebmwebsourcing.petalsview.persistence.dao.flowref.FlowStepRefDAO;
import com.ebmwebsourcing.petalsview.persistence.model.flow.Flow;
import com.ebmwebsourcing.petalsview.persistence.model.flow.FlowStep;
import com.ebmwebsourcing.petalsview.persistence.model.flowref.FlowRef;
import com.ebmwebsourcing.petalsview.persistence.model.flowref.FlowStepRef;


@Service("flowManager")
public class FlowManagerImpl implements FlowManager {

    @Resource(name = "flowDAO")
    private FlowDAO flowDAO;

    @Resource(name = "flowStepDAO")
    private FlowStepDAO flowStepDAO;

    @Resource(name = "flowStepRefDAO")
    private FlowStepRefDAO flowStepRefDAO;

    @Resource(name = "flowRefDAO")
    private FlowRefDAO flowRefDAO;

    public Flow load(final String id) throws FlowException {
       
    	final Flow flow = flowDAO.loadByPetalsId(id);
        if (flow != null) {
            this.updateFlowStatusAndDates(flow);
        }
        else{
        	throw new FlowException("no flow whith this id in database ! ");
        }
        return flow;
    }

    public void remove(final String id) throws FlowException{
        
    	 Flow flow = flowDAO.loadByPetalsId(id);
    	if(flow == null){
    		throw new FlowException("no flow whith this petals id in database ! ");
    	}
    	 
    	 flowDAO.remove(flow);
    }
    
 public String saveOrUpdate(final Flow flow) throws FlowException {
        
    	flowDAO.save(flow);
        final String result = flow.getIdpetals();
        return result;
      
    }

    public List<Flow> loadAll() {
        final List<Flow> result = flowDAO.loadAll();
        return result;
    }

    public List<Flow> loadAllStarted() {
        final List<Flow> result = flowDAO.loadAllStarted();
        if (result != null) {
            for (final Flow flow : result) {
                this.updateFlowStatusAndDates(flow);
            }
        }
        return result;
    }

    public List<Flow> loadUnknowsFlows() {

        List<FlowRef> listTmp = flowRefDAO.getAll();
        List<Integer> types = new ArrayList<Integer>();

        for (FlowRef fR : listTmp) {
            types.add(fR.getType());
        }

        final List<Flow> result = flowDAO.loadUnknowsFlow(types);

        for (int i = 0; i < result.size(); i++) {
            int j = i + 1;

            while (j < result.size()) {
                if (result.get(i).getType() == result.get(j).getType()) {
                    result.remove(j);
                } else {
                    j++;
                }
            }
        }

        return result;
    }

    public List<Flow> loadByType(final int type) {
        final List<Flow> result = flowDAO.loadByType(type);
        if (result != null) {
            for (final Flow flow : result) {
                this.updateFlowStatusAndDates(flow);
            }
        }
        return result;
    }

    public List<Flow> loadAllStartedByType(final int type) {
        final List<Flow> result = flowDAO.loadAllStartedByType(type);
        if (result != null) {
            for (final Flow flow : result) {
                this.updateFlowStatusAndDates(flow);
            }
        }
        return result;
    }

    public List<Flow> loadAllStartedByTypeWithDateFilter(final int type, final Date startDate,
            final Date endDate) {
        final List<Flow> result = flowDAO.loadAllStartedByTypeWithDateFilter(type, startDate,
                endDate);
        if (result != null) {
            for (final Flow flow : result) {
                this.updateFlowStatusAndDates(flow);
            }
        }
        return result;
    }

    public List<Flow> loadAllStartedWithDateFilter(final Date startDate, final Date endDate) {
        final List<Flow> result = flowDAO.loadAllStartedWithDateFilter(startDate, endDate);
        return result;
    }

    /**
     * Flow status is calculated at runtime because it could change when the
     * corresponding flow ref is defined
     * 
     * @param idpetals
     * @return
     */
    private void updateFlowStatusAndDates(final Flow flow) {
        // Try to find a flow step ref defined as the end of the flow
        final FlowStepRef endStepRef = flowStepRefDAO.loadEndStepByFlowType(flow.getType());
        if (endStepRef != null) {
            // The flow status is equal to this step status
            // Load the dynamic flow step corresponding to this step service
            // name and interface name
            final List<FlowStep> flowSteps = flowStepDAO.loadByFlowIdAndMeProps(flow.getIdpetals(),
                    endStepRef.getInterfaceName(), endStepRef.getServiceName());
            if (flowSteps != null && flowSteps.size() == 1) {
                flow.setStatus(flowSteps.get(0).getStatus());
                // Date endDate = new Date(flowSteps.get(0).getEndDate());
                flow.setEndDate(flowSteps.get(0).getEndDate());
            }
        }
        // Try to find a flow step ref defined as the start of the flow
        final FlowStepRef startStepRef = flowStepRefDAO.loadStartStepByFlowType(flow.getType());
        if (startStepRef != null) {
            // The flow status is equal to this step status
            // Load the dynamic flow step corresponding to this step service
            // name and interface name
            final List<FlowStep> flowSteps = flowStepDAO.loadByFlowIdAndMeProps(flow.getIdpetals(),
                    startStepRef.getInterfaceName(), startStepRef.getServiceName());
            if (flowSteps != null && flowSteps.size() == 1) {
                // Date startDate = new Date(flowSteps.get(0).getStartDate());
                flow.setStartDate(flowSteps.get(0).getStartDate());
            }
        } else {
            flow.setStatus(-1);
        }
    }
}
