/**
 * Petals View - Functional Supervision.
 * Copyright (c) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * FlowRefManagerImpl.java
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.petalsview.service.flowref;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.springframework.stereotype.Service;

import com.ebmwebsourcing.petalsview.persistence.dao.flow.FlowStepDAO;
import com.ebmwebsourcing.petalsview.persistence.dao.flowref.FlowParamsDetails;
import com.ebmwebsourcing.petalsview.persistence.dao.flowref.FlowRefDAO;
import com.ebmwebsourcing.petalsview.persistence.dao.flowref.FlowStepRefDAO;
import com.ebmwebsourcing.petalsview.persistence.dao.flowref.FlowWithParams;
import com.ebmwebsourcing.petalsview.persistence.dao.flowref.GlobalFlowParam;
import com.ebmwebsourcing.petalsview.persistence.model.flow.FlowStep;
import com.ebmwebsourcing.petalsview.persistence.model.flowref.FlowRef;
import com.ebmwebsourcing.petalsview.persistence.model.flowref.FlowStepRef;

@Service("flowRefManager")
public class FlowRefManagerImpl implements FlowRefManager {

	@Resource(name = "flowRefDAO")
	private FlowRefDAO flowRefDAO;

	@Resource(name = "flowStepRefDAO")
	private FlowStepRefDAO flowStepRefDAO;

	@Resource(name = "flowStepDAO")
	private FlowStepDAO flowStepDAO;

	public FlowRef getFlowRef(final int type) {
		final FlowRef result = flowRefDAO.loadByType(type);
		return result;
	}

	public FlowRef loadFlowRef(final String id) throws FlowRefException {
		if(id == null){
			throw new FlowRefException("you must specify a valid flow referential id ");
		}
		final FlowRef result = flowRefDAO.get(id);
		if(result == null){
			throw new FlowRefException("you must specify a valid flow referential id ");
		}
		return result;
	}

	public List<String> loadAllServiceName() {
		List<FlowStep> allSteps = flowStepDAO.getAllOrderByServiceName();

		List<String> allServiceName = new ArrayList<String>();
		if (allSteps != null && !allSteps.isEmpty()) {
			for (FlowStep fS : allSteps) {
				String name = fS.getServiceName();

				allServiceName.add(name);
			}
		}
		allServiceName = sortList(allServiceName);

		return allServiceName;
	}

	public List<String> loadAllInterfaceName() {
		List<FlowStep> allSteps = flowStepDAO.getAll();
		List<String> allInterfaceName = new ArrayList<String>();

		if (allSteps != null && !allSteps.isEmpty()) {
			for (FlowStep fS : allSteps) {
				String name = fS.getInterfaceName();

				allInterfaceName.add(name);
			}
		}
		allInterfaceName = sortList(allInterfaceName);
		return allInterfaceName;
	}

	public List<String> loadAllEndpointName() {
		List<FlowStep> allSteps = flowStepDAO.getAll();
		List<String> allEndpointName = new ArrayList<String>();

		if (allSteps != null && !allSteps.isEmpty()) {
			for (FlowStep fS : allSteps) {
				String name = fS.getEndpointName();

				allEndpointName.add(name);
			}
		}
		return allEndpointName;
	}

	public List<FlowRef> loadAll() {
		final List<FlowRef> result = flowRefDAO.getAll();
		return result;
	}

	public String createFlowRef(final int type, final String name)
			throws FlowRefException {
		final FlowRef flowRef = new FlowRef();
		flowRef.setName(name);
		flowRef.setType(type);
		flowRef.setCreationDate(new Date());
		
		FlowRef tmp = flowRefDAO.loadByType(flowRef.getType());
		
		verification(flowRef, tmp);

		flowRefDAO.save(flowRef);
		final String id = flowRef.getId();
		return id;
	}

	public void update(final FlowRef flowRef) throws FlowRefException {
		
		FlowRef tmp = flowRefDAO.loadByType(flowRef.getType());
		
		verification(flowRef, tmp);
		
		tmp.setName(flowRef.getName());
		
		flowRefDAO.save(tmp);
	}

	public void removeFlowRef(final int[] flowRefTypes) throws FlowRefException{
		if(flowRefTypes == null){
			throw new FlowRefException("you must specify a valid flow referential id ");
		}
		
		for (final int type : flowRefTypes) {
			
			final List<FlowStepRef> steps = flowStepRefDAO.loadByFlowType(type);
			if (steps != null) {
				for (final FlowStepRef step : steps) {
					flowStepRefDAO.remove(step);
				}
			}
			flowRefDAO.remove(flowRefDAO.loadByType(type));
		}
	}

	private List<String> sortList(List<String> list) {
		for (int i = 0; i < list.size(); i++) {
			int j = i + 1;

			while (j < list.size()) {
				if (list.get(i).equals(list.get(j))) {
					list.remove(j);
				} else {
					j++;
				}
			}
		}
		return list;
	}
	
	private void verification (FlowRef flowRef, FlowRef tmp) throws FlowRefException{
		
		
		
		if(flowRef.getName() == null){
			throw new FlowRefException("You must specify a flow referential name ");
		}
		
		if(flowRef.getId() != null){
			
			if(tmp !=null && !flowRef.getId().equals(tmp.getId())) 
			{
				throw new FlowRefException("A flow referential with that type already exists : "
						+ flowRef.getType());
			}
			
			FlowRef fr = new FlowRef();
			fr = flowRefDAO.loadByName(flowRef.getName());
			if(fr != null && !fr.getId().equals(flowRef.getId())){
				throw new FlowRefException("A flow referential with that name already exists : "
						+ flowRef.getName());
			}
		}
		else {
			
			if(tmp !=null) 
			{
				throw new FlowRefException("A flow referential with that type already exists : "
						+ flowRef.getType());
			}
			
			tmp = new FlowRef();
			tmp = flowRefDAO.loadByName(flowRef.getName());
			if(tmp != null){
				throw new FlowRefException("A flow referential with that name already exists : "
						+ flowRef.getName());
			}
		}
		
		
		
	}

	// =====================================================================
	// === Global Flow Parameters Management
	// =====================================================================

	public List<GlobalFlowParam> getGlobalFlowParameters(final short flowType) {
		final List<GlobalFlowParam> result = flowStepRefDAO
				.getGlobalFlowParameters(flowType);

		return result;
	}

	public List<FlowWithParams> getFlowsWithParameters(
			final List<GlobalFlowParam> globalParams, final short type,
			final Date after, final Date before) {

		final List<FlowWithParams> result = flowStepRefDAO
				.getFlowsWithParameters(globalParams, type, after, before);

		return result;
	}

	public List<FlowParamsDetails> getDetailFlowParameter(final short flowType,
			final String flowId) throws FlowRefException {
		
		if(flowId == null) {
			throw new FlowRefException("You must specify a flow : ");
		
		}
		
		final List<FlowParamsDetails> result = flowStepRefDAO
				.getDetailFlowParameter(flowType, flowId);

		return result;
	}
}
