/**
 * su-generator - JBI Su Management - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.ebmwebsourcing.petals.services.generation;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * The common elements related to the generation of JBI descriptors for Petals components.
 * <p>
 * The version and the component name should be defined by any sub-class
 * constructor.
 * </p>
 * 
 * @author Vincent Zurczak - EBM WebSourcing
 */
public class AbstractJbiXml {

	protected boolean generateComments = true;
	protected String version = "";
	protected String componentName;
	protected String encoding = "UTF-8";



	/**
	 * Create a JBI descriptor for a Petals component.
	 * @param beans a list of {@link JbiBeanDelegate} (not null and not empty).
	 * @return the jbi.xml content
	 */
	public StringBuilder createJbiDescriptor( List<JbiBeanDelegate> beans ) {

		if( beans == null )
			throw new IllegalArgumentException( "The list of beans cannot be empty." );

		StringBuilder sb = new StringBuilder();
		sb.append( "<?xml version=\"1.0\" encoding=\"" + this.encoding + "\"?>\n" );
		if( this.generateComments ) {
			sb.append( "<!-- JBI descriptor for the Petals component " );
			sb.append( this.componentName + " " + this.version.toString() + " -->\n" );
		}

		Map<String,String> namespaces = new TreeMap<String,String>();
		namespaces.put( "jbi", "http://java.sun.com/xml/ns/jbi" );
		namespaces.put( "xsi", "http://www.w3.org/2001/XMLSchema-instance" );
		for( JbiBeanDelegate bean : beans )
			namespaces.putAll( bean.getNamespaces());

		sb.append( "<jbi:jbi version=\"1.0\" \n" );
		for( Iterator<Map.Entry<String,String>> it = namespaces.entrySet().iterator(); it.hasNext(); ) {
			Map.Entry<String,String> entry = it.next();
			if( it.hasNext())
				sb.append( "\txmlns:" + entry.getKey() + "=\"" + entry.getValue() + "\"\n" );
			else
				sb.append( "\txmlns:" + entry.getKey() + "=\"" + entry.getValue() + "\">\n\n" );
		}

		sb.append( "\t<jbi:services binding-component=\"false\">\n" );
		for( JbiBeanDelegate bean : beans ) {
			if( bean.isProvides())
				createProvideSection( sb, bean );
			else
				createConsumeSection( sb, bean );
		}

		sb.append( "\t</jbi:services>\n" );
		sb.append( "</jbi:jbi>" );
		return sb;
	}


	/**
	 * Creates a 'provides' section in the XML document.
	 * @param sb
	 * @param bean
	 */
	protected void createProvideSection( StringBuilder sb, JbiBeanDelegate bean ) {

		boolean sameNs =
			( bean.interfaceNsUri == null && bean.serviceNsUri == null )
			|| ( bean.interfaceNsUri != null && bean.interfaceNsUri.equals( bean.serviceNsUri ));

		sb.append( "\t\t<jbi:provides\n" );
		sb.append( "\t\t\tinterface-name=\"");
		sb.append(( sameNs ? "generatedNs" : "interfaceNs" ) + ":" + bean.interfaceName + "\"\n" );
		sb.append( "\t\t\tservice-name=\"");
		sb.append(( sameNs ? "generatedNs" : "serviceNs" ) + ":" + bean.serviceName + "\"\n" );
		sb.append( "\t\t\tendpoint-name=\"" + bean.endpointName + "\"\n" );

		if( sameNs )
			sb.append( "\t\t\txmlns:generatedNs=\"" + bean.interfaceNsUri + "\">\n\n" );
		else {
			sb.append( "\t\t\txmlns:interfaceNs=\"" + bean.interfaceNsUri + "\"\n" );
			sb.append( "\t\t\txmlns:serviceNs=\"" + bean.serviceNsUri + "\">\n\n" );
		}

		String cdk = bean.getCdkSection();
		if( this.generateComments ) {
			if( cdk != null && cdk.trim().length() > 0 )
				sb.append( "\t\t\t<!-- CDK elements -->\n" );
			else
				sb.append( "\t\t\t<!-- No CDK element -->\n" );
		}

		sb.append( cdk + "\n" );
		String component = bean.getComponentSection();
		if( this.generateComments ) {
			if( component != null && component.trim().length() > 0 )
				sb.append( "\t\t\t<!-- Component specific elements -->\n" );
			else
				sb.append( "\t\t\t<!-- No component specific element -->\n" );
		}

		sb.append( component + "\n" );
		sb.append( "\t\t</jbi:provides>\n" );
	}


	/**
	 * Creates a 'consumes' section in the XML document.
	 * @param sb
	 * @param bean
	 */
	protected void createConsumeSection( StringBuilder sb, JbiBeanDelegate bean ) {

		boolean sameNs =
			( bean.interfaceNsUri == null && bean.serviceNsUri == null )
			|| ( bean.interfaceNsUri != null && bean.interfaceNsUri.equals( bean.serviceNsUri ));

		sb.append( "\t\t<jbi:consumes\n" );
		sb.append( "\t\t\tinterface-name=\"");
		sb.append(( sameNs ? "generatedNs" : "interfaceNs" ) + ":" + bean.interfaceName + "\"\n" );

		if( bean.getServiceName() != null
				&& bean.getServiceName().trim().length() > 0
				&& bean.getEndpointName() != null
				&& bean.getEndpointName().trim().length() > 0 ) {

			sb.append( "\t\t\tservice-name=\"");
			sb.append(( sameNs ? "generatedNs" : "serviceNs" ) + ":" + bean.serviceName + "\"\n" );
			sb.append( "\t\t\tendpoint-name=\"" + bean.endpointName + "\"\n" );
		}

		if( sameNs )
			sb.append( "\t\t\txmlns:generatedNs=\"" + bean.interfaceNsUri + "\">\n\n" );
		else {
			sb.append( "\t\t\txmlns:interfaceNs=\"" + bean.interfaceNsUri + "\"\n" );
			sb.append( "\t\t\txmlns:serviceNs=\"" + bean.serviceNsUri + "\">\n\n" );
		}

		if( this.generateComments )
			sb.append( "\t\t\t<!-- CDK elements -->\n" );

		sb.append( bean.getCdkSection() + "\n" );
		if( this.generateComments )
			sb.append( "\t\t\t<!-- Component specific elements -->\n" );

		sb.append( bean.getComponentSection() + "\n" );
		sb.append( "\t\t</jbi:consumes>\n" );
	}


	/**
	 * @return the generateComments (true by default)
	 */
	public boolean isGenerateComments() {
		return this.generateComments;
	}


	/**
	 * @param generateComments the generateComments to set (true by default)
	 */
	public void setGenerateComments( boolean generateComments ) {
		this.generateComments = generateComments;
	}


	/**
	 * @return the version
	 */
	public String getVersion() {
		return this.version;
	}


	/**
	 * @return the encoding (UTF-8 by default)
	 */
	public String getEncoding() {
		return this.encoding;
	}


	/**
	 * @param encoding the encoding to set (UTF-8 by default)
	 */
	public void setEncoding( String encoding ) {
		this.encoding = encoding;
	}


	/**
	 * @return the componentName
	 */
	public String getComponentName() {
		return this.componentName;
	}


	/**
	 * @param componentName the componentName to set
	 */
	public void setComponentName( String componentName ) {
		this.componentName = componentName;
	}


	/**
	 * @param version the version to set
	 */
	public void setVersion( String version ) {
		this.version = version;
	}
}
