/**
 * bpmn-deployer - Generic web application for BPMN deployment - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.ebmwebsourcing.bpmn.deployer.server;

import java.io.File;
import java.net.MalformedURLException;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.ebmwebsourcing.bpmn.deployer.client.exception.BPMNDeploymentException;
import com.ebmwebsourcing.bpmn.deployer.client.exception.ProcessExecutorServiceException;
import com.ebmwebsourcing.bpmn.deployer.client.exception.WrongURLException;
import com.ebmwebsourcing.bpmn.deployer.client.service.BPMNDeploymentService;
import com.ebmwebsourcing.bpmn.deployer.client.to.ProcessExecutorDescription;
import com.ebmwebsourcing.bpmn.deployer.client.to.RunningProcess;
import com.ebmwebsourcing.easybox.api.XmlContextFactory;
import com.ebmwebsourcing.easybox.api.XmlObjectReadException;
import com.ebmwebsourcing.easybpel.model.bpel.api.BPELProcess;
import com.ebmwebsourcing.easybpmn.bpmn20.api.BPMNException;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Definitions;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Process;
import com.ebmwebsourcing.easybpmn.bpmn2bpel.BPELGenerator;
import com.ebmwebsourcing.easybpmn.bpmn2bpel.BPELGeneratorHelper;
import com.ebmwebsourcing.petalsbpm.utils.server.FileHelper;
import com.ebmwebsourcing.petalsbpm.utils.server.ZipHelper;
import com.google.gwt.user.server.rpc.RemoteServiceServlet;

public class BPMNDeploymentServiceImpl extends RemoteServiceServlet implements BPMNDeploymentService {

    private Map<String, ProcessExecutorService> executors = Collections.synchronizedMap(new HashMap<String, ProcessExecutorService>());


    @Override
    public void deployBPMNProcesses(List<String> processIds, String bpmnFileAbsolutePath,
            ProcessExecutorDescription desc) throws BPMNDeploymentException {
        if(bpmnFileAbsolutePath==null || bpmnFileAbsolutePath.isEmpty()) {
            throw new BPMNDeploymentException("The provided BPMN file does not exists : "+bpmnFileAbsolutePath);
        }
        
        File bpmnFile = new File(bpmnFileAbsolutePath);
        
        try {
            if(ZipHelper.getInstance().isZipFile(bpmnFile)) {
                File zipDir = FileHelper.createTemporaryDirectory();
                ZipHelper.getInstance().unzipFile(bpmnFile, zipDir);
                for(File f : zipDir.listFiles()){
                    if((f.getName().startsWith("BPMN_") && f.getName().endsWith(".xml"))
                            || f.getName().endsWith(".bpmn")){
                        bpmnFile = f;
                        break;
                    }
                }
            }

            Definitions bpmnDefs = new XmlContextFactory().newContext().createReader().readDocument(bpmnFile.toURI().toURL(), Definitions.class);

            Set<String> processNames = new HashSet<String>();
            for(String processId : processIds) {
                for(Process process : bpmnDefs.getProcesses()) {
                    if(process.getId().equals(processId)) {
                        processNames.add(BPELGeneratorHelper.getBPELProcessName(process));
                        break;
                    }
                }
            }

            File tmpDir = FileHelper.createTemporaryDirectory();

            for(BPELProcess bpelProcess : new BPELGenerator().generate(bpmnDefs, tmpDir.getAbsolutePath())) {
                if(processNames.contains(bpelProcess.getName())) {
                    File bpelFile = new File(tmpDir, bpelProcess.getName()+".bpel");
                    assert bpelFile.exists();
                    getExecutorService(desc).deployProcess(bpelFile);
                }
            }
        } catch (MalformedURLException e) {
            e.printStackTrace();
            throw new BPMNDeploymentException("The provided BPMN file does not exist : "+bpmnFileAbsolutePath);
        } catch (XmlObjectReadException e) {
            e.printStackTrace();
            throw new BPMNDeploymentException("The provided BPMN file could not be read : "+e.getMessage());
        } catch (BPMNException e) {
            e.printStackTrace();
            throw new BPMNDeploymentException("The provided BPMN file could not be converted to BPEL : "+e.getMessage());
        } catch (ProcessExecutorServiceException e) {
            e.printStackTrace();
            throw new BPMNDeploymentException("The provided BPMN file could not deployed : "+e.getMessage());
        } catch (Throwable t) {
            t.printStackTrace();
            throw new BPMNDeploymentException("An unexpected exception occured : "+t.getMessage());
        }
    }


    @Override
    public List<RunningProcess> getDeployedProcesses(ProcessExecutorDescription desc) throws BPMNDeploymentException {
        try {
            return getExecutorService(desc).getRunningProcesses();
        } catch (ProcessExecutorServiceException e) {
            e.printStackTrace();
            throw new BPMNDeploymentException(e.getMessage());
        } catch (Throwable t) {
            t.printStackTrace();
            throw new BPMNDeploymentException("An unexpected exception occured : "+t.getMessage());
        }
    }


    private synchronized ProcessExecutorService getExecutorService(ProcessExecutorDescription desc) throws BPMNDeploymentException {
        String sessionID = this.getThreadLocalRequest().getSession().getId();
        ProcessExecutorService executor;
        
        //if an executor exists for this session
        if(executors.containsKey(sessionID)) {
            executor = executors.get(sessionID);
            //if this executor has the same host as the requested one
            if(executor.getHost()!=null && executor.getHost().equals(desc.getHost())) {
                //and if this executor targets the same kind of server
                if(executor.getClass().getName().equals(desc.getProcessExecutorService())) {
                    return executor;
                }
            }
        }

        try {
            executor = ProcessExecutorServiceFactory.getExecutorService(desc.getProcessExecutorService());
        } catch (Exception e) {
            e.printStackTrace();
            throw new BPMNDeploymentException(e.getMessage());
        }

        try {
            executor.testServiceHost(desc.getHost());
        } catch (ProcessExecutorServiceException e) {
            e.printStackTrace();
            throw new WrongURLException(desc.getHost(), e.getMessage());
        }

        executors.put(sessionID, executor);
        return executor;
    }

}
