/**
 * bpmn-diagram - SVG/VML web based editor for BPMN Standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.petalsbpm.bpmndiagram.ui.activities;

import java.util.HashSet;
import java.util.LinkedHashSet;

import com.ebmwebsourcing.geasytools.diagrameditor.api.graphic.IDiagramElementGraphicState;
import com.ebmwebsourcing.geasytools.diagrameditor.api.modeleditor.IEditorModel;
import com.ebmwebsourcing.geasytools.diagrameditor.api.validation.IDiagramElementViewConformityRule;
import com.ebmwebsourcing.geasytools.geasysvg.core.impl.Path;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.IConnectableElement;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.IConnectableElementDefaulHandlers;
import com.ebmwebsourcing.geasytools.geasyui.api.contextualmenu.IHasDragProxyContextualMenu;
import com.ebmwebsourcing.geasytools.geasyui.api.contextualmenu.IHasMenuDragProxy;
import com.ebmwebsourcing.geasytools.geasyui.api.contextualmenu.events.IDragProxyContextualMenuHandler;
import com.ebmwebsourcing.geasytools.geasyui.api.draggable.IDraggableElement;
import com.ebmwebsourcing.geasytools.geasyui.api.droppable.events.IDropAcceptedEvent;
import com.ebmwebsourcing.geasytools.geasyui.impl.connectable.ConnectableElementDefaultHandlers;
import com.ebmwebsourcing.geasytools.geasyui.impl.contextualmenu.DragProxyContextualMenu;
import com.ebmwebsourcing.geasytools.geasyui.impl.draggable.DraggableProxy;
import com.ebmwebsourcing.geasytools.geasyui.impl.draggable.events.ProxyAcceptedAfterDropEvent;
import com.ebmwebsourcing.geasytools.geasyui.impl.draggable.events.ProxyAcceptedBeforeDropEvent;
import com.ebmwebsourcing.geasytools.geasyui.impl.draggable.events.ProxyDragMoveEvent;
import com.ebmwebsourcing.geasytools.geasyui.impl.draggable.events.ProxyDragStartEvent;
import com.ebmwebsourcing.geasytools.geasyui.impl.draggable.events.ProxyDragStopEvent;
import com.ebmwebsourcing.geasytools.geasyui.impl.draggable.events.ProxyRefusedAfterDropEvent;
import com.ebmwebsourcing.geasytools.geasyui.impl.draggable.events.ProxyRefusedBeforeDropEvent;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.BPMNElementsPath;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.dragproxy.impl.activities.TaskMenuDragProxy;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.dragproxy.impl.artifact.TextAnnotationMenuDragProxy;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.dragproxy.impl.connectors.SequenceFlowMenuDragProxy;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.dragproxy.impl.data.DataObjectMenuDragProxy;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.dragproxy.impl.events.EndNoneMenuDragProxy;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.dragproxy.impl.events.IntermediateCatchingMessageMenuDragProxy;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.dragproxy.impl.events.IntermediateThrowingMessageMenuDragProxy;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.dragproxy.impl.gateways.ExclusiveGatewayMenuDragProxy;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.ui.common.ConnectableFlowElement;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.ui.common.ContextualMenuHandler;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.ui.common.connector.ConnectionMode;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.ui.connectors.DataAssociation;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.ui.connectors.MessageFlow;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.ui.connectors.SequenceFlow;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.ui.diagram.ProcessPanel;

public abstract class Activity extends ConnectableFlowElement implements IHasDragProxyContextualMenu{
	
	private boolean hasSubProcessMarker;
	private boolean hasLoopMarker;
	private boolean hasParallelMIMarker;
	private boolean hasSequentialMIMarker;
	private boolean hasAdHocMarker;
	private boolean hasCompensationMarker;
	
	private Path subProcessMarker;
	private Path loopMarker;
	private Path parallelMIMarker;
	private Path sequentialMIMarker;
	private Path adHocMarker;
	private Path compensationMarker;
	
	protected final int DEFAULT_WIDTH = 100;
	protected final int DEFAULT_HEIGHT = 50;
	
	private ActivityStates activityStates;
	
	private IConnectableElementDefaulHandlers defaultHandlers;
	
	public Activity(ProcessPanel definitionPanel,String id) {
		super(definitionPanel,id,ConnectionMode.ALL);
		//Add contextual menu ability
		DragProxyContextualMenu dpm = new DragProxyContextualMenu(this);
		
		this.addContextualMenuHandler(new ContextualMenuHandler(this));
		
		
		//Attach default handlers on load as 
		this.defaultHandlers	= new ConnectableElementDefaultHandlers(this);
		this.defaultHandlers.attachDefaultHandlers();
		
		this.activityStates = new ActivityStates();
		
		this.addDropHandler(this);
	}
	
	public IConnectableElementDefaulHandlers getDefaultHandlers(){
		return defaultHandlers;
	}
	
	@Override
	protected void init() {
		super.init();
		
		subProcessMarker 			= this.getDefinitionPanel().getCanvas().createPath(BPMNElementsPath.ACTIVITY_MARKER_SUBPROCESS_EXTENDED, 0, 0);
		loopMarker		 			= this.getDefinitionPanel().getCanvas().createPath(BPMNElementsPath.ACTIVITY_MARKER_LOOP, 0, 0); 
		parallelMIMarker		 	= this.getDefinitionPanel().getCanvas().createPath(BPMNElementsPath.ACTIVITY_MARKER_PARALLEL, 0, 0); 
		sequentialMIMarker		 	= this.getDefinitionPanel().getCanvas().createPath(BPMNElementsPath.ACTIVITY_MARKER_SEQUENCIAL, 0, 0); 
		adHocMarker		 			= this.getDefinitionPanel().getCanvas().createPath(BPMNElementsPath.ACTIVITY_MARKER_AD_HOC, 0, 0); 
		compensationMarker		 	= this.getDefinitionPanel().getCanvas().createPath(BPMNElementsPath.ACTIVITY_MARKER_COMPENSATION, 0, 0); 
	
		getGroup().appendChild(subProcessMarker);
		getGroup().appendChild(loopMarker);
		getGroup().appendChild(parallelMIMarker);
		getGroup().appendChild(sequentialMIMarker);
		getGroup().appendChild(adHocMarker);
		getGroup().appendChild(compensationMarker);
		
		subProcessMarker.setVisible(false);
		subProcessMarker.setStokeColour("#A4376B");
		subProcessMarker.setFillColour("#A4376B");
		
		loopMarker.setVisible(false);
		loopMarker.setStokeColour("#A4376B");
		loopMarker.setFillColour("#A4376B");
		
		
		parallelMIMarker.setVisible(false);
		parallelMIMarker.setStokeColour("#A4376B");
		parallelMIMarker.setFillColour("#A4376B");
		
		sequentialMIMarker.setVisible(false);
		sequentialMIMarker.setStokeColour("#A4376B");
		sequentialMIMarker.setFillColour("#A4376B");
		
		adHocMarker.setVisible(false);
		adHocMarker.setStokeColour("#A4376B");
		adHocMarker.setFillColour("#A4376B");
		
		compensationMarker.setVisible(false);
		compensationMarker.setStokeColour("#A4376B");
		compensationMarker.setFillColour("#A4376B");
		
	
	}
	
	public void addContextualMenuHandler(IDragProxyContextualMenuHandler handler) {
		handlerManager.addHandler(ProxyDragStartEvent.TYPE, handler);
		handlerManager.addHandler(ProxyDragMoveEvent.TYPE, handler);
		handlerManager.addHandler(ProxyDragStopEvent.TYPE, handler);
		handlerManager.addHandler(ProxyAcceptedBeforeDropEvent.TYPE, handler);
		handlerManager.addHandler(ProxyRefusedBeforeDropEvent.TYPE, handler);
		handlerManager.addHandler(ProxyAcceptedAfterDropEvent.TYPE, handler);
		handlerManager.addHandler(ProxyRefusedAfterDropEvent.TYPE, handler);	
	}
	
	
	public void hasSubProcessMarker(boolean b){
		this.hasSubProcessMarker = b;
		processMarkers();
	}
	
	public void hasLoopMarker(boolean b){
		this.hasLoopMarker = b;
		processMarkers();
	}

	public void hasParallelMIMarker(boolean b){
		this.hasParallelMIMarker = b;
		processMarkers();
	}

	public void hasSequentialMIMarker(boolean b){
		this.hasSequentialMIMarker = b;
		processMarkers();
	}
	
	public void hasAdHocMarker(boolean b){
		this.hasAdHocMarker = b;
		processMarkers();
	}
	
	public void hasCompensationMarker(boolean b){
		this.hasCompensationMarker = b;
		processMarkers();
	}
	
	public boolean hasSubProcessMarker(){
		return this.hasSubProcessMarker;
	}
	
	public boolean hasLoopMarker(){
		return this.hasLoopMarker;
	}
	
	public boolean hasParallelMIMarker(){
		return this.hasParallelMIMarker;
	}
	
	public boolean hasSequentialMIMarker(){
		return this.hasSequentialMIMarker;
	}
	
	public boolean hasAdHocMarker(){
		return this.hasAdHocMarker;
	}
	
	public boolean hasCompensationMarker(){
		return this.hasCompensationMarker;
	}
	
	
	private void processMarkers(){
		
		float activityMiddle = this.getWidth()/2;
		
		float totalMarkerWidth = 0;
		float markerWidth = 20;
		float markerHeight = 20;
		
		if (hasSubProcessMarker){
			totalMarkerWidth = totalMarkerWidth + markerWidth;
		}
		if (hasLoopMarker){
			totalMarkerWidth = totalMarkerWidth + markerWidth;
		}
		if (hasParallelMIMarker){
			totalMarkerWidth = totalMarkerWidth + markerWidth;
		}
		if (hasSequentialMIMarker){
			totalMarkerWidth = totalMarkerWidth + markerWidth;
		}
		if (hasAdHocMarker){
			totalMarkerWidth = totalMarkerWidth + markerWidth; 
		}
		if (hasCompensationMarker){
			totalMarkerWidth = totalMarkerWidth + markerWidth;
		}
		
		
		float totalMarkerWidthMiddle = totalMarkerWidth/2;
		
		float startDisplayX = activityMiddle - totalMarkerWidthMiddle;
		float startDisplayY = this.getHeight() - markerHeight ;
		
		float actualX = startDisplayX;
		if (hasSubProcessMarker){
			subProcessMarker.setX(actualX);
			subProcessMarker.setY(startDisplayY);
			actualX = actualX + markerWidth;
			subProcessMarker.setVisible(true);
		}else{
			subProcessMarker.setVisible(false);
		}
		
		if (hasLoopMarker){
			loopMarker.setX(actualX);
			loopMarker.setY(startDisplayY);
			actualX = actualX + markerWidth;
			loopMarker.setVisible(true);
		}else{
			loopMarker.setVisible(false);
		}
		
		if (hasParallelMIMarker){
			parallelMIMarker.setX(actualX);
			parallelMIMarker.setY(startDisplayY);
			
			actualX = actualX + markerWidth;
			parallelMIMarker.setVisible(true);
		}else{
			parallelMIMarker.setVisible(false);
		}
		
		if (hasSequentialMIMarker){
			sequentialMIMarker.setX(actualX);
			sequentialMIMarker.setY(startDisplayY);
			actualX = actualX + markerWidth;
			sequentialMIMarker.setVisible(true);
		}else{
			sequentialMIMarker.setVisible(false);
		}
		
		if (hasAdHocMarker){
			adHocMarker.setX(actualX);
			adHocMarker.setY(startDisplayY);
			actualX = actualX + markerWidth;
			adHocMarker.setVisible(true);
		}else{
			adHocMarker.setVisible(false);
		}
		
		if (hasCompensationMarker){
			compensationMarker.setX(actualX);
			compensationMarker.setY(startDisplayY);
			actualX = actualX + markerWidth;
			compensationMarker.setVisible(true);
		}else{
			compensationMarker.setVisible(false);
		}
		
	}
	
	protected abstract IEditorModel createEditorModel();
	
	public HashSet<Class<? extends IDraggableElement>> getAcceptedTypes() {
		 HashSet<Class<? extends IDraggableElement>> result = new HashSet<Class<? extends IDraggableElement>>();
		
		 result.add(SequenceFlow.class);
		 result.add(DataAssociation.class);
		 result.add(MessageFlow.class);
		 
		 return result;
	}
	
	public LinkedHashSet<IHasMenuDragProxy> getDragProxies() {
		
		LinkedHashSet<IHasMenuDragProxy> dps = new LinkedHashSet<IHasMenuDragProxy>();
		
		TaskMenuDragProxy tdp = new TaskMenuDragProxy(this.getUIPanel());
		tdp.setLabelVisible(false);
		tdp.setSmallIcon(true);
		

		
		//GATEWAY
		ExclusiveGatewayMenuDragProxy g1 = new ExclusiveGatewayMenuDragProxy(getUIPanel());
		g1.setLabelVisible(false);
		g1.setSmallIcon(true);
		

		

		//INTERMEDIATE THROWING
		IntermediateThrowingMessageMenuDragProxy it1 = new IntermediateThrowingMessageMenuDragProxy(getUIPanel());
		it1.setLabelVisible(false);
		it1.setSmallIcon(true);
		
		//INTERMEDIATE CATCHING
		IntermediateCatchingMessageMenuDragProxy ic1 = new IntermediateCatchingMessageMenuDragProxy(getUIPanel());
		ic1.setLabelVisible(false);
		ic1.setSmallIcon(true);		
		

		
		//END EVENT
		EndNoneMenuDragProxy endNoneDragProxy = new EndNoneMenuDragProxy(getUIPanel());
		endNoneDragProxy.setLabelVisible(false);
		endNoneDragProxy.setSmallIcon(true);
			
			
		//CONNECTORS
		SequenceFlowMenuDragProxy sdp = new SequenceFlowMenuDragProxy(getUIPanel());
		sdp.setLabelVisible(false);
		sdp.setSmallIcon(true);

		
		//DATA
		DataObjectMenuDragProxy dodp = new DataObjectMenuDragProxy(getUIPanel());
		dodp.setLabelVisible(false);
		dodp.setSmallIcon(true);
		
		
		//Artifacts
		TextAnnotationMenuDragProxy tamdp = new TextAnnotationMenuDragProxy(getUIPanel());
		tamdp.setLabelVisible(false);
		tamdp.setSmallIcon(true);
		
		dps.add(tdp);
		dps.add(g1);
		dps.add(it1);
		dps.add(ic1);
		dps.add(endNoneDragProxy);
		dps.add(sdp);
		dps.add(dodp);
		dps.add(tamdp);
		
		return dps;
	}

	
	@Override
	public HashSet<IDiagramElementViewConformityRule> getConformityRules() {
		
		HashSet<IDiagramElementViewConformityRule> rules = new HashSet<IDiagramElementViewConformityRule>();
		
		DescriptiveProcessActivityRules descRules = new DescriptiveProcessActivityRules(this);
		rules.addAll(descRules.getRules());
		
		if (getDefinitionPanel().isExecutableProcess()){
			ExecutableProcessActivityRules execRules = new ExecutableProcessActivityRules(this);	
			rules.addAll(execRules.getRules());
		}
		
		return rules;
	}
	
	
	public void onDropAccepted(IDropAcceptedEvent event) {

		//if a sequence flow is dropped from a HasMenuDragProxy connect  
		if (event.getDraggableElement() instanceof DraggableProxy){
			
			DraggableProxy dp = (DraggableProxy) event.getDraggableElement();
			
			if (dp.getHasDraggableElementProxy() instanceof IHasMenuDragProxy){
				
				IHasMenuDragProxy hmdp = (IHasMenuDragProxy) (dp.getHasDraggableElementProxy());
				
				if (dp.getHasDraggableElementProxy().getIUIElementType()==SequenceFlow.class){
					
					SequenceFlow sf = (SequenceFlow) getUIPanel().getElementFactory().getElement(SequenceFlow.class);
					getUIPanel().addUIElement(sf);
					
					sf.connect((IConnectableElement) hmdp.getContextualMenuSubject(), this);
					
				}else if (dp.getHasDraggableElementProxy().getIUIElementType()==MessageFlow.class){
					
					MessageFlow mf = (MessageFlow) getUIPanel().getElementFactory().getElement(MessageFlow.class);
					getUIPanel().addUIElement(mf);
					
					mf.connect((IConnectableElement) hmdp.getContextualMenuSubject(), this);
					
				}else if (dp.getHasDraggableElementProxy().getIUIElementType()==DataAssociation.class){
					
					DataAssociation da = (DataAssociation) getUIPanel().getElementFactory().getElement(DataAssociation.class);
					getUIPanel().addUIElement(da);
					
					da.connect((IConnectableElement) hmdp.getContextualMenuSubject(), this);
				}
				
			}
			
			
		}
		
		
	}
	
	@Override
	public void setWidth(float width) {
		if (width>DEFAULT_WIDTH){
			super.setWidth(width);
		}else{
			super.setWidth(DEFAULT_WIDTH);
		}

	}
	
	@Override
	public void setHeight(float height) {
		if (height>DEFAULT_HEIGHT){
			super.setHeight(height);
		}else{
			super.setHeight(DEFAULT_HEIGHT);			
		}
	}
	
	@Override
	public HashSet<IDiagramElementGraphicState> getStates() {
		
		HashSet<IDiagramElementGraphicState> states = new HashSet<IDiagramElementGraphicState>();
		
		states.addAll(super.getStates());
		states.addAll(this.activityStates.getStates());
		
		return states;
	}
	
}
