/**
 * bpmn-diagram - SVG/VML web based editor for BPMN Standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.petalsbpm.bpmndiagram.ui.connectors;

import java.util.HashSet;

import com.ebmwebsourcing.geasytools.diagrameditor.api.graphic.IDiagramElementView;
import com.ebmwebsourcing.geasytools.diagrameditor.api.validation.IDiagramElementViewConformityRule;
import com.ebmwebsourcing.geasytools.diagrameditor.api.validation.IRuleLevel;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IDiagramElement;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.validation.AbstractDiagramElementViewConformityRule;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.validation.RuleLevel;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.IConnector;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.editor.model.connector.SequenceFlowEditorModel;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.ui.common.connector.ConnectorElement;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.IInteractionNodeBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IFlowNodeBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.collaboration.LaneBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.activity.ActivityBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.gateway.ExclusiveGatewayBean;

public class DescriptiveProcessConnectorRules {
	
	private HashSet<IDiagramElementViewConformityRule> rules;
	
	public DescriptiveProcessConnectorRules(ConnectorElement connector) {
		
		this.rules = new HashSet<IDiagramElementViewConformityRule>();
		
		this.rules.add(new ConnectorMustHaveSourceAndTarget(connector));
		this.rules.add(new ConnectorCannotHaveSameSourceAndTarget(connector));
		
		if (connector instanceof SequenceFlow){
			
			this.rules.add(new SequenceFlowSourceAndTargetMustBeAFlowNodeFromSame(connector));
			this.rules.add(new SequenceFlowCanHaveConditionOnlyWhenSourceIsActivityOrExclusiveGateway(connector));
			
		}else if (connector instanceof MessageFlow){
			
			this.rules.add(new MessageFlowSourceAndTargetMustBeAnInteractionNodeFromDifferentProcess(connector));
			
		}
		
	}
	
	
	public HashSet<IDiagramElementViewConformityRule> getRules() {
		return rules;
	}
	
	
	/////////////////////////
	//ERRORS
	////////////
	
	
	////////////A CONNECTOR MUST HAVE A SOURCE AND A TARGET
	private class ConnectorMustHaveSourceAndTarget extends AbstractDiagramElementViewConformityRule {

		public ConnectorMustHaveSourceAndTarget(
				IDiagramElementView diagramElementView) {
			super(diagramElementView);

		}

		@Override
		public boolean isConform(IDiagramElementView diagramElementView) {
			
			IConnector connector = (IConnector) diagramElementView;
			
			if (connector.getSource()==null || connector.getTarget()==null){
				return false;
			}
			
			return true;
		}

		@Override
		public boolean canResolveNonConformity() {

			return false;
		}

		@Override
		public String getResolveConformityDescription() {
			return "Connect actual connector to a source and a target";
		}

		@Override
		public String getRuleDescription() {	
			return "A connector must have both extremities connected to an element";
		}

		@Override
		public IRuleLevel getRuleLevel() {
			return RuleLevel.ERROR;
		}

		@Override
		public String getRuleName() {
			return "Connector source and target required";
		}

		@Override
		public void resolveNonConformity() {
			
		}

	}
	
	
	/////////////////A SEQUENCE FLOW SOURCE AND TARGET MUST BE A FLOW NODE FROM THE SAME PARTICIPANT/PROCESS
	private class SequenceFlowSourceAndTargetMustBeAFlowNodeFromSame extends AbstractDiagramElementViewConformityRule {

		public SequenceFlowSourceAndTargetMustBeAFlowNodeFromSame(
				IDiagramElementView diagramElementView) {
			super(diagramElementView);

		}

		@Override
		public boolean isConform(IDiagramElementView diagramElementView) {
			
			IConnector connector = (IConnector) diagramElementView;
			
			IDiagramElementView source = (IDiagramElementView) connector.getSource();
			IDiagramElementView target = (IDiagramElementView) connector.getTarget();
			
			//only if there is a source and target => their should be a rule for that case
			if (source!=null && target!=null){
				
				//first check that source and target are FlowNodes
				if (source.getDiagramElement().getModelElement() instanceof IFlowNodeBean
					&& target.getDiagramElement().getModelElement() instanceof IFlowNodeBean){
					
					IDiagramElement sourceFlowNode = source.getDiagramElement();
					IDiagramElement targetFlowNode = target.getDiagramElement();
					
					if (sourceFlowNode.getOwningElement().getModelElement() instanceof LaneBean &&
						targetFlowNode.getOwningElement().getModelElement() instanceof LaneBean){
						
						//then check if the pool of both is the same
						if (sourceFlowNode.getOwningElement().getOwningElement()!=targetFlowNode.getOwningElement().getOwningElement()){
							return false;
						}	
						
					}
					
					
					
				//if source and target aren't flow nodes => Error		
				}else{
					return false;
				}
			
			}
			
			return true;
		}

		@Override
		public boolean canResolveNonConformity() {

			return false;
		}

		@Override
		public String getResolveConformityDescription() {
			return "Connect the sequence flow to a flow node of the same process / participant / pool";
		}

		@Override
		public String getRuleDescription() {
			return "A Sequence Flow source and target must be a flow node (Activity/Event/Gateway) that belongs to the same process / participant / pool";
		}

		@Override
		public IRuleLevel getRuleLevel() {
			return RuleLevel.ERROR;
		}

		@Override
		public String getRuleName() {
			return "Invalid Sequence Flow source/target";
		}

		@Override
		public void resolveNonConformity() {
		
			
		}
		
		
		
	}
	
	/////////////////A MESSAGE FLOW SOURCE AND TARGET MUST BE AN INTERACTION NODE FROM TWO DIFFERENT PARTICIPANT/PROCESS
	private class MessageFlowSourceAndTargetMustBeAnInteractionNodeFromDifferentProcess extends AbstractDiagramElementViewConformityRule {

		public MessageFlowSourceAndTargetMustBeAnInteractionNodeFromDifferentProcess(
				IDiagramElementView diagramElementView) {
			super(diagramElementView);
		}

		@Override
		public boolean isConform(IDiagramElementView diagramElementView) {
			
			IConnector connector = (IConnector) diagramElementView;
			
			IDiagramElementView source = (IDiagramElementView) connector.getSource();
			IDiagramElementView target = (IDiagramElementView) connector.getTarget();
			
			//only if there is a source and target => their should be a rule for that case
			if (source!=null && target!=null){
				
				//first check that source and target are IInteractionNode
				if (source.getDiagramElement().getModelElement() instanceof IInteractionNodeBean
					&& target.getDiagramElement().getModelElement() instanceof IInteractionNodeBean){
					
					//if the source and target are two flow node => check that they dont belong
					//to the same process
					if (source.getDiagramElement().getModelElement() instanceof IFlowNodeBean
						&& target.getDiagramElement().getModelElement() instanceof IFlowNodeBean){
						
						IFlowNodeBean sourceFlowNode = (IFlowNodeBean) source.getDiagramElement().getModelElement();
						IFlowNodeBean targetFlowNode = (IFlowNodeBean) target.getDiagramElement().getModelElement();
						
						//then check if the process of both are different
						if (sourceFlowNode.getProcess()==targetFlowNode.getProcess()){
							return false;
						}
					
					}
					
				//if source and target aren't Interaction NOde => Error		
				}else{
					return false;
				}
			
			}
			
			return true;
		}

		@Override
		public boolean canResolveNonConformity() {
			
			return false;
		}

		@Override
		public String getResolveConformityDescription() {
			return "Connect the Message Flow to an interaction node from two different process / participant / pool";
		}

		@Override
		public String getRuleDescription() {
			return "A Message Flow source and target must be an interaction node (Activity/Event/Participant) from two different process / participant / pool";
		}

		@Override
		public IRuleLevel getRuleLevel() {
			return RuleLevel.ERROR;
		}

		@Override
		public String getRuleName() {
			return "Invalid Message Flow source/target";
		}

		@Override
		public void resolveNonConformity() {
			// TODO Auto-generated method stub
			
		}
		

	}
	
	
	
	///////////////A CONNECTOR CANNOT HAVE THE SAME SOURCE AND TARGET
	private class ConnectorCannotHaveSameSourceAndTarget extends AbstractDiagramElementViewConformityRule {

		public ConnectorCannotHaveSameSourceAndTarget(
				IDiagramElementView diagramElementView) {
			super(diagramElementView);
			
		}

		@Override
		public boolean isConform(IDiagramElementView diagramElementView) {
			
			IConnector connector = (IConnector) diagramElementView;
			
			IDiagramElementView source = (IDiagramElementView) connector.getSource();
			IDiagramElementView target = (IDiagramElementView) connector.getTarget();
			
			if (source==target){
				return false;
			}
			
			return true;
		}

		@Override
		public boolean canResolveNonConformity() {

			return false;
		}

		@Override
		public String getResolveConformityDescription() {
			return "Select a different source or target";
		}

		@Override
		public String getRuleDescription() {
			return "A connector source cannot be its target";
		}

		@Override
		public IRuleLevel getRuleLevel() {
			return RuleLevel.ERROR;
		}

		@Override
		public String getRuleName() {
			return "Invalid connector source/target";
		}

		@Override
		public void resolveNonConformity() {
			// TODO Auto-generated method stub
			
		}
		
		
		
		
		
	}
	
	
	/////////////////A SEQUENCE FLOW CAN HAVE A CONDITION ONLY WHEN SOURCE IS AN ACTIVITY OR AN EXCLUSIVE GATEWAY
	private class SequenceFlowCanHaveConditionOnlyWhenSourceIsActivityOrExclusiveGateway extends AbstractDiagramElementViewConformityRule {

		public SequenceFlowCanHaveConditionOnlyWhenSourceIsActivityOrExclusiveGateway(
				IDiagramElementView diagramElementView) {
			super(diagramElementView);
		}

		@Override
		public boolean isConform(IDiagramElementView diagramElementView) {
			
			IConnector connector = (IConnector) diagramElementView;
			
			IDiagramElementView source = (IDiagramElementView) connector.getSource();
			
			SequenceFlowEditorModel model = (SequenceFlowEditorModel) diagramElementView.getEditorModel();
			
			if (model.getConditionnalExpression()!=null){
			
				if (model.getConditionnalExpression().equals("")==false){
				
					if (source.getDiagramElement().getModelElement() instanceof ActivityBean 
					||  source.getDiagramElement().getModelElement() instanceof ExclusiveGatewayBean){
						
						return true;
						
					}
				
				}else{
					return true;
				}
				
			}else{
				return true;
			}
			
			
			return false;
		}

		@Override
		public boolean canResolveNonConformity() {
			// TODO Auto-generated method stub
			return false;
		}

		@Override
		public String getResolveConformityDescription() {
			return "Remove condition from Sequence Flow";
		}

		@Override
		public String getRuleDescription() {
			return "A Sequence Flow can only have a condition when its source is an Activity or an Exclusive Gateway";
		}

		@Override
		public IRuleLevel getRuleLevel() {
			return RuleLevel.ERROR;
		}

		@Override
		public String getRuleName() {
			return "Condition not supported";
		}

		@Override
		public void resolveNonConformity() {
			// TODO Auto-generated method stub
			
		}
		
		
		
	}
	
}
