/**
 * petalsbpm-domain - Domain Objects for BPMN standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.infrastructure;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.extension.client.AttributeExtension;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.choreography.IChoreographyBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.ICollaborationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IErrorBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IEscalationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IItemDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IMessageBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IParticipantBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IPartnerRoleBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.infrastructure.IDefinitionsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.infrastructure.IImportBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.infrastructure.INamespaceDeclaration;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.IProcessBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.ISignalBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.definition.IConditionalEventDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.definition.IEventDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.definition.IMessageEventDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.definition.ISignalEventDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.definition.ITimerEventDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.service.IEndPointBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.service.IInterfaceBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.service.IOperationBean;


public class DefinitionsBean implements Serializable, IDefinitionsBean {
    
    public static final String DEFAULT_TNS = "http://com.ebmwebsourcing.petalsbpm/model";
    public static final String DEFAUTL_ExpressionLanguage = "http://www.w3.org/1999/XPath";
    public static final String DEFAULT_TypeLanguage = "http://www.w3.org/2001/XMLSchema";

	private String targetNamespace;
	private String expressionLanguage;
	private String typeLanguage;
	private String id;
	private String exporter;
	private String exporterVersion;
	
	
	private List<IImportBean> imports;
	private List<ICollaborationBean> collaborations;
	private List<IChoreographyBean> choreographies;
	private List<IProcessBean> processes;
	private List<IInterfaceBean> interfaces;
	private List<IPartnerRoleBean> partnerRoles;
	private List<IMessageBean> messages;
	private List<IItemDefinitionBean> itemDefinitions;
	private List<ISignalBean> signals;
	private List<IEndPointBean> endpoints;
	private List<IEscalationBean> escalations;
	private List<IErrorBean> errors;
	private List<IEventDefinitionBean> eventDefinitions;
	
	private List<AttributeExtension> attributeExtensions;
	
	
	
	// not part of bpmn2.0 specification
	private String name;
	//private HashSet<INamespaceDeclaration> importNsDeclarations;
	private HashMap<String,INamespaceDeclaration> importNsDeclarations;
	
	protected DefinitionsBean(){
		
		imports = new ArrayList<IImportBean>();
		collaborations = new ArrayList<ICollaborationBean>();
		processes = new ArrayList<IProcessBean>();
		interfaces = new ArrayList<IInterfaceBean>();
		partnerRoles = new ArrayList<IPartnerRoleBean>();
		messages = new ArrayList<IMessageBean>();
		itemDefinitions = new ArrayList<IItemDefinitionBean>();
		signals = new ArrayList<ISignalBean>();
		choreographies = new ArrayList<IChoreographyBean>();
		attributeExtensions = new ArrayList<AttributeExtension>();
		endpoints = new ArrayList<IEndPointBean>();
		escalations = new ArrayList<IEscalationBean>();
		errors = new ArrayList<IErrorBean>();
		eventDefinitions = new ArrayList<IEventDefinitionBean>();
		
		
		this.importNsDeclarations = new HashMap<String,INamespaceDeclaration>();
		
		targetNamespace = DEFAULT_TNS;
		expressionLanguage = DEFAUTL_ExpressionLanguage;
		typeLanguage = DEFAULT_TypeLanguage;
	}

	public DefinitionsBean(String id){
		this();
		this.id = id;
	}

	public void addImport(IImportBean i){
		imports.add(i);
		importNsDeclarations.put(i.getNamespace(),i.getNSDeclaration());
	}
	
	public void removeImport(IImportBean i){
		imports.remove(i);
		for(IImportBean impt : imports) {
		    if(impt.getNamespace().equals(i.getNamespace())) {
		        return;
		    }
		}
		importNsDeclarations.remove(i.getNamespace());
	}
	
	public void addChoreography(IChoreographyBean c){
	    if (choreographies.contains(c)==false) choreographies.add(c);
	    c.setParentDefinitions(this);
	    for(IMessageBean m:c.getMessages()){
	    	addMessage(m);
	    }
	}

	public void removeChoreography(IChoreographyBean c){
        choreographies.add(c);
        c.setParentDefinitions(null);
    }
	
	public void addItemDefinition(IItemDefinitionBean i){
		itemDefinitions.add(i);
	}
	
	public void removeItemDefinition(IItemDefinitionBean i){
		itemDefinitions.remove(i);
	}
	
	public IItemDefinitionBean getItemDefinitionById(String id){
		for(IItemDefinitionBean e : getItemDefinitions()){
			if(e.getId().equals(id)){
				return e;
			}
		}
		return null;
	}

	public void addCollaboration(ICollaborationBean c){
	    if(!(c instanceof IChoreographyBean)){
	        if (collaborations.contains(c)==false) collaborations.add(c);
	        c.setParentDefinitions(this);
	        sync(c);
	    }
	}

	public void sync(ICollaborationBean c){
		for(IParticipantBean p:c.getParticipants()){
		    if(p.getProcess()!=null) {
		        addProcess(p.getProcess());
		    }
		}
		for(IMessageBean m:c.getMessages()){
			addMessage(m);
		}
	}
	
	public void removeCollaboration(ICollaborationBean c){
	    if(!(c instanceof IChoreographyBean)){
            collaborations.remove(c);
            c.setParentDefinitions(null);
        }
	}
	
	public List<IParticipantBean> getParticipants() {
		List<IParticipantBean> l = new ArrayList<IParticipantBean>();
		for(ICollaborationBean c: collaborations){
			l.addAll(c.getParticipants());
		}
		return l;
	}
	
	public IParticipantBean getParticipantById(String id) {
		for(IParticipantBean p: getParticipants()){
			if(p.getId().equals(id)){
				return p;
			}
		}
		return null;
	}
	
	public void addInterface(IInterfaceBean i){
		interfaces.add(i);
	}
	
	public void removeInterface(IInterfaceBean i){
		interfaces.remove(i);
	}
	
	public IOperationBean getOperationById(String id){
		for(IInterfaceBean itf : interfaces){
		    IOperationBean op = itf.getOperationById(id);
			if(op!=null){
				return op;
			}
		}
		return null;
	}
	
	public void addProcess(IProcessBean p){
		if (p!=null && !processes.contains(p)) {
		    processes.add(p);
		    p.setParentDefinitions(this);
		}
	}
	
	public void removeProcess(IProcessBean p){
		processes.remove(p);
		p.setParentDefinitions(null);
	}
	
	public IProcessBean getProcessById(String id){
	    for(IProcessBean p : processes){
	        if(p.getId().equals(id)){
	            return p;
	        }
	    }
	    return null;
	}
	
	public void addPartnerRole(IPartnerRoleBean pr){
		partnerRoles.add(pr);
	}
	
	public void removePartnerRole(IPartnerRoleBean pr){
		partnerRoles.remove(pr);
	}
	
	public void addMessage(IMessageBean m){
		if (messages.contains(m)==false) messages.add(m);
	}
	
	public void removeMessage(IMessageBean m){
		messages.remove(m);
	}
	
	public void addSignal(ISignalBean m){
        signals.add(m);
    }
    
    public void removeSignal(ISignalBean m){
        signals.remove(m);
    }
	
	public IMessageBean getMessageById(String id){
		for(IMessageBean e : getMessages()){
			if(e.getId().equals(id)){
				return e;
			}
		}
		return null;
	}
	
	public void addAttributeExtension(AttributeExtension att){
		for(AttributeExtension a : attributeExtensions){
			if(a.hasSameErasure(att)){
				throw new IllegalArgumentException(
						"This element ("+id+") already has an attribute with QName "
						+a.getAttributeQNameNS()+":"+a.getAttributeQNameLocalPart()+".");
			}
		}
		this.attributeExtensions.add(att);
	}
	
	public void removeAttributeExtension(AttributeExtension att){
		this.attributeExtensions.remove(att);
	}
	
	public AttributeExtension getAttributeExtension(String ns, String lp){
		for(AttributeExtension att : attributeExtensions){
			if(att.getAttributeQNameNS().equals(ns) && att.getAttributeQNameLocalPart().equals(lp)){
				return att;
			}
		}
		return null;
	}

	public void addEndPoint(IEndPointBean ep){
        endpoints.add(ep);
    }
    
    public void removeEndPoint(IEndPointBean ep){
        endpoints.add(ep);
    }
    
    @Override
    public void addError(IErrorBean ep) {
        errors.add(ep);
    }

    @Override
    public void addEscalation(IEscalationBean ep) {
        escalations.add(ep);
    }

    @Override
    public void removeError(IErrorBean ep) {
        errors.remove(ep);
    }

    @Override
    public void removeEscalation(IEscalationBean ep) {
        escalations.remove(ep);
    }
    
    @Override
    public INamespaceDeclaration getNsDeclarationByNamespace(String ns) {
        for(INamespaceDeclaration nsd : importNsDeclarations.values()) {
            if(nsd.getURI().equals(ns)) {
                return nsd;
            }
        }
        return null;
    }
    
    @Override
    public INamespaceDeclaration getNsDeclarationByPrefix(String prefix) {
        for(INamespaceDeclaration nsd : importNsDeclarations.values()) {
            if(nsd.getPrefix().equals(prefix)) {
                return nsd;
            }
        }
        return null;
    }
	////////////////////////////////////////////
	/////                                  /////
	/////       GETTERS AND SETTERS        /////
	/////                                  /////
	////////////////////////////////////////////
	public String getId() {
		return id;
	}
	public void setId(String id) {
		this.id = id;
	}
	public String getName(){
		if (name==null){
			return id;
		}
		return this.name;
	}
	public void setName(String name){
		this.name = name;
	}
	public String getTargetNamespace() {
		return targetNamespace;
	}
	public void setTargetNamespace(String targetNamespace) {
		this.targetNamespace = targetNamespace;
	}
	public String getExpressionLanguage() {
		return expressionLanguage;
	}
	public void setExpressionLanguage(String expressionLanguage) {
		this.expressionLanguage = expressionLanguage;
	}
	public String getTypeLanguage() {
		return typeLanguage;
	}
	public void setTypeLanguage(String typeLanguage) {
		this.typeLanguage = typeLanguage;
	}
	public List<IImportBean> getImports() {
		return imports;
	}
	public void setImports(List<IImportBean> imports) {
		this.imports = imports;
	}
	public List<ICollaborationBean> getCollaborations() {
		return collaborations;
	}
	public void setCollaborations(List<ICollaborationBean> collaborations) {
		this.collaborations = collaborations;
	}
	public List<IProcessBean> getProcesses() {
		return processes;
	}
	public List<IInterfaceBean> getInterfaces() {
		return interfaces;
	}
	
	public List<IOperationBean> getOperations(){
		
		List<IOperationBean> result = new ArrayList<IOperationBean>();
		
		for(IInterfaceBean i:getInterfaces()){
			
			for(IOperationBean o: i.getOperations()){
				result.add(o);
			}
			
		}
		
		return result;
	}
	
	public List<IPartnerRoleBean> getPartnerRoles() {
		return partnerRoles;
	}
	public List<IMessageBean> getMessages() {
		return messages;
	}
	public void setMessages(List<IMessageBean> messages) {
		this.messages = messages;
	}
	public void setProcesses(List<IProcessBean> processes) {
		this.processes = processes;
	}
	public void setInterfaces(List<IInterfaceBean> interfaces) {
		this.interfaces = interfaces;
	}
	public void setPartnerRoles(List<IPartnerRoleBean> partnerRoles) {
		this.partnerRoles = partnerRoles;
	}
	public List<IItemDefinitionBean> getItemDefinitions() {
		return itemDefinitions;
	}
	public void setItemDefinitions(List<IItemDefinitionBean> itemDefinitions) {
		this.itemDefinitions = itemDefinitions;
	}
	public List<AttributeExtension> getAttributeExtensions() {
		return attributeExtensions;
	}
	public void setAttributeExtensions(
			List<AttributeExtension> attributeExtensions) {
		this.attributeExtensions = attributeExtensions;
	}
    public List<ISignalBean> getSignals() {
        return signals;
    }
    public void setSignals(List<ISignalBean> signals) {
        this.signals = signals;
    }
    public List<IChoreographyBean> getChoreographies() {
        return choreographies;
    }
    public void setChoreographies(List<IChoreographyBean> choreographies) {
        this.choreographies = choreographies;
    }
    public List<IEndPointBean> getEndPoints() {
        return endpoints;
    }
    public void setEndPoints(List<IEndPointBean> endpoints) {
        this.endpoints = endpoints;
    }
    public List<IErrorBean> getErrors() {
        return errors;
    }
    public List<IEscalationBean> getEscalations() {
        return escalations;
    }
    public void setErrors(List<IErrorBean> ep) {
        errors = ep;
    }
    public void setEscalations(List<IEscalationBean> ep) {
        escalations = ep;
    }

	@Override
	public Collection<INamespaceDeclaration> getImportsNsDeclarations() {
		return this.importNsDeclarations.values();
	}

	@Override
	public void addEventDefinition(IEventDefinitionBean ev) {
	    if(!eventDefinitions.contains(ev)) {
	        this.eventDefinitions.add(ev);
	    }
	}

	@Override
	public List<IEventDefinitionBean> getEventDefinitions() {
		return this.eventDefinitions;
	}

	@Override
	public void removeEventDefinition(IEventDefinitionBean ev) {
		this.eventDefinitions.remove(ev);
	}

	@Override
	public void setEventDefinitions(List<IEventDefinitionBean> evs) {
		this.eventDefinitions = evs;
	}

	@Override
	public List<IConditionalEventDefinitionBean> getConditionalEventDefinitions() {
		
		List<IConditionalEventDefinitionBean> result = new ArrayList<IConditionalEventDefinitionBean>();
		
		for(IEventDefinitionBean evd:eventDefinitions){
			
			if (evd instanceof IConditionalEventDefinitionBean){
				result.add((IConditionalEventDefinitionBean) evd);
			}
			
		}
		
		
		return result;
	}

	@Override
	public List<IMessageEventDefinitionBean> getMessageEventDefinitions() {
		
		List<IMessageEventDefinitionBean> result = new ArrayList<IMessageEventDefinitionBean>();
		
		for(IEventDefinitionBean evd:eventDefinitions){
			
			if (evd instanceof IMessageEventDefinitionBean){
				result.add((IMessageEventDefinitionBean) evd);
			}
			
		}
		
		
		return result;
	}

	@Override
	public List<ISignalEventDefinitionBean> getSignalEventDefinitions() {
		
		List<ISignalEventDefinitionBean> result = new ArrayList<ISignalEventDefinitionBean>();
		
		for(IEventDefinitionBean evd:eventDefinitions){
			
			if (evd instanceof ISignalEventDefinitionBean){
				result.add((ISignalEventDefinitionBean) evd);
			}
			
		}
		
		return result;
	}

	@Override
	public List<ITimerEventDefinitionBean> getTimerEventDefinitions() {
		
		List<ITimerEventDefinitionBean> result = new ArrayList<ITimerEventDefinitionBean>();
		
		for(IEventDefinitionBean evd:eventDefinitions){
			
			if (evd instanceof ITimerEventDefinitionBean){
				result.add((ITimerEventDefinitionBean) evd);
			}
			
		}
		
		return result;
	}

	@Override
	public String getExporter() {
		return this.exporter;
	}

	@Override
	public String getExporterVersion() {
		return this.exporterVersion;
	}

	@Override
	public void setExporter(String exporter) {
		this.exporter = exporter;
	}

	@Override
	public void setExporterVersion(String exporterVersion) {
		this.exporterVersion = exporterVersion;
	}

}
