/**
 * Web commons : user service.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * UserManager.java
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.webcommons.user.api.service;

import java.io.IOException;
import java.util.List;

import org.springframework.transaction.annotation.Transactional;

import com.ebmwebsourcing.webcommons.user.api.to.UserTO;
import com.ebmwebsourcing.webcommons.user.persistence.bo.RoleGroup;

/**
 * A manager of User. Provides methods to create, retrieve, delete or update
 * User and to manage role group related to User. Method arguments must be non
 * null, unless the contrary is explicitly specified.
 * 
 * @author strino - eBM WebSourcing
 * 
 */

@Transactional
public interface UserManager {

    /**
     * Create default users if they don't exist in database
     * 
     */
    public void init() throws IOException, RoleGroupException;

    /**
     * Add a new User in registry. The User name must be specified.
     * 
     * @param userTO
     *            the {@link UserTO} handling data of the {@link User} to create
     * @return the id of the added {@link User}
     * @throws UserException
     *             if you try to create an user with name that already exists or
     *             an user without name
     */
    public String createUser(UserTO userTO) throws UserException;

    /**
     * Return the list of all users in the registry
     * 
     * @return the {@link List} of all {@link UserTO} in the registry, couldn't
     *         be null, can be empty
     */
    @Transactional(readOnly = true)
    public List<UserTO> getAllUser() throws UserException;

    /**
     * Return the list of users associate with role group
     * 
     * @param idRoleGroup
     *            an {@link RoleGroup} id
     * @return the {@link List} of {@link UserTO} associate with
     *         {@link RoleGroup}, couldn't be null, can be empty
     */
    @Transactional(readOnly = true)
    public List<UserTO> getUserNotInRoleGroup(String idRoleGroup) throws UserException;

    /**
     * Retrieve the user matching the given name
     * 
     * @param userName
     *            an {@link User} name
     * @return the {@link UserTO} handling data of the {@link User} that matches
     *         the given id
     * @throws UserException
     *             if no role group found for the given name
     */
    @Transactional(readOnly = true)
    public UserTO getUser(String userId) throws UserException;

    /**
     * Retrieve the user matching the given login
     * 
     * @param login
     *            an {@link User} login
     * @return the {@link UserTO} handling data of the {@link User} that matches
     *         the given id
     * @throws UserException
     *             if no role group found for the given name
     */
    @Transactional(readOnly = true)
    public UserTO getUserByLogin(String login) throws UserException;

    /**
     * Remove the user matching the given id from the registry.
     * 
     * @param userId
     *            an {@link User} id
     */
    public void removeUser(String userId) throws UserException;

    /**
     * Update the user in the registry.
     * 
     * @param userTO
     *            the {@link UserTO} handling data of the {@link User} to update
     * @return the updated user ID
     * @throws UserException
     *             if an error occurs during user update : user to update
     *             doesn't exist or have the same name as an other registered
     *             user
     */
    public String updateUser(UserTO userTO) throws UserException;

    /**
     * Add the role group matching the given idRoleGroup to the user matching
     * the given idUser
     * 
     * @param idUser
     *            an {@link User} id
     * @param idRoleGroup
     *            a {@link RoleGroup} id
     * @throws UserException
     *             if at least one of the given id doesn't exist in database
     */
    public void addRoleGroup(String idUser, String idRoleGroup) throws UserException;

    /**
     * Remove the role group matching the given idRoleGroup from the list of
     * role groups of the user matching the given idUser. The role group isn't
     * removed from the registry and can be re-affected to any organization
     * 
     * @param idUser
     *            an {@link User} id
     * @param idRoleGroup
     *            a {@link RoleGroup} id
     * @throws UserException
     *             if no organization found for the given id
     */
    public void removeRoleGroup(String idUser, String idRoleGroup) throws UserException;

}
