/**
 * Web commons : user service.
 * Copyright (c) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * RoleGroupManagerImpl.java
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.webcommons.user.service;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;

import org.apache.log4j.Logger;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PropertiesLoaderUtils;

import com.ebmwebsourcing.webcommons.aop.annotation.CheckAllArgumentsNotNull;
import com.ebmwebsourcing.webcommons.user.api.service.RoleGroupException;
import com.ebmwebsourcing.webcommons.user.api.service.RoleGroupManager;
import com.ebmwebsourcing.webcommons.user.api.to.RoleGroupTO;
import com.ebmwebsourcing.webcommons.user.persistence.bo.Role;
import com.ebmwebsourcing.webcommons.user.persistence.bo.RoleGroup;
import com.ebmwebsourcing.webcommons.user.persistence.bo.User;
import com.ebmwebsourcing.webcommons.user.persistence.dao.RoleDAO;
import com.ebmwebsourcing.webcommons.user.persistence.dao.RoleGroupDAO;
import com.ebmwebsourcing.webcommons.user.persistence.dao.UserDAO;
import com.ebmwebsourcing.webcommons.util.StringHelper;

/**
 * @author strino - eBM WebSourcing
 * 
 */
public class RoleGroupManagerImpl implements RoleGroupManager {

    private final Logger logger = Logger.getLogger(this.getClass());

    private RolesAndUsersTransfertObjectAssembler rolesAndUsersTransfertObjectAssembler;

    private RoleGroupDAO roleGroupDAO;

    private RoleDAO roleDAO;

    private UserDAO userDAO;

    private Resource roleGroupsResource;

    private RoleGroupManager roleGroupManager;

    public void nonTransactionalInit() throws IOException, RoleGroupException {
        logger.debug("###### Initialize RoleGroupManager bean");
        roleGroupManager.init();
        logger.debug("###### RoleGroupManager bean initialized");
    }

    public void setRoleGroupManager(RoleGroupManager roleGroupManager) {
        this.roleGroupManager = roleGroupManager;
    }

    public void setRoleGroupsResource(Resource roleGroupsResource) {
        this.roleGroupsResource = roleGroupsResource;
    }

    // create the roles group admin if not exists in database
    public void init() throws IOException, RoleGroupException {

        // Create role group to role map
        Properties rolesGroupProps = PropertiesLoaderUtils.loadProperties(roleGroupsResource);
        Map<String, List<String>> roleGroupsMap = this
                .createRoleGroupToRoleListMap(rolesGroupProps);

        // retrieve if the role group already exist
        for (String roleGroup : roleGroupsMap.keySet()) {
            logger.debug("Create Role group : " + roleGroup);
            createRoleGroup(roleGroup, roleGroupsMap.get(roleGroup));
        }

    }

    private void createRoleGroup(String groupName, List<String> roleNames)
            throws RoleGroupException {
        RoleGroup roleGroupBO = roleGroupDAO.getRoleGroupByName(groupName);

        if (roleGroupBO == null) {
            RoleGroup roleGroup = new RoleGroup();
            roleGroup.setName(groupName);

            // retrieve role related to role group
            List<Role> listRoles = this.retrieveRole(roleNames);
            roleGroup.setListRoles(listRoles);

            // save the role group in database
            roleGroupDAO.save(roleGroup);
        }
    }

    private List<Role> retrieveRole(List<String> roleNames) throws RoleGroupException {
        List<Role> roles = new ArrayList<Role>();
        for (String string : roleNames) {
            Role role = this.roleDAO.getRoleByName(string);
            if (role != null) {
                roles.add(role);
            } else {
                throw new RoleGroupException("Can't create role group because role '" + string
                        + "' doesn't exist");
            }
        }
        return roles;
    }

    private Map<String, List<String>> createRoleGroupToRoleListMap(Properties rolesGroupProps) {
        Map<String, List<String>> result = new HashMap<String, List<String>>();
        for (Object roleGroupProp : rolesGroupProps.keySet()) {
            List<String> roleList = new ArrayList<String>();
            String roles = rolesGroupProps.getProperty((String) roleGroupProp);
            StringTokenizer tokenizer = new StringTokenizer(roles, ",");
            while (tokenizer.hasMoreElements()) {
                String roleName = tokenizer.nextToken();
                roleList.add(roleName.trim());
            }
            result.put((String) roleGroupProp, roleList);
        }
        return result;
    }

    @CheckAllArgumentsNotNull
    public String createRoleGroup(RoleGroupTO roleGroupTO) throws RoleGroupException {

        RoleGroup roleGroup = new RoleGroup();

        this.validateBeforeSaveOrUpdate(roleGroupTO, null);

        this.rolesAndUsersTransfertObjectAssembler.toRoleGroupBO(roleGroupTO, roleGroup);

        List<Role> listR = new ArrayList<Role>();
        List<Role> listRole = roleGroup.getListRoles();

        for (Role ro : listRole) {

            // retrieve the id role
            ro = this.roleDAO.getRoleByName(ro.getName());
            listR.add(ro);
        }

        roleGroup.setListRoles(listR);

        // create role group
        return this.roleGroupDAO.save(roleGroup).getId();

    }

    public List<RoleGroupTO> getAllRoleGroup() throws RoleGroupException {

        // retrieve all role group
        List<RoleGroup> listAllGroupe = this.roleGroupDAO.getAll();

        // transform role groups
        List<RoleGroupTO> listRole = this.rolesAndUsersTransfertObjectAssembler
                .toAllGroupes(listAllGroupe);

        return listRole;
    }

    @CheckAllArgumentsNotNull
    public List<RoleGroupTO> getRoleGroupNotInUser(String idUser) throws RoleGroupException {

        List<RoleGroupTO> listRoleGroupTO = new ArrayList<RoleGroupTO>();

        // retrieve user with its id
        User user = this.userDAO.get(idUser);

        List<RoleGroup> listRoleGroupInUser = new ArrayList<RoleGroup>();

        if (user != null) {
            listRoleGroupInUser = user.getListRoleGroup();
        } else {
            throw new RoleGroupException("A user with that name doesn't exist ! ");
        }

        // retrieve all role groups
        List<RoleGroup> allRoleGroup = this.roleGroupDAO.getAll();

        if (allRoleGroup != null) {

            if (!listRoleGroupInUser.isEmpty()) {

                // remove user role's groups from list all role groups
                allRoleGroup.removeAll(listRoleGroupInUser);

            }
            for (RoleGroup rG : allRoleGroup) {

                // transform role groups
                listRoleGroupTO.add(this.rolesAndUsersTransfertObjectAssembler.toRoleGroupTO(rG));
            }
        } else {
            throw new RoleGroupException("no role groups in database ! ");
        }

        return listRoleGroupTO;
    }

    @CheckAllArgumentsNotNull
    public RoleGroupTO getRoleGroup(String idGroup) throws RoleGroupException {

        // retrieve role group
        RoleGroup roleGroup = this.roleGroupDAO.get(idGroup);

        if (roleGroup == null) {
            throw new RoleGroupException("no role groups whith this id in database ! ");
        }

        // transform role group
        RoleGroupTO role = this.rolesAndUsersTransfertObjectAssembler.toRoleGroupTO(roleGroup);

        return role;
    }

    @CheckAllArgumentsNotNull
    public void removeRoleGroup(String idGroup) throws RoleGroupException {

        // retrieve role group by its id
        RoleGroup roleGroup = this.roleGroupDAO.get(idGroup);

        if (roleGroup == null) {
            throw new RoleGroupException("no role groups whith this id in database ! ");
        }
        
        roleGroup.clearUserList();
        roleGroupDAO.save(roleGroup);
        
        // remove role group
        this.roleGroupDAO.remove(roleGroup);
    }
    
    @CheckAllArgumentsNotNull
    public String updateRoleGroup(RoleGroupTO roleGroupTO) throws RoleGroupException {

        String id = roleGroupTO.getId();

        if (id == null) {
            throw new RoleGroupException("this role doesn't already exist in database : ");
        }

        // retrieve role group by its id
        RoleGroup group = this.roleGroupDAO.get(roleGroupTO.getId());

        if (group == null) {
            throw new RoleGroupException("this role doesn't already exist in database : " + id);
        }

        this.validateBeforeSaveOrUpdate(roleGroupTO, group.getName());

        this.rolesAndUsersTransfertObjectAssembler.toRoleGroupBO(roleGroupTO, group);

        // transform List<String> to List<Role>
        List<Role> listR = new ArrayList<Role>();
        List<Role> listRole = group.getListRoles();

        for (Role ro : listRole) {
            // retrieve the id role
            ro = this.roleDAO.getRoleByName(ro.getName());
            listR.add(ro);
        }

        group.setListRoles(listR);

        this.roleGroupDAO.save(group);

        return group.getId();
    }

    @CheckAllArgumentsNotNull
    public void removeUser(String idUser, String idRoleGroup) throws RoleGroupException {
        // retrieve roleGroup by its id
        RoleGroup roleGroup = this.roleGroupDAO.get(idRoleGroup);
        if (roleGroup == null) {
            throw new RoleGroupException(
                    "You are trying to remove a user to a non existing role whit id : "
                            + idRoleGroup);
        }

        // retrieve user by its id
        User user = this.userDAO.get(idUser);
        if (user == null) {
            throw new RoleGroupException(
                    "You are trying to remove a non existing user to a role whith id : "
                            + idRoleGroup);
        }

        roleGroup.removeUser(user);

        this.roleGroupDAO.save(roleGroup);
    }

    @CheckAllArgumentsNotNull
    public void addUser(String idUser, String idRoleGroup) throws RoleGroupException {

        // retrieve role group with its id
        RoleGroup roleGroup = this.roleGroupDAO.get(idRoleGroup);

        if (roleGroup == null) {
            throw new RoleGroupException(
                    "You are trying to add a user to a non existing role whit id : " + idRoleGroup);
        }

        // retrieve user with its id
        User user = this.userDAO.get(idUser);

        if (user == null) {
            throw new RoleGroupException(
                    "You are trying to add a non existing user to a role whith id : " + idRoleGroup);
        }
        // insert the new user in the RoleGroup list<User>
        roleGroup.addUser(user);

        // save role Group
        this.roleGroupDAO.save(roleGroup);

    }

    private void validateBeforeSaveOrUpdate(RoleGroupTO roleGroup, String nameRoleGroup)
            throws RoleGroupException {
        String roleGroupNameTO = roleGroup.getName();
        if (StringHelper.isNullOrEmpty(roleGroupNameTO)) {
            throw new RoleGroupException("Role Name must be speficied.");
        }

        if (!roleGroupNameTO.equalsIgnoreCase(nameRoleGroup)) {
            RoleGroup group = this.roleGroupDAO.getRoleGroupByName(roleGroupNameTO);
            if (group != null) {
                throw new RoleGroupException("A role with that name already exists : "
                        + roleGroupNameTO);
            }
        }

    }

    public void setRolesAndUsersTransfertObjectAssembler(
            RolesAndUsersTransfertObjectAssembler rolesAndUsersTransfertObjectAssembler) {
        this.rolesAndUsersTransfertObjectAssembler = rolesAndUsersTransfertObjectAssembler;
    }

    public void setRoleGroupDAO(RoleGroupDAO roleGroupDAO) {
        this.roleGroupDAO = roleGroupDAO;
    }

    public void setRoleDAO(RoleDAO roleDAO) {
        this.roleDAO = roleDAO;
    }

    public void setUserDAO(UserDAO userDAO) {
        this.userDAO = userDAO;
    }

}
