/**
 * Web commons : user service.
 * Copyright (c) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * UserManagerImpl.java
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.webcommons.user.service;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;

import org.apache.log4j.Logger;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PropertiesLoaderUtils;

import com.ebmwebsourcing.webcommons.aop.annotation.CheckAllArgumentsNotNull;
import com.ebmwebsourcing.webcommons.user.api.service.RoleGroupException;
import com.ebmwebsourcing.webcommons.user.api.service.UserException;
import com.ebmwebsourcing.webcommons.user.api.service.UserManager;
import com.ebmwebsourcing.webcommons.user.api.to.UserTO;
import com.ebmwebsourcing.webcommons.user.persistence.bo.RoleGroup;
import com.ebmwebsourcing.webcommons.user.persistence.bo.User;
import com.ebmwebsourcing.webcommons.user.persistence.dao.RoleGroupDAO;
import com.ebmwebsourcing.webcommons.user.persistence.dao.UserDAO;
import com.ebmwebsourcing.webcommons.util.StringHelper;

/**
 * @author strino - eBM WebSourcing
 * 
 */
public class UserManagerImpl implements UserManager {

    private final Logger logger = Logger.getLogger(this.getClass());

    private RolesAndUsersTransfertObjectAssembler rolesAndUsersTransfertObjectAssembler;

    private UserDAO userDAO;

    private RoleGroupDAO roleGroupDAO;

    private Resource usersResource;

    private Resource passwordsResource;

    private UserManager userManager;

    public void nonTransactionalInit() throws IOException, RoleGroupException {
        logger.debug("###### Initialize usermanagerbean");
        userManager.init();
        logger.debug("######  usermanagerbean Initialized");
    }

    // create the user master if not exists in database
    public void init() throws IOException, RoleGroupException {

        // Create user to role group map
        Properties usersProps = PropertiesLoaderUtils.loadProperties(usersResource);
        Map<String, List<String>> usersMap = this.createUserToRoleGroupMap(usersProps);

        // Create user to password map
        Properties passwordsProps = PropertiesLoaderUtils.loadProperties(passwordsResource);

        // retrieve if the user already exist
        for (String userName : usersMap.keySet()) {
            logger.debug("Create user : " + userName);
            createUser(userName, usersMap.get(userName), passwordsProps.getProperty(userName));
        }

    }

    private Map<String, List<String>> createUserToRoleGroupMap(Properties usersProps) {
        Map<String, List<String>> result = new HashMap<String, List<String>>();
        for (Object userProp : usersProps.keySet()) {
            List<String> roleGroupList = new ArrayList<String>();
            String roleGroups = usersProps.getProperty((String) userProp);
            StringTokenizer tokenizer = new StringTokenizer(roleGroups, ",");
            while (tokenizer.hasMoreElements()) {
                String roleGroupName = tokenizer.nextToken();
                roleGroupList.add(roleGroupName.trim());
            }
            result.put((String) userProp, roleGroupList);
        }
        return result;
    }

    private void createUser(String userName, List<String> roleGroupNames, String password)
            throws RoleGroupException {
        User userBO = userDAO.getUserByName(userName);
        if (userBO == null) {

            User user = new User();
            user.setName(userName);
            user.setPassword(password);

            // retrieve the role group related to user
            List<RoleGroup> listRoleGroup = this.retrieveRoleGroups(roleGroupNames);
            user.setListRoleGroup(listRoleGroup);

            // save the user in database
            userDAO.save(user);
        }
    }

    private List<RoleGroup> retrieveRoleGroups(List<String> roleGroupNames)
            throws RoleGroupException {
        List<RoleGroup> roleGroups = new ArrayList<RoleGroup>();
        for (String string : roleGroupNames) {
            RoleGroup roleGroup = this.roleGroupDAO.getRoleGroupByName(string);
            if (roleGroup != null) {
                roleGroups.add(roleGroup);
            } else {
                throw new RoleGroupException("Can't create user because role group '" + string
                        + "' doesn't exist");
            }
        }
        return roleGroups;
    }

    @CheckAllArgumentsNotNull
    public String createUser(UserTO userTO) throws UserException {

        User user = new User();

        this.validateUserBeforSaveOrUpdate(userTO, null);

        this.rolesAndUsersTransfertObjectAssembler.toUserBO(userTO, user);

        return this.userDAO.save(user).getId();

    }

    public List<UserTO> getAllUser() throws UserException {

        List<UserTO> listUserTO = new ArrayList<UserTO>();

        // retrieve all users
        List<User> listUser = this.userDAO.getAll();

        for (User us : listUser) {

            // transform user to UserTO
            UserTO userTO = this.rolesAndUsersTransfertObjectAssembler.toUserTO(us);

            listUserTO.add(userTO);
        }

        return listUserTO;
    }

    @CheckAllArgumentsNotNull
    public List<UserTO> getUserNotInRoleGroup(String idRoleGroup) throws UserException {

        List<UserTO> listUserTO = new ArrayList<UserTO>();

        // retrieve role group with its id
        RoleGroup roleGroup = this.roleGroupDAO.get(idRoleGroup);

        List<User> listUserInRole;

        if (roleGroup != null) {
            listUserInRole = roleGroup.getListUser();
        } else {
            throw new UserException("A role with that name not exists ! ");

        }

        // retrieve all users
        List<User> allUser = this.userDAO.getAll();

        if (allUser != null) {
            if (!listUserInRole.isEmpty()) {

                // remove user from list all users
                allUser.removeAll(listUserInRole);
            }

            for (User us : allUser) {

                // transform users
                listUserTO.add(this.rolesAndUsersTransfertObjectAssembler.toUserTO(us));
            }
        }

        return listUserTO;
    }

    @CheckAllArgumentsNotNull
    public UserTO getUser(String userId) throws UserException {

        // retrieve user with its id
        User user = this.userDAO.get(userId);

        UserTO userTO;

        if (user != null) {
            // transform User to UserTO
            userTO = this.rolesAndUsersTransfertObjectAssembler.toUserTO(user);
        } else {
            throw new UserException("Can not find user in database ");
        }
        return userTO;
    }

    public UserTO getUserByLogin(String login) throws UserException {

        // retrieve user with its name/login
        User user = this.userDAO.getUserByName(login);
        UserTO userTO;

        if (user != null) {

            // transform User to UserTO
            userTO = this.rolesAndUsersTransfertObjectAssembler.toUserTO(user);

        } else {
            throw new UserException("Authentication failed : user '" + login
                    + "' not found in database : ");
        }
        return userTO;
    }

    @CheckAllArgumentsNotNull
    public void removeUser(String userId) throws UserException {

        User user = userDAO.get(userId);

        if (user == null) {
            throw new UserException("Can not find and remove user in database ");
        }

        // remove user by its id
        this.userDAO.remove(user);
    }

    @CheckAllArgumentsNotNull
    public String updateUser(UserTO userTO) throws UserException {

        String id = userTO.getId();

        if (id == null) {
            throw new UserException("This user doesn't exist in database ");
        }

        // retrieve user with its id
        User user = this.userDAO.get(userTO.getId());

        if (user == null) {
            throw new UserException("This user doesn't exist in database ");
        }

        this.validateUserBeforSaveOrUpdate(userTO, user.getName());

        this.rolesAndUsersTransfertObjectAssembler.toUserBO(userTO, user);

        return this.userDAO.save(user).getId();

    }

    @CheckAllArgumentsNotNull
    public void addRoleGroup(String idUser, String idRoleGroup) throws UserException {
        // retrieve user with its id
        User user = this.userDAO.get(idUser);

        if (user == null) {
            throw new UserException("You are trying to add a role to a non existing user with id: "
                    + idUser);
        }

        // retrieve role group with its id
        RoleGroup roleGroup = this.roleGroupDAO.get(idRoleGroup);

        if (roleGroup == null) {
            throw new UserException(
                    "You are trying to add a non existing role to an user. Role id: " + idRoleGroup);
        }

        // insert the new role group in the user's list<RoleGroup>
        user.addRoleGroup(roleGroup);

        // save user
        this.userDAO.save(user);

    }

    @CheckAllArgumentsNotNull
    public void removeRoleGroup(String idUser, String idRoleGroup) throws UserException {
        // retrieve user with its id
        User user = this.userDAO.get(idUser);
        if (user == null) {
            throw new UserException(
                    "You are trying to remove a role to a non existing user with id: " + idUser);
        }

        // retrieve role group with its id
        RoleGroup roleGroup = this.roleGroupDAO.get(idRoleGroup);

        if (roleGroup == null) {
            throw new UserException(
                    "You are trying to remove a non existing role to an user. Role id: "
                            + idRoleGroup);
        }

        // insert the role group in the user's list<RoleGroup>
        user.removeRoleGroup(roleGroup);

        // save user
        this.userDAO.save(user);
    }

    private void validateUserBeforSaveOrUpdate(UserTO userTO, String nameUser) throws UserException {
        String nameUserTO = userTO.getName();
        if (StringHelper.isNullOrEmpty(nameUserTO)
                || StringHelper.isNullOrEmpty(userTO.getPassword())) {
            throw new UserException("User Name and its password must be speficied.");
        }

        if (!nameUserTO.equalsIgnoreCase(nameUser)) {
            User user = this.userDAO.getUserByName(nameUserTO);
            if (user != null) {
                throw new UserException("A user with that name already exists : " + nameUserTO);
            }
        }
    }

    public void setRolesAndUsersTransfertObjectAssembler(
            RolesAndUsersTransfertObjectAssembler rolesAndUsersTransfertObjectAssembler) {
        this.rolesAndUsersTransfertObjectAssembler = rolesAndUsersTransfertObjectAssembler;
    }

    public void setUserDAO(UserDAO userDAO) {
        this.userDAO = userDAO;
    }

    public void setRoleGroupDAO(RoleGroupDAO roleGroupDAO) {
        this.roleGroupDAO = roleGroupDAO;
    }

    public void setUsersResource(Resource usersResource) {
        this.usersResource = usersResource;
    }

    public void setPasswordsResource(Resource passwordsResource) {
        this.passwordsResource = passwordsResource;
    }

    public void setUserManager(UserManager userManager) {
        this.userManager = userManager;
    }

}
