/**
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * $Id$
 * -------------------------------------------------------------------------
 */
package com.ebmwebsourcing.wsstar.notification.service.basenotification.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.Duration;
import javax.xml.namespace.QName;

import com.ebmwebsourcing.wsstar.addressing.definition.WSAddressingFactory;
import com.ebmwebsourcing.wsstar.addressing.definition.api.EndpointReferenceType;
import com.ebmwebsourcing.wsstar.addressing.definition.api.ReferenceParametersType;
import com.ebmwebsourcing.wsstar.addressing.definition.api.WSAddressingException;
import com.ebmwebsourcing.wsstar.notification.definition.WSNotificationFactory;
import com.ebmwebsourcing.wsstar.notification.definition.basenotification.api.FilterType;
import com.ebmwebsourcing.wsstar.notification.definition.basenotification.api.Renew;
import com.ebmwebsourcing.wsstar.notification.definition.basenotification.api.RenewResponse;
import com.ebmwebsourcing.wsstar.notification.definition.basenotification.api.Subscribe;
import com.ebmwebsourcing.wsstar.notification.definition.basenotification.api.SubscriptionManagerRP;
import com.ebmwebsourcing.wsstar.notification.definition.basenotification.api.SubscriptionPolicyType;
import com.ebmwebsourcing.wsstar.notification.definition.basenotification.api.TopicExpressionType;
import com.ebmwebsourcing.wsstar.notification.definition.basenotification.api.Unsubscribe;
import com.ebmwebsourcing.wsstar.notification.definition.basenotification.api.UnsubscribeResponse;
import com.ebmwebsourcing.wsstar.notification.definition.utils.WSNotificationException;
import com.ebmwebsourcing.wsstar.notification.extension.WSNotificationExtensionFactory;
import com.ebmwebsourcing.wsstar.notification.extension.api.ResourcesUuidType;
import com.ebmwebsourcing.wsstar.notification.extension.api.SOAParameterType;
import com.ebmwebsourcing.wsstar.notification.extension.api.TerminationTimeType;
import com.ebmwebsourcing.wsstar.notification.extension.utils.WSNotificationExtensionException;
import com.ebmwebsourcing.wsstar.notification.extension.utils.WsnSpecificTypeHelper;
import com.ebmwebsourcing.wsstar.notification.service.basenotification.WsnbSubscriptionManager;
import com.ebmwebsourcing.wsstar.notification.service.fault.ResourceNotDestroyedFault;
import com.ebmwebsourcing.wsstar.notification.service.fault.ResourceUnknownFault;
import com.ebmwebsourcing.wsstar.notification.service.fault.UnacceptableInitialTerminationTimeFault;
import com.ebmwebsourcing.wsstar.notification.service.fault.WSNotificationFault;
import com.ebmwebsourcing.wsstar.notification.service.fault.WsnFaultMessageConstants;
import com.ebmwebsourcing.wsstar.notification.service.persistence.WsnPersistence;
import com.ebmwebsourcing.wsstar.notification.service.topic.WstopTopicManager;
import com.ebmwebsourcing.wsstar.notification.service.util.WSNotificationNotImplementedException;
import com.ebmwebsourcing.wsstar.notification.service.util.WSNotificationTopicManagerBadUsageException;

public class SubscriptionManagerMgr implements WsnbSubscriptionManager {
	
	protected Logger logger;
	protected Map<String, SubscriptionManagerRP> subscriptions;
	protected WstopTopicManager topicsMgr;
	protected WsnPersistence persistenceMgr;
		
	protected String subscriptionsManagerEdp = "http://www.ebmwebsourcing.com/subscriptionManager/default";
	protected QName subscriptionsManagerService = new QName("http://www.ebmwebsourcing.com/default","SubscriptionManagerService");
	protected QName subscriptionsManagerInterface = new QName("http://www.ebmwebsourcing.com/default","SubscriptionManager");	
	
	public SubscriptionManagerMgr(Logger logger, WstopTopicManager topicsMgr, WsnPersistence persistenceMgr) {
		super();
		this.logger = logger;
		this.subscriptions = Collections.synchronizedMap(new HashMap<String, SubscriptionManagerRP>());
		this.topicsMgr = topicsMgr;
		this.persistenceMgr = persistenceMgr ;
		
	}
	
	public SubscriptionManagerMgr(Logger logger, WstopTopicManager topicsMgr) {
		super();
		this.logger = logger;
		this.subscriptions = Collections.synchronizedMap(new HashMap<String, SubscriptionManagerRP>());
		this.topicsMgr = topicsMgr;
		this.persistenceMgr = null;
		
	}
	
	public SubscriptionManagerMgr(Logger logger) {
		super();
		this.logger = logger;
		this.subscriptions = Collections.synchronizedMap(new HashMap<String, SubscriptionManagerRP>());
		this.persistenceMgr = null;
		
	}
	// ##################################################
	//  		----- {Getter,Setter} methods ------
	// ##################################################
	
	/**
	 * 
	 * @return
	 */
	public String getSubscriptionsManagerEdp() {
		return subscriptionsManagerEdp;
	}
	
	public void setSubscriptionsManagerEdp(String subscriptionsMgrEdp) {
		this.subscriptionsManagerEdp = subscriptionsMgrEdp;
	}
	
	public QName getSubscriptionsManagerService() {
		return subscriptionsManagerService;
	}
	
	public void setSubscriptionsManagerService(QName subscriptionsMgrService) {
		this.subscriptionsManagerService = subscriptionsMgrService;
	}
	
	public QName getSubscriptionsManagerInterface() {
		return subscriptionsManagerInterface;
	}
	
	public void setSubscriptionsManagerInterface(QName subscriptionsMgrInterface) {
		this.subscriptionsManagerInterface = subscriptionsMgrInterface;
	}
	
	public List<String> getStoredSubscriptionIds(){
		return new ArrayList<String>(this.subscriptions.keySet());
	}
	
	public WstopTopicManager getTopicsMgr() {
		return topicsMgr;
	}
	
	public void setTopicsMgr(WstopTopicManager topicsMgr) {
		this.topicsMgr = topicsMgr;
	}
	
	// #####################################################################################
	// 	----- Methods' implementation of WS-Notification SubscriptionManager Interface ----
	// #####################################################################################
	
	/*
	 * (non-Javadoc)
	 * @see com.ebmwebsourcing.wsstar.notification.service.basenotification.WsnbSubscriptionManager#unsubscribe(com.ebmwebsourcing.wsstar.notification.definition.basenotification.api.Unsubscribe)
	 */
	public UnsubscribeResponse unsubscribe(Unsubscribe request)
			throws WSNotificationException, WSNotificationFault, WSNotificationExtensionException {
		logger.log(Level.FINE, "performs a \"Unsubscribe\" request ...");
		//TODO : implement this method !!!
		logger.log(Level.FINE, "\"Unsubscribe\"  method implementation in progress ... " );
		if (this.topicsMgr == null)
			throw new WSNotificationTopicManagerBadUsageException(this.getClass().getSimpleName(),"unsubscribe()");

		UnsubscribeResponse response = null;
		
		// ---- perform a subscription destruction ---

		ResourcesUuidType uuids = WsnSpecificTypeHelper.getResourcesUuidType(request);
		if (uuids != null){
			List<String> resourcesId = uuids.getUuids();
			if (resourcesId != null){
				for (String rId : resourcesId) {										
					if  (subscriptions.containsKey(rId)){
						subscriptions.remove(rId);
						if (subscriptions.containsKey(rId))
							throw new ResourceNotDestroyedFault(WsnFaultMessageConstants.FAULT_DESCRIPTION_LANGUAGE,
									WsnFaultMessageConstants.WsnbUnsubcribeFaultDescriptions.UNABLE_TO_DESTROY_SUBSCRIPTION_FAULT_DESC);
						else							
							this.topicsMgr.removeExistingSubscription(rId);

						if (this.persistenceMgr != null)
							this.persistenceMgr.removeSubscription(rId);

					} else 
						throw new ResourceUnknownFault(WsnFaultMessageConstants.FAULT_DESCRIPTION_LANGUAGE,
								WsnFaultMessageConstants.WsnbUnsubcribeFaultDescriptions.RESOURCE_UNKNOWN_FAULT_DESC);
				}
			}
		}

		response = WSNotificationFactory.getInstance().createUnsubscribeResponse();
		return response;
	}
	
	public RenewResponse renew(Renew request) throws WSNotificationException,
			WSNotificationFault, WSNotificationExtensionException {	
		logger.log(Level.FINE, "performs a \"Renew\" request ...");
		//TODO : implement this method !!!
		// throw new WSNotificationNotImplementedException(this.getClass().getName(),"Renew");
		
		RenewResponse response = null;
		
		ResourcesUuidType uuids = WsnSpecificTypeHelper.getResourcesUuidType(request);
		if (uuids != null) {
			String ruuid = uuids.getUuids().get(0);
			
			SubscriptionManagerRP subscriptionRP = this.subscriptions.get(ruuid);
			
			if (subscriptionRP == null)
				throw new ResourceUnknownFault(WsnFaultMessageConstants.FAULT_DESCRIPTION_LANGUAGE,
					WsnFaultMessageConstants.WsnbRenewFaultDescriptions.RESOURCE_UNKNOWN_FAULT_DESC);

			TerminationTimeType newTerminationTime = WSNotificationExtensionFactory.getInstance().createTerminationTimeType();
			
			String newTerminationTimeVal = request.getTerminationTime();
			
			Date timeVar = new GregorianCalendar().getTime(); 
			response = WSNotificationFactory.getInstance().createRenewResponse(); 
			response.setCurrentTime(timeVar);
			try {
				if (newTerminationTimeVal.startsWith("P")){
					DatatypeFactory.newInstance().newDuration(newTerminationTimeVal).addTo(timeVar);
				} else {
					timeVar = (DatatypeFactory.newInstance().newXMLGregorianCalendar(newTerminationTimeVal)).toGregorianCalendar().getTime();
				}
			} catch (DatatypeConfigurationException e) {					
				throw new WSNotificationException(e);
			}	
			newTerminationTime.setValue(timeVar);		
			WsnSpecificTypeHelper.setTerminationTimeToSubscription(newTerminationTime, subscriptionRP);
			
			response.setTerminationTime(timeVar);
			
			if (this.persistenceMgr != null)			
				this.persistenceMgr.persist(subscriptionRP, ruuid);
		}
		
		return response;
	}
	
	// ###############################################
	// 			---- Others methods ----
	// ###############################################
	 	
	
	public EndpointReferenceType createAndStoreSubscriptionResource(String subscriptionId,Subscribe payload) 
		throws WSNotificationException, WSAddressingException, WSNotificationFault ,WSNotificationExtensionException{
		if (this.topicsMgr == null)
			throw new WSNotificationTopicManagerBadUsageException(this.getClass().getSimpleName(),"createAndStoreSubscriptionResource()");

		EndpointReferenceType subsRef = null;
		
		SubscriptionManagerRP subscription = null;
		// ------- Generate UUID as endpoint reference for Subscription :
		subsRef = WSAddressingFactory.getInstance().newEndpointReferenceType();
		
		// use @http of consumerReferenceRequest //EndpointParameterType.DEFAULT_PETALS_ADDRESS
		subsRef.setAddress(payload.getConsumerReference().getAddress());		

		ReferenceParametersType ref = subsRef.newReferenceParameters();
		subsRef.setReferenceParameters(ref);
		
		SOAParameterType soaParam = WSNotificationExtensionFactory.getInstance().createSOAParameterType();
		                                    
		soaParam.setService(this.subscriptionsManagerService);
		soaParam.setInterface(this.subscriptionsManagerInterface);
		soaParam.setEndpoint(this.subscriptionsManagerEdp);
		
		WsnSpecificTypeHelper.setSOAParameter(soaParam, ref);
		
		//endpointParam.addResourceUuid(subscriptionId);
		ResourcesUuidType rUuids = WSNotificationExtensionFactory.getInstance().createResourcesUuidType();
		rUuids.addUuid(subscriptionId);
		WsnSpecificTypeHelper.setResourcesUuidType(rUuids, ref);
		
		// -- Create Subscription resource to manage 
		subscription = WSNotificationFactory.getInstance().createSubscriptionManagerRP();
		subscription.setConsumerReference(payload.getConsumerReference());
		// --- to test ---
		FilterType ft = payload.getFilter();
		// ---------------
		subscription.setFilter(ft);
		Date now = (new GregorianCalendar()).getTime();
		subscription.setCreationTime(now);
		subscription.setSubscriptionPolicy(payload.getSubscriptionPolicy());
		
		// -------- Manage Termination time ------------												
		try {
			String initTermTime = payload.getInitialTerminationTime();
			if ((initTermTime != null) && (initTermTime.length() > 0)){
				TerminationTimeType termTime = WSNotificationExtensionFactory.getInstance().createTerminationTimeType();

				Date termTimeValue = null;

				if (initTermTime.startsWith("P")){
					Duration duration = DatatypeFactory.newInstance().newDuration(initTermTime);
					termTimeValue = new GregorianCalendar().getTime();
					duration.addTo(termTimeValue);				
				} else {				
					termTimeValue = (DatatypeFactory.newInstance().newXMLGregorianCalendar(initTermTime)).toGregorianCalendar().getTime(); 												
				}

				if (!termTimeValue.after(now))
					throw new UnacceptableInitialTerminationTimeFault(WsnFaultMessageConstants.FAULT_DESCRIPTION_LANGUAGE,
							WsnFaultMessageConstants.WsnbSubcribeFaultDescriptions.UNACCEPTABLE_INITIAL_TERMINATION_TIME_FAULT_DESC);

				termTime.setValue(termTimeValue);
				WsnSpecificTypeHelper.setTerminationTimeToSubscription(termTime, subscription);
			}
		} catch (DatatypeConfigurationException e) {
			throw new WSNotificationExtensionException(e);
		}
		// --------------------------------------------
		
		this.subscriptions.put(subscriptionId, subscription);
		
		if (this.persistenceMgr != null){			
			this.persistenceMgr.persist(subscription, subscriptionId);
					
		}
		return subsRef;
	}
	
	public EndpointReferenceType getConsumerEdpRefOfSubscription(String subscriptionId) throws WSNotificationException{
		if (this.topicsMgr == null)
			throw new WSNotificationTopicManagerBadUsageException(this.getClass().getSimpleName(),"getConsumerEdpRefOfSubscription()");

		EndpointReferenceType consumerEdp = null;
		SubscriptionManagerRP subsManResource = this.subscriptions.get(subscriptionId);
        if (subsManResource != null)        
		consumerEdp = subsManResource.getConsumerReference();
		return consumerEdp;
	}
	
	public TopicExpressionType getTopicExpressionOfSubscription(String subscriptionId) throws WSNotificationException{
		if (this.topicsMgr == null)
			throw new WSNotificationTopicManagerBadUsageException(this.getClass().getSimpleName(),"getTopicExpressionOfSubscription()");

		TopicExpressionType topExpr = null;
		SubscriptionManagerRP subsManResource = this.subscriptions.get(subscriptionId);
		if (subsManResource != null)
			topExpr = subsManResource.getFilter().getTopicExpression();
		return topExpr;
	}
	
	public FilterType getFilterOfSubscription(String subscriptionId) throws WSNotificationException{

		if (this.topicsMgr == null)
			throw new WSNotificationTopicManagerBadUsageException(this.getClass().getSimpleName(),"getFilterOfSubscription()");

		FilterType filter = null;
		
		SubscriptionManagerRP subsManResource = this.subscriptions.get(subscriptionId);
		if (subsManResource != null)
			filter = subsManResource.getFilter();
		return filter;
	}
	
	public SubscriptionPolicyType getPolicyOfSubscription(String subscriptionId) throws WSNotificationException {
		if (this.topicsMgr == null)
			throw new WSNotificationTopicManagerBadUsageException(this.getClass().getSimpleName(),"getPolicyOfSubscription()");

		SubscriptionPolicyType policy = null;
		SubscriptionManagerRP subsManResource = this.subscriptions.get(subscriptionId);
        if (subsManResource != null)
        	policy = subsManResource.getSubscriptionPolicy();
		return policy;
		
	}
		
	public TerminationTimeType getTerminationTimeOfSubscription(String subscriptionId) throws WSNotificationException{	
		if (this.topicsMgr == null)
			throw new WSNotificationTopicManagerBadUsageException(this.getClass().getSimpleName(),"getTerminationTimeOfSubscription()");

		TerminationTimeType terminationTime = null;
		SubscriptionManagerRP subsManRP = this.subscriptions.get(subscriptionId);
		if (subsManRP != null)
			try {
				terminationTime = WsnSpecificTypeHelper.getTerminationTimeFromSubscription(subsManRP);
			} catch (WSNotificationExtensionException e) {
				throw new WSNotificationException(e);
			}		
		return terminationTime;
	}
	
	public void removeExpiredSubscription(String subscriptionId) throws WSNotificationException{
		if (this.topicsMgr == null)
			throw new WSNotificationTopicManagerBadUsageException(this.getClass().getSimpleName(),"removeExpiredSubscription()");
				
		if  (subscriptions.containsKey(subscriptionId)){
			
			subscriptions.remove(subscriptionId);							
			try {
				this.topicsMgr.removeExistingSubscription(subscriptionId);
			} catch (WSNotificationFault e) {
				throw new WSNotificationException(e);
			}
			if (this.persistenceMgr != null)
				this.persistenceMgr.removeSubscription(subscriptionId);
		} 
	}
	
	public void restorePersistedSubscriptions() throws WSNotificationException, WSNotificationFault{		

		if (this.topicsMgr == null)
			throw new WSNotificationTopicManagerBadUsageException(this.getClass().getSimpleName(),"restorePersistedSubscriptions()");

		if (this.persistenceMgr != null){
			Map<String, SubscriptionManagerRP> subsRPToRestore = this.persistenceMgr.getSubscriptionsToRestore();

			Set<String> rIds = subsRPToRestore.keySet();
			SubscriptionManagerRP currentSubsRP = null;
			for (String idItem : rIds) {			
				currentSubsRP = subsRPToRestore.get(idItem);
				this.subscriptions.put(idItem, currentSubsRP);
				this.topicsMgr.storeNewSubscription(currentSubsRP.getFilter().getTopicExpression(), idItem);				
			}						
		}		
	}
	
	
	public Map<String,Subscribe> getSubscriptionsRPAsSubscribePayoad() throws WSNotificationException{
		Map<String,Subscribe> result = new HashMap<String, Subscribe>();
				
		 
		SubscriptionManagerRP currentSubscriptionManagerRP = null;

		for (Map.Entry<String, SubscriptionManagerRP> mapPairItem : this.subscriptions.entrySet()) {			
			
			currentSubscriptionManagerRP = mapPairItem.getValue();

			Subscribe currentSubscribePayload = WSNotificationFactory.getInstance().createSubscribe();
		
			currentSubscribePayload.setConsumerReference(currentSubscriptionManagerRP.getConsumerReference());
			currentSubscribePayload.setFilter(currentSubscriptionManagerRP.getFilter());
			currentSubscribePayload.setSubscriptionPolicy(currentSubscriptionManagerRP.getSubscriptionPolicy());
			
			// TODO : made followin code available when life time will be implemented (in ver. 1.0-SNAPSHOT)
			//currentSubscribePayload.setInitialTerminationTime(currentSubscriptionManagerRP.getTerminationTime());
			
			result.put(mapPairItem.getKey(), currentSubscribePayload);	
		}
		
		return result;
	}
	
}


