/**
 * EasierSBS project - Java file
 * Copyright (C) 2011 EBM WebSourcing - Petals Link
 * 
 * EasySWS is free project: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * EasySWS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this program.
 * If not, see <http://www.gnu.org/licenses/lgpl-3.0.txt>.	
 * 
 */ 
package com.petalslink.easiersbs.reasoner.impl.engine;

import java.net.URI;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.semanticweb.owlapi.model.IRI;
import org.semanticweb.owlapi.model.OWLAxiom;
import org.semanticweb.owlapi.model.OWLClass;
import org.semanticweb.owlapi.model.OWLDataFactory;
import org.semanticweb.owlapi.model.OWLNamedIndividual;
import org.semanticweb.owlapi.model.OWLNamedObject;
import org.semanticweb.owlapi.model.OWLOntology;
import org.semanticweb.owlapi.model.OWLOntologyManager;
import org.semanticweb.owlapi.reasoner.ConsoleProgressMonitor;
import org.semanticweb.owlapi.reasoner.InferenceType;
import org.semanticweb.owlapi.reasoner.OWLReasoner;
import org.semanticweb.owlapi.reasoner.OWLReasonerConfiguration;
import org.semanticweb.owlapi.reasoner.OWLReasonerFactory;
import org.semanticweb.owlapi.reasoner.SimpleConfiguration;
import org.semanticweb.owlapi.util.InferredAxiomGenerator;
import org.semanticweb.owlapi.util.InferredOntologyGenerator;
import org.semanticweb.owlapi.util.InferredSubClassAxiomGenerator;

import uk.ac.manchester.cs.factplusplus.owlapiv3.FaCTPlusPlusReasonerFactory;

import com.petalslink.easiersbs.reasoner.api.ReasonerException;
import com.petalslink.easiersbs.reasoner.api.engine.Reasoner;
import com.petalslink.easiersbs.reasoner.api.ontology.OntologyManager;
import com.petalslink.easiersbs.reasoner.impl.ontology.OntologyManagerImpl;

/**
 * @author Nicolas Boissel-Dallier - Petals Link
 */
public class ReasonerImpl implements Reasoner {
	
	// Just change the OWLReasonerFactory to change HermiT to another one.
	private OWLReasonerFactory reasonerFactory = new FaCTPlusPlusReasonerFactory();
	
	private OntologyManager easierOntoManager = null;
	private OWLOntologyManager owlapiManager = null;
	private OWLDataFactory owlapiDataFactory = null;
	private OWLOntology onto = null;
	private OWLReasoner reasoner = null;

	public ReasonerImpl(OntologyManager manager) throws ReasonerException{
		easierOntoManager = manager;
		owlapiManager = ((OntologyManagerImpl)manager).getOntologyManager();
		owlapiDataFactory = owlapiManager.getOWLDataFactory();
		onto = ((OntologyManagerImpl)manager).getMergedOntology();
		
		//OWLAPI Reasoner 
		ConsoleProgressMonitor progressMonitor = new ConsoleProgressMonitor();
		OWLReasonerConfiguration config = new SimpleConfiguration(progressMonitor); 
		reasoner = reasonerFactory.createReasoner(onto, config);
		
		reasoner.precomputeInferences(InferenceType.CLASS_HIERARCHY);
		
		if(!reasoner.isConsistent()){
			throw new ReasonerException("Inconsistent ontology" + onto.getOntologyID().toString());
		}
	}
	
	public Set<URI> getEquivalentClasses(URI conceptURI) {
		OWLClass conceptClass = owlapiDataFactory.getOWLClass(IRI.create(conceptURI));
		Set<OWLClass> eqClasses = reasoner.getEquivalentClasses(conceptClass).getEntities();
		return this.getURIs(eqClasses);
	}
	
	public Set<URI> getSuperClasses(URI concept) {
		OWLClass conceptClass = owlapiDataFactory.getOWLClass(IRI.create(concept));
		Set<OWLClass> superClasses = reasoner.getSuperClasses(conceptClass, true).getFlattened();
		return this.getURIs(superClasses);
	}
	
	public Set<URI> getSubClasses(URI concept) {
		OWLClass conceptClass = owlapiDataFactory.getOWLClass(IRI.create(concept));
		Set<OWLClass> subClasses = reasoner.getSubClasses(conceptClass, true).getFlattened();
		return this.getURIs(subClasses);
	}

	public Set<URI> getIndividuals(URI concept) {
		OWLClass conceptClass = owlapiDataFactory.getOWLClass(IRI.create(concept));
		Set<OWLNamedIndividual> individuals = reasoner.getInstances(conceptClass, true).getFlattened();
		return this.getURIs(individuals);
	}

	
	public Set<URI> getSimilarIndividuals(URI individualURI) {
		OWLNamedIndividual individual = owlapiDataFactory.getOWLNamedIndividual(IRI.create(individualURI));
		Set<OWLClass> types = reasoner.getTypes(individual, true).getFlattened();
		Set<URI> brothers = new HashSet<URI>();
		for(OWLClass parent : types){
			brothers.addAll(this.getURIs(reasoner.getInstances(parent, true).getFlattened()));
		}
		return brothers;
	}
	
	public Set<URI> getTypes(URI individualURI) {
		OWLNamedIndividual individual = owlapiDataFactory.getOWLNamedIndividual(IRI.create(individualURI));
		Set<OWLClass> types = reasoner.getTypes(individual, true).getFlattened();
		return this.getURIs(types);
	}
	
	public void infer() {
		List<InferredAxiomGenerator<? extends OWLAxiom>> gens = new ArrayList<InferredAxiomGenerator<? extends OWLAxiom>>();
		gens.add(new InferredSubClassAxiomGenerator());
		InferredOntologyGenerator iog = new InferredOntologyGenerator(reasoner, gens);
		iog.fillOntology(owlapiManager, onto);
	}

	public boolean isConsistent() {
		return reasoner.isConsistent();
	}
	
	public boolean isInstance(URI uri){
		OWLNamedIndividual indiv = owlapiDataFactory.getOWLNamedIndividual(IRI.create(uri));
		return onto.isDeclared(indiv);
	}
	
	public boolean isConcept(URI uri) {
		OWLClass conceptClass = owlapiDataFactory.getOWLClass(IRI.create(uri));
		//return conceptClass.isDefined(onto);
		return onto.isDeclared(conceptClass);
	}
	
	public boolean isThingOrNothing(URI uri){
		OWLClass conceptClass = owlapiDataFactory.getOWLClass(IRI.create(uri));
		return (conceptClass.isOWLThing() || conceptClass.isOWLNothing());
	}
	
	// Extract URIs from Set of OWLAPI OWLObjects
	private Set<URI> getURIs(Set<? extends OWLNamedObject> owlObjects) {
		Set<URI> uris = new HashSet<URI>();
		Iterator<? extends OWLNamedObject> it = owlObjects.iterator();
		while(it.hasNext()){
			uris.add(it.next().getIRI().toURI());
		}
		return uris;
	}
	
	public OWLOntology getReasonerOntology(){
		return onto;
	}
	
	public boolean isTechnicalConcept(URI concept){
		String conceptBase = concept.toString().split("#")[0];
		URI conceptBaseUri = URI.create(conceptBase);
		return easierOntoManager.isTechnicalOntology(conceptBaseUri);
	}
	
	
}
