
package com.ebmwebsourcing.easycommons.stream;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.StringWriter;

import com.ebmwebsourcing.easycommons.io.IOHelper;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;

/**
 * @author ofabre
 * 
 */
public class StreamHelper {

    /**
     * Default value is 2048.
     */
    public static final int DEFAULT_BUFFER_SIZE = 2048;

    /**
     * Copies information from the input stream to the output stream using a
     * default buffer size of 2048 bytes.
     * 
     * @param fromStream
     *            the {@link InputStream} to read
     * @param toStream
     *            the {@link OutputStream} to write
     */
    public static void copy(InputStream fromStream, OutputStream toStream) {
        copy(fromStream, toStream, DEFAULT_BUFFER_SIZE);
    }

    /**
     * Copies information from the input stream to the output stream using the
     * specified buffer size
     * 
     * @param fromStream
     *            the {@link InputStream} to read
     * @param toStream
     *            the {@link OutputStream} to write
     * @param bufferSize
     *            size of read blocks
     */
    public static void copy(InputStream fromStream, OutputStream toStream, int bufferSize) {
        assert fromStream != null;
        assert toStream != null;
        assert bufferSize > 0;
        try {
            byte[] buf = new byte[bufferSize];
            int bytesRead = fromStream.read(buf);
            while (bytesRead != -1) {
                toStream.write(buf, 0, bytesRead);
                bytesRead = fromStream.read(buf);
            }
            toStream.flush();
        } catch (IOException e) {
            throw new UncheckedException("Cannot read or write bytes on given streams", e);
        }
    }

    /**
     * @returns a byte[] containing the information contained in the specified
     *          InputStream.
     */
    public static byte[] getBytes(InputStream input) {
        ByteArrayOutputStream result = new ByteArrayOutputStream();
        try {
            copy(input, result);
        } finally {
            IOHelper.close(result);
        }
        return result.toByteArray();
    }

    /**
     * Return the {@link String} representation of an {@link InputStream}
     * <p>
     * NOTE:</br> At the end, the {@link InputStream} is closed.
     * </p>
     * 
     * @param is
     * @return Return the {@link String} representation of an
     *         {@link InputStream}
     * @throws IOException
     */
    public static String getString(InputStream is) throws IOException {
        StringWriter w = null;
        try {
            BufferedReader reader = new BufferedReader(new InputStreamReader(is));
            w = new StringWriter();
            char[] buffer = new char[1024];
            int n;
            while ((n = reader.read(buffer)) != -1) {
                w.write(buffer, 0, n);
            }
        } finally {
            is.close();
        }
        return w.toString();
    }
}
