/****************************************************************************
 * Copyright (c) 2010-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easycommons.stream;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.StringWriter;

import com.ebmwebsourcing.easycommons.io.IOHelper;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;

/**
 * @author ofabre
 * 
 */
public class StreamHelper {

    /**
     * Default value is 2048.
     */
    public static final int DEFAULT_BUFFER_SIZE = 2048;

    /**
     * Copies information from the input stream to the output stream using a
     * default buffer size of 2048 bytes.
     * 
     * @param fromStream
     *            the {@link InputStream} to read
     * @param toStream
     *            the {@link OutputStream} to write
     */
    public static void copy(InputStream fromStream, OutputStream toStream) {
        copy(fromStream, toStream, DEFAULT_BUFFER_SIZE);
    }

    /**
     * Copies information from the input stream to the output stream using the
     * specified buffer size
     * 
     * @param fromStream
     *            the {@link InputStream} to read
     * @param toStream
     *            the {@link OutputStream} to write
     * @param bufferSize
     *            size of read blocks
     */
    public static void copy(InputStream fromStream, OutputStream toStream, int bufferSize) {
        assert fromStream != null;
        assert toStream != null;
        assert bufferSize > 0;
        try {
            byte[] buf = new byte[bufferSize];
            int bytesRead = fromStream.read(buf);
            while (bytesRead != -1) {
                toStream.write(buf, 0, bytesRead);
                bytesRead = fromStream.read(buf);
            }
            toStream.flush();
        } catch (IOException e) {
            throw new UncheckedException("Cannot read or write bytes on given streams", e);
        }
    }

    /**
     * @returns a byte[] containing the information contained in the specified
     *          InputStream.
     */
    public static byte[] getBytes(InputStream input) {
        ByteArrayOutputStream result = new ByteArrayOutputStream();
        try {
            copy(input, result);
        } finally {
            IOHelper.close(result);
        }
        return result.toByteArray();
    }

    /**
     * Return the {@link String} representation of an {@link InputStream}
     * <p>
     * NOTE:</br> At the end, the {@link InputStream} is closed.
     * </p>
     * 
     * @param is
     * @return Return the {@link String} representation of an
     *         {@link InputStream}
     * @throws IOException
     */
    public static String getString(InputStream is) throws IOException {
        StringWriter w = null;
        try {
            BufferedReader reader = new BufferedReader(new InputStreamReader(is));
            w = new StringWriter();
            char[] buffer = new char[1024];
            int n;
            while ((n = reader.read(buffer)) != -1) {
                w.write(buffer, 0, n);
            }
        } finally {
            is.close();
        }
        return w.toString();
    }
}
