/****************************************************************************
 * Copyright (c) 2009-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easyschema10.impl;

import java.math.BigInteger;
import java.util.Arrays;

import javax.xml.namespace.QName;

import com.ebmwebsourcing.easybox.api.XmlContext;
import com.ebmwebsourcing.easybox.impl.JaxbDuplicator;
import com.ebmwebsourcing.easyschema10.api.Constants;
import com.ebmwebsourcing.easyschema10.api.SchemaHelper;
import com.ebmwebsourcing.easyschema10.api.element.ComplexType;
import com.ebmwebsourcing.easyschema10.api.element.Element;
import com.ebmwebsourcing.easyschema10.api.element.Schema;
import com.ebmwebsourcing.easyschema10.api.element.SimpleType;
import com.ebmwebsourcing.easyschema10.api.type.Form;
import com.ebmwebsourcing.easyschema10.api.type.Type;

import easybox.org.w3._2001.xmlschema.EJaxbElement;
import easybox.org.w3._2001.xmlschema.EJaxbFormChoice;
import easybox.org.w3._2001.xmlschema.EJaxbLocalComplexType;
import easybox.org.w3._2001.xmlschema.EJaxbLocalElement;
import easybox.org.w3._2001.xmlschema.EJaxbLocalSimpleType;
import easybox.org.w3._2001.xmlschema.EJaxbTopLevelElement;

final class ElementImpl extends AbstractAnnotatedImpl<EJaxbElement> implements
        com.ebmwebsourcing.easyschema10.api.element.Element {

    protected ElementImpl(XmlContext xmlContext, EJaxbElement jaxbModel) {
        super(xmlContext, jaxbModel);
    }

    
    @Override
    protected Class<? extends EJaxbElement> getCompliantModelClass() {
    	if (getXmlObjectParent() == null) {
    		return EJaxbTopLevelElement.class;
    	} else if (getXmlObjectParent() instanceof Schema) {
    		return EJaxbTopLevelElement.class;
    	} else {
    		return EJaxbLocalElement.class;
    	}
    }

    
    @Override
    public String[] getBlock() {
        if (!hasBlock())
            return null;
        return getModelObject().getBlock().toArray(
                new String[getModelObject().getBlock().size()]);
    }

    @Override
    public ComplexType getComplexType() {
        if (getModelObject().getComplexType() == null)
            return null;
        return getXmlContext().getXmlObjectFactory().wrap(getModelObject().getComplexType(),
                ComplexTypeImpl.class);
    }

    @Override
    public String getDefault() {
        return getModelObject().getDefault();
    }

    @Override
    public String[] getFinal() {
        if (!hasFinal())
            return null;
        return getModelObject().getFinal().toArray(
                new String[getModelObject().getFinal().size()]);
    }

    @Override
    public Form getForm() {
        if (getModelObject().getForm() == null)
            return null;
        return Form.valueOf(getModelObject().getForm().toString().toUpperCase());
    }

    @Override
    public int getMaxOccurs() {
        if (Constants.UNBOUNDED_STRING.equalsIgnoreCase(getModelObject()
                .getMaxOccurs()))
            return Constants.UNBOUNDED;
        return Integer.parseInt(getModelObject().getMaxOccurs());
    }

    @Override
    public int getMinOccurs() {
        return getModelObject().getMinOccurs().intValue();
    }

    @Override
    public String getName() {
        return getModelObject().getName();
    }

    @Override
    public QName getRef() {
        return getModelObject().getRef();
    }

    @Override
    public SimpleType getSimpleType() {
        if (getModelObject().getSimpleType() == null)
            return null;
        return getXmlContext().getXmlObjectFactory().wrap(getModelObject().getSimpleType(),
                SimpleTypeImpl.class);
    }

    @Override
    public QName getType() {
        return getModelObject().getType();
    }

    @Override
    public boolean hasBlock() {
        return getModelObject().isSetBlock();
    }

    @Override
    public boolean hasComplexType() {
        return getModelObject().getComplexType() != null;
    }

    @Override
    public boolean hasDefault() {
        return getModelObject().getDefault() != null;
    }

    @Override
    public boolean hasFinal() {
        return getModelObject().isSetFinal();
    }

    @Override
    public boolean hasForm() {
        return getModelObject().getForm() != null;
    }

    @Override
    public boolean hasName() {
        return getModelObject().getName() != null;
    }

    @Override
    public boolean hasRef() {
        return getModelObject().getRef() != null;
    }

    @Override
    public boolean hasSimpleType() {
        return getModelObject().getSimpleType() != null;
    }

    @Override
    public boolean hasType() {
        return getModelObject().getType() != null;
    }

    @Override
    public boolean isAbstract() {
        return getModelObject().isAbstract();
    }

    @Override
    public boolean isNillable() {
        return getModelObject().isNillable();
    }

    @Override
    public void setAbstract(boolean value) {
        getModelObject().setAbstract(value);
    }

    @Override
    public void setBlock(String[] blockStrings) {
        getModelObject().unsetBlock();
        if (blockStrings != null) {
            getModelObject().getBlock().addAll(Arrays.asList(blockStrings));
        }
    }

    @Override
    public void setComplexType(ComplexType complexType) {
        if (complexType == null) {
            getModelObject().setComplexType(null);
            return;
        }
        assert complexType instanceof ComplexTypeImpl;
        ComplexTypeImpl complexTypeImpl = (ComplexTypeImpl) complexType;
        getModelObject().setComplexType(
                JaxbDuplicator.duplicateAs(complexTypeImpl.getModelObject(), EJaxbLocalComplexType.class));
    }

    @Override
    public void setDefault(String value) {
        getModelObject().setDefault(value);
    }

    @Override
    public void setFinal(String[] finalStrings) {
        getModelObject().unsetFinal();
        if (finalStrings != null) {
            getModelObject().getFinal().addAll(Arrays.asList(finalStrings));
        }
    }

    @Override
    public void setForm(Form form) {
        if (form == null) {
            getModelObject().setForm(null);
        } else if (form.equals(Form.QUALIFIED)) {
            getModelObject().setForm(EJaxbFormChoice.QUALIFIED);
        } else {
            getModelObject().setForm(EJaxbFormChoice.UNQUALIFIED);
        }
    }

    @Override
    public void setMaxOccurs(int maxOccurs) {
        if (maxOccurs == Constants.UNBOUNDED) {
            getModelObject().setMaxOccurs(Constants.UNBOUNDED_STRING);
        } else {
            getModelObject().setMaxOccurs(String.valueOf(maxOccurs));
        }
    }

    @Override
    public void setMinOccurs(int minOccurs) {
        getModelObject().setMinOccurs(BigInteger.valueOf(minOccurs));
    }

    @Override
    public void setName(String name) {
        getModelObject().setName(name);
    }

    @Override
    public void setNillable(boolean nillable) {
        getModelObject().setNillable(nillable);

    }

    @Override
    public void setRef(QName ref) {
        getModelObject().setRef(ref);
    }

    @Override
    public void setSimpleType(SimpleType simpleType) {
        if (simpleType == null) {
            getModelObject().setSimpleType(null);
            return;
        }
        
        assert simpleType instanceof SimpleTypeImpl;
        SimpleTypeImpl simpleTypeImpl = (SimpleTypeImpl) simpleType;
        assert simpleTypeImpl.getModelObject() instanceof EJaxbLocalSimpleType;
        getModelObject().setSimpleType(
                (EJaxbLocalSimpleType) simpleTypeImpl.getModelObject());
    }

    @Override
    public void setType(QName type) {
        getModelObject().setType(type);
    }

    
    @Override
    public QName inferQName() {
        if (hasRef()) {
            return getRef();
        } else {
            Schema parentSchema = SchemaHelper.findParentSchema(this);
            String localPart = getName();
            if (localPart == null) return null;
            if (parentSchema == null) return new QName(localPart);
            return new QName(parentSchema.getTargetNamespace(), localPart);
        }
    }
    
    
    @Override
    public Type findType() {
        if (hasSimpleType()) {
            return getSimpleType();
        } else if (hasComplexType()) {
            return getComplexType();
        } else if (hasType()) {
            QName typeQName = getType();
            Schema parentSchema = SchemaHelper.findParentSchema(this);
            if (parentSchema == null) return null;
            Type type = SchemaHelper.findTypeByQName(parentSchema, typeQName);
            return type;
        }
        return null;
    }
    
    
    @Override
    public Element findRef() {
        if (!hasRef()) return null;
        return SchemaHelper.findElementByQName(SchemaHelper.findParentSchema(this), getRef());
    }
    
    
    
}
