/**
 * Raphael Diagram - A crossbrowser SVG/VML library creating diagrams - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.gwt.raphael.client.diagram.element;

import java.util.ArrayList;

import com.ebmwebsourcing.gwt.raphael.client.core.SVGElement;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementDragListenerAdapter;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementDropListenerAdapter;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramGroupElementListener;
import com.ebmwebsourcing.gwt.raphael.client.diagram.type.DiagramElementType;
import com.google.gwt.json.client.JSONNumber;
import com.google.gwt.json.client.JSONObject;




/**
 * A group Element accepts a list of DiagramElementType
 * This list of elements is dragged with the group element
 * without being included into it 
 * 
 * @author nfleury
 *
 */
public class DiagramGroupElement extends DiagramElement {
	
	private ArrayList<DiagramElementType> acceptedElementTypes =  new ArrayList<DiagramElementType>();
	
	private ArrayList<DiagramElementType> filteredElementTypes =  new ArrayList<DiagramElementType>();
	
	private ArrayList<DiagramElementChildData> childs 		   = new ArrayList<DiagramElementChildData>();
	
	private ArrayList<DiagramGroupElementListener> listeners   = new ArrayList<DiagramGroupElementListener>();  
	
	public DiagramGroupElement(SVGElement svgElement) {
	
			super(svgElement);
			
	}
	
	public DiagramGroupElement(SVGElement svgElement, DiagramElementType type) {
		super(svgElement, type);
		
	}

	@Override
	protected void onLoad() {
		super.onLoad();
		
		this.isDroppable(true);
		
		this.refreshSVGPosition();
	}
	
	/**
	 * Adds a element type that will be accepted by current <br> 
	 * element 
	 * @param type
	 */
	public void addAcceptedType(DiagramElementType type){
		this.acceptedElementTypes.add(type);
	}
	
	/**
	 * Add an element type that will be ignored while dragging <br>
	 * over current element
	 * 
	 * @param type
	 */
	public void addFilteredElementType(DiagramElementType type){
		this.filteredElementTypes.add(type);
	}
	
	/**
	 * Return true if current group element would ignore the <br>
	 * specified element
	 * @param diagramElement
	 * @return
	 */
	public boolean ingnoresElement(DiagramElement diagramElement){
		return this.filteredElementTypes.contains(diagramElement.getDiagramElementType());
	}
	
	/**
	 * 
	 * @param element
	 * @param x
	 * @param y
	 */
	public void addChild(DiagramElement element, int x, int y){
		
		DiagramElementChildData cdata = new DiagramElementChildData();
		cdata.setX(x);
		cdata.setY(y);
		cdata.setDiagraElement(element);
		
		this.childs.add(cdata);
		
		for(DiagramGroupElementListener listener:listeners){
			listener.onDiagramElementAdded(element, x, y);
		}
	}
	
	/**
	 * Update the position of a child in current group element
	 * @param element
	 * @param x
	 * @param y
	 */
	protected void updateChildPosition(DiagramElement element,int x,int y){
		
		for(DiagramElementChildData cdata: childs){
			
			if (cdata.getDiagramElement()==element){
				
				cdata.setX(x);
				cdata.setY(y);
			}
			
		}
		
	}
	
	public void remove(){
		super.remove();
		
		ArrayList<DiagramElement> elementsToRemove = new ArrayList<DiagramElement>();

		for(DiagramElementChildData cdata:childs){
			
			elementsToRemove.add(cdata.getDiagramElement());

		}

		for(DiagramElement el:elementsToRemove){
			el.remove();
			
		}
		
		childs.clear();
		
	}
	
	/**
	 * Remove the specified child from the children list
	 * @param element
	 */
	public void removeChild(DiagramElement element){
		
		DiagramElementChildData childToRemove = null;
		
		for(DiagramElementChildData cdata : childs){
			
			if (cdata.getDiagramElement()==element){
				childToRemove = cdata;
				break;
			}
			
		}
		

		childs.remove(childToRemove);
		
	}
	
	
	public boolean containsChild(DiagramElement element){
		return this.getChilds().contains(element);
	}
	
	/**
	 * Triggered when an acceptable element is dragged over current element
	 * @param acceptableElement that is being dragged over
	 */
	protected void acceptableElementOver(DiagramElement acceptableElement){
		
		acceptableElement.getJqueryObject().draggableRevert(false);
		
		dropPossibleIndicator();
		
		
		//Once the acceptable element is dropped 
		//restore the colors of the group element
		acceptableElement.addDragListener(new DiagramElementDragListenerAdapter(){
			@Override
			public void onStop(DiagramElement diagramElement) {
				
				restorInitialBorderAndBackGroundColor();
				
				//diagramElement.getDragListeners().remove(this);
				
				super.onStop(diagramElement);
			}
		});
		
	}
	
	protected void nonAcceptableElementOver(DiagramElement nonAcceptableElement){
		
		nonAcceptableElement.getJqueryObject().draggableRevert(true);
		

		dropImpossibleIndicator();

		
		// When the non-acceptable element is dropped
		// take it back to it initial location
		// restore the group element propeties		
		nonAcceptableElement.addDragListener(new DiagramElementDragListenerAdapter(){
			@Override
			public void onStop(DiagramElement diagramElement) {
				
				restorInitialBorderAndBackGroundColor();
				
				diagramElement.getJqueryObject().draggableRevert(false);
				
				diagramElement.getDragListeners().remove(this);
				
				
			}
		});
		
	}
	
	protected boolean diagramElementIsAcceptable(DiagramElement element){
		
		if (this.acceptedElementTypes.contains(element.getDiagramElementType())){
			return true;
		}
		
		return false;
	}
	
	
	
	
	@Override
	public void refreshSVGPosition() {
		
		//Move Parent svg Element		
		super.refreshSVGPosition();
		
	
		//But also move all of its childs relatively
		for(DiagramElementChildData cdata:this.childs){
			
			//The child is translated with the same variation as its parent
			DiagramElement element = cdata.getDiagramElement();
						
			element.setX( cdata.getX() + this.getDiagramPanel().getRelativeX(this));
			element.setY( cdata.getY() + this.getDiagramPanel().getRelativeY(this));
			
			element.refreshSVGPosition();
		
		}
		
		
		
	}
	
	
	public ArrayList<DiagramElement> getChilds() {
		
		ArrayList<DiagramElement> result = new ArrayList<DiagramElement>();
		
		for(DiagramElementChildData cdata : childs){
			result.add(cdata.getDiagramElement());
		}
		
	return result;
	}
	
	
	
	
	public ArrayList<DiagramElementChildData> getChildData(){
		return this.childs;
	}
	
	
	public DiagramElementChildData getChildDataForElement(DiagramElement element){
		
		for(DiagramElementChildData c:this.childs){
			if (c.getDiagramElement().equals(element)){
				return c;
			}
		}
		
		return null;
	}
	
	public void dropPossibleIndicator(){
		
		this.setBackgroundColor("blue");
		this.setOpacity("0.5");
		
		
		
	}
	
	
	public void dropImpossibleIndicator(){
		
		this.getSvgElement().attr("fill", "red");
		this.getSvgElement().attr("opacity", "0.5");
		
	}
	
	
	
	
	
	@Override
	protected void attachDefaultListeners() {
		super.attachDefaultListeners();
		
		//////////
		//Droppable events : check if the drop element is acceptable and add it
		//as a child if it is
		//////////////////
		
		this.addDropListener(new DiagramElementDropListenerAdapter(){
			
			@Override
			public void onOver(DiagramElement diagramElement,DiagramElement overElement) {
				
				
				
				//if the over element is an acceptable element
				//and is not in the ignoring list
				
				if (DiagramGroupElement.this.diagramElementIsAcceptable(overElement)==true && DiagramGroupElement.this.ingnoresElement(overElement)==false && DiagramGroupElement.this.containsChild(overElement)==false){
					
					acceptableElementOver(overElement);
					
				}else if (DiagramGroupElement.this.diagramElementIsAcceptable(overElement)==false && DiagramGroupElement.this.ingnoresElement(overElement)==false){
					
					nonAcceptableElementOver(overElement);
					
				}
				
				

			}
			
			@Override
			public void onOut(DiagramElement targetElement,DiagramElement outElement) {
			
					DiagramGroupElement.this.getSvgElement().attr("fill", "white");
					
					
					DiagramGroupElement.this.removeChild(outElement);
					
					for(DiagramGroupElementListener listener:listeners){
						listener.onDiagramElementRemoved(outElement,new Integer(outElement.getX()),new Integer(outElement.getY()));
					}
					
			}
			
			
			@Override
			public void onDrop(DiagramElement targetElement,DiagramElement droppedElement) {

					//if child already exists update its positions
					if (DiagramGroupElement.this.containsChild(droppedElement)){
						
						
						int relativePositionX = droppedElement.getDiagramPanel().getRelativeX(droppedElement) - targetElement.getX();
						int relativePositionY = droppedElement.getDiagramPanel().getRelativeY(droppedElement) - targetElement.getY();
						
						
						DiagramGroupElement.this.updateChildPosition(droppedElement, relativePositionX, relativePositionY);
						
					}
				
					//child doesn't exist in the group yet
					if (DiagramGroupElement.this.diagramElementIsAcceptable(droppedElement)==true  && DiagramGroupElement.this.ingnoresElement(droppedElement)==false && DiagramGroupElement.this.containsChild(droppedElement)==false){
						
						
						
						int relativePositionX = droppedElement.getDiagramPanel().getRelativeX(droppedElement) - targetElement.getX();
						int relativePositionY = droppedElement.getDiagramPanel().getRelativeY(droppedElement) - targetElement.getY();
						

						DiagramGroupElement.this.addChild(droppedElement,relativePositionX,relativePositionY);
						
						
						
						
					//dragged element is not acceptable	
					}else if (DiagramGroupElement.this.diagramElementIsAcceptable(droppedElement)==false && DiagramGroupElement.this.ingnoresElement(droppedElement)==false){
						
						
						  	
						
						  JSONObject json = new JSONObject();
						     
						     json.put("y", new JSONNumber(droppedElement.getContext().getLastYPosition()));
						     json.put("x", new JSONNumber(droppedElement.getContext().getLastXPosition()));

						     
						     droppedElement.getSvgElement().animate(json.getJavaScriptObject(), 400);
						
						  
						     for(DiagramGroupElementListener listener:listeners){
						    	 listener.onDiagramElementRejected(droppedElement);
						     }
						
					}
				

					
					
			}
			
			
			
		});
		
		
		
		
	}

	@Override
	public void hideSpecific() {
		super.hideSpecific();
		for (DiagramElementChildData child : childs) {
			child.getDiagramElement().hide();
		}
	}

	@Override
	protected void showSpecific() {
		super.showSpecific();
		for (DiagramElementChildData child : childs) {
			child.getDiagramElement().show();
		}
	}
	
	
	public void addListener(DiagramGroupElementListener listener){
		this.listeners.add(listener);
	}
	
	
	
	
}
