/**
 * Raphael Diagram - A crossbrowser SVG/VML library creating diagrams - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.gwt.raphael.client.diagram.connector;




import java.util.ArrayList;


import com.ebmwebsourcing.gwt.raphael.client.core.Line;
import com.ebmwebsourcing.gwt.raphael.client.core.Path;
import com.ebmwebsourcing.gwt.raphael.client.core.Rectangle;
import com.ebmwebsourcing.gwt.raphael.client.core.SVGElement;
import com.ebmwebsourcing.gwt.raphael.client.diagram.DiagramPanel;
import com.ebmwebsourcing.gwt.raphael.client.diagram.connector.MagnetPositionHelper.ClosestMagnets;
import com.ebmwebsourcing.gwt.raphael.client.diagram.element.DiagramElement;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramConnectorListener;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementDragListenerAdapter;
import com.ebmwebsourcing.gwt.raphael.client.diagram.semantic.Selectable;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.ui.Widget;





public class DiagramConnector extends Widget{

	private String id ;

	private ConnectorBeginPoint beginPoint;
	private ConnectorEndPoint endPoint;
	private Line line;
	private DiagramPanel diagramPanel;
	
	private Connectable source;
	private Connectable target;
	
	private ArrayList<DiagramConnectorListener> listeners = new ArrayList<DiagramConnectorListener>();
	
	public static int X2_DISTANCE = 100;
	public static int Y2_DISTANCE = 100;
	


	/**
	 * Default constructor, with default begin point and end point
	 * @param x1
	 * @param y1
	 */
	public DiagramConnector(String id,int x1, int y1){
		
	     this.id			= id;
	     
	     this.beginPoint 	= new ConnectorBeginPoint(new Path(DOM.createUniqueId(),"M 0 0 L 10 5 L 0 10 z",x1,y1),this);
	     this.endPoint 		= new ConnectorEndPoint(new Rectangle(DOM.createUniqueId(), x1+X2_DISTANCE, y1+Y2_DISTANCE, 6, 6, 3),this);
	     

		init();
		
	}
	
	
	/**
	 * Constructor to set a specific begin point and a specif end point svg Element
	 * @param beginPoint
	 * @param endPoint
	 */
	public DiagramConnector(String id,SVGElement beginPoint,SVGElement endPoint){
	
		this.id			= id;		
		this.beginPoint = new ConnectorBeginPoint(beginPoint, this);
		this.endPoint	= new ConnectorEndPoint(endPoint,this);
		

		init();
	}
	
	public String getId() {
		return id;
	}
	
	public void setId(String id) {
		this.id = id;
	}
	

	/**
	 * Connect connectable target and source together once the connector is loaded
	 * @param target
	 * @param source
	 */
	public void connect(Connectable source,Connectable target){
		
		this.target = target;
		this.source = source;
		
		if (target !=null && source !=null){
			this.connectConnectables();
		}
		
	}
	
	/**
	 * Actually connect connectable1 and connectable2 <br>
	 * by the closest magnet on both connectable
	 */
	private void connectConnectables(){
		
		MagnetPositionHelper mph = new MagnetPositionHelper(source,target);

		ClosestMagnets cm = mph.getClosestsMagnets();
		
		
		cm.getConnectable2Magnet().addExtremity(beginPoint);
		
		beginPoint.setAttachedToMagnet(cm.getConnectable2Magnet());
		
		cm.getConnectable1Magnet().addExtremity(endPoint);
		
		endPoint.setAttachedToMagnet(cm.getConnectable1Magnet());
		
		source.refreshConnectedConnectors();
		target.refreshConnectedConnectors();
		
	}
	
	
	
	public void setBorderColor(String color){
		
		this.getBeginPoint().getSvgElement().attr("stroke", color);
		this.getEndPoint().getSvgElement().attr("stroke", color);
		this.getLine().attr("stroke", color);

	}
	
	
	public void setBackgroundColor(String color){

		this.getBeginPoint().getSvgElement().attr("fill", color);
		this.getEndPoint().getSvgElement().attr("fill", color);
		this.getLine().attr("fill", color);

		
	}
	
	public void remove() {
		
		this.getBeginPoint().remove();
		this.getEndPoint().remove();
		this.line.remove();
		
		//check if begin point is connected - if yes remove it from connectors extremities
		if (this.getBeginPoint().getAttachedToMagnet()!=null) {
			this.getBeginPoint().getAttachedToMagnet().getConnectable().removeConnectorExtremity(beginPoint);
		}
		
		//check if end point is connected - if yes remove it from connectors extremities
		if (this.getEndPoint().getAttachedToMagnet()!=null) {
			this.getEndPoint().getAttachedToMagnet().getConnectable().removeConnectorExtremity(endPoint);
		}		
		
		for (DiagramConnectorListener listener:listeners) {
			
			listener.onRemove();
			
		}
		
	}
	
	
	public void addListener(DiagramConnectorListener listener){
		this.listeners.add(listener);
	}
	
	
	public ArrayList<DiagramConnectorListener> getListeners() {
		return listeners;
	}
	
	public void init(){
		
		
	     beginPoint.isDraggable(true);
	     endPoint.isDraggable(true);
	    
		
		beginPoint.addDragListener(new DiagramElementDragListenerAdapter(){
			@Override
			public void onDrag(DiagramElement diagramElement) {
					
				refresh();
				
			}
			
			@Override
			public void onStop(DiagramElement diagramElement) {
				
				refresh();
				
			}
			
			@Override
			public void onStart(DiagramElement diagramElement) {
			
				refresh();
			}
			
		});
		
		
		endPoint.addDragListener(new DiagramElementDragListenerAdapter(){
			@Override
			public void onDrag(DiagramElement diagramElement) {

				refresh();
				
			}
			
			
			@Override
			public void onStop(DiagramElement diagramElement) {
				refresh();
				
			}
			
			@Override
			public void onStart(DiagramElement diagramElement) {
				refresh();
			}
			
		});
		
	}
	
	
	public Line getLine() {
		return line;
	}
	
	
	protected void update(int x1,int y1,int x2,int y2){
		
		
		if (this.line!=null){
			this.line.remove();
		}


		int beginPointWidth 	= beginPoint.getSvgElement().getWidth();
		int beginPointHeight 	= beginPoint.getSvgElement().getHeight();
		
		int endPointWidth 		= endPoint.getSvgElement().getWidth();		
		int endPointHeight		= endPoint.getSvgElement().getHeight();

			x1 = x1 + (beginPointWidth/2);
			y1 = y1 + (beginPointHeight /2);
		
			x2 = x2 + (endPointWidth/2);
			y2 = y2 + (endPointHeight/2);
			
		this.line = new Line(DOM.createUniqueId(),x1,y1,x2,y2);
		
		
		diagramPanel.getRaphael().addElement(line);
		this.line.attr("stroke-width", "2");
		
		if (this.diagramPanel.getSelectedElements().contains(this) ){
			this.line.attr("stroke", "red");
		}
		
		
	}
	

	
	private double calculateRotation(int deltaX,int deltaY){
		
        double angle_rad = Math.atan2(deltaY,deltaX);
        double angle_deg = angle_rad*180.0/Math.PI;
        
        return angle_deg;
	}
	
	private double getEndPointRotation(){
		
		int deltaX = this.endPoint.getX() 		-  		this.beginPoint.getX();
	    int deltaY = this.endPoint.getY()  		-   	this.beginPoint.getY();

		
		return calculateRotation(deltaX, deltaY);
	}
	
	private double getBeginPointRotation(){
		
		int deltaX = this.beginPoint.getX() 	-  		this.endPoint.getX();
	    int deltaY = this.beginPoint.getY()  	-   	this.endPoint.getY();

		
		return calculateRotation(deltaX, deltaY);
	}	
	
	
	public ConnectorBeginPoint getBeginPoint() {
		return beginPoint;
	}
	
	public ConnectorEndPoint getEndPoint() {
		return endPoint;
	}
	

	public void setDiagramPanel(DiagramPanel diagramPanel) {
		this.diagramPanel = diagramPanel;
	}
	
	public void refresh(){
		
		this.beginPoint.getSvgElement().rotate((int) getBeginPointRotation(), true);
		this.endPoint.getSvgElement().rotate((int) getEndPointRotation(), true);
		
		this.update(beginPoint.getX(), beginPoint.getY(), endPoint.getX(), endPoint.getY());
	}
	
	public void onLoad(){
	
		
		if (source!=null && target !=null){
			connectConnectables();
		}
		
	}

	public Connectable getSource() {
		return source;
	}

	public void setSource(Connectable source) {
		this.source = source;
	}

	public Connectable getTarget() {
		return target;
	}

	public void setTarget(Connectable target) {
		this.target = target;
	}
	
	public void hide(){
		beginPoint.hide();
		endPoint.hide();
		line.attr("opacity", "0.0");
	}
	
	public void show(){
		beginPoint.show();
		endPoint.show();
		line.attr("opacity","1.0");
	}
	
	
}
