/**
 * webeditor-service - Petalslink is an open source editor providing tools for Service Oriented Architecture. - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.webeditor.server.impl.service.project;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpSession;

import com.ebmwebsourcing.webeditor.api.IProjectDao;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectFile;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectInstance;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectInstanceFormat;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectInstanceMetaData;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectType;
import com.ebmwebsourcing.webeditor.client.api.service.IProjectService;
import com.ebmwebsourcing.webeditor.impl.domain.exception.ServiceException;
import com.ebmwebsourcing.webeditor.impl.xstream.ProjectDao;
import com.ebmwebsourcing.webeditor.server.impl.service.download.DownloadService;
import com.ebmwebsourcing.webeditor.server.impl.service.upload.UploadService;
import com.ebmwebsourcing.webeditor.server.impl.service.user.UserServiceImpl;
import com.ebmwebsourcing.webeditor.server.impl.service.webeditor.WebEditorService;
import com.ebmwebsourcing.webeditor.shared.server.api.service.project.IProjectInstanceLoader;
import com.ebmwebsourcing.webeditor.shared.server.api.service.project.IProjectInstanceMetaDataLoader;
import com.ebmwebsourcing.webeditor.shared.server.api.service.project.IProjectInstanceWriter;

/**
 * Info: Temporary Singleton class until Spring dependency injection is implemented 
 * 
 * @author nfleury
 *
 */
public class ProjectServiceImpl implements IProjectService{
	
	
	private static ProjectServiceImpl instance;
	
	private ProjectClassLoaderService classLoaderService = ProjectClassLoaderService.getInstance();

	private UserServiceImpl userService = UserServiceImpl.getInstance();
	private WebEditorService webeditorService = WebEditorService.getInstance();
	
	private HttpSession session;
	
	private IProjectInstance currentProjectInstance;
	
	private IProjectDao projectDao;
	
	private ProjectServiceImpl() {
		this.projectDao = new ProjectDao();
	}
	
	public static ProjectServiceImpl getInstance(){
		
		if (instance==null){
			instance = new ProjectServiceImpl();
			
		}
		
		return instance;
	}
	
	public void setSession(HttpSession session) {
		this.session = session;
	}
	
	public IProjectInstance getCurrentProjectInstance() {
	    return currentProjectInstance;
	}
	
	
	@Override
	public String exportProjectInstance(IProjectInstance instance,
			IProjectInstanceFormat format)
			throws ServiceException {
		
		//NOTE: Before exporting => projectInstance should have been save. We should have a projectFile
		try{
		String writerQname						= instance.getProjectType().getProjectInstanceWriteQname();
		
		IProjectInstanceWriter piwriter			= classLoaderService.getInstanceWriter(writerQname);
		
		//if the selected export format is the same as actual projectType format we dont have to write the file
		//again.Otherwise we write the file in the selected format an put it in a temporary path. Then we put the file
		//input stream into a variable so that we can grab it later on to download the stuff 
		
		File fileToDownload 					= null;
		
		if (format.equals(instance.getProjectType().getProjectTypeFormat())){
		
			fileToDownload 						= ProjectServiceHelper.convert(instance.getFile()); 
		
		}else{
		
			File directory						= webeditorService.getTemporaryUploadFolderDirectory(userService.getLoggedUser(), instance.getProjectType());
			
			File projectFile					= ProjectServiceHelper.convert(instance.getFile());
			
			if(instance.getExportData()!=null){
				fileToDownload 					= piwriter.writeProjectFile(projectFile,instance.getExportData(),format,directory);
			}
			else{
				fileToDownload 					= piwriter.writeProjectFile(projectFile,format,directory);
			}
		
		}
		
		String id = String.valueOf(Math.random());
		
		DownloadService.getInstance().registerFileToDownload(id, fileToDownload);
		
		return id;
		}
		catch (Exception e) {
			e.printStackTrace();
			throw new ServiceException(e.getMessage());
		}
		
		
	}

	@Override
	public List<IProjectFile> getProjectFilesByProjectType(
			IProjectType projectType) throws ServiceException {
		

		ArrayList<IProjectFile> files = new ArrayList<IProjectFile>();
		
		try{
		
			List<IProjectInstance> instances = projectDao.getUserProjectsByType(userService.getLoggedUser(), projectType);

			for(IProjectInstance i : instances){
				
				files.add(i.getFile());
				
			}
		
		}catch (Exception e) {
			throw new ServiceException("Could not load project files",e.getCause());
		}

		
		
		return files;
	}
	
	@Override
	public List<IProjectInstance> importProjectInstance(String uploadedRef,
			IProjectType projectType,IProjectInstanceFormat format)
			throws ServiceException {
		
		IProjectInstanceLoader piloader = (IProjectInstanceLoader) classLoaderService.getInstance().getInstanceLoader(projectType.getProjectInstanceLoaderQname());
		
		File uploadedFile 				= UploadService.getInstance(session).getUploadedFileFromSession(uploadedRef, userService.getLoggedUser(), projectType);
		
		ArrayList<File> projectFiles 	= new ArrayList<File>();
		
		projectFiles.add(uploadedFile);
		
		List<IProjectInstance> instances = piloader.getProjectInstances(projectFiles,format);
		
	
		return instances;
	}
	
	

	@Override
	public List<IProjectInstance> loadProjectInstances(
			List<IProjectFile> projectFiles)
			throws ServiceException {
		
		if (projectFiles.size()==0){
			throw new ServiceException("There should be at least 1 project file in order to load a project instance");
		}
		
		if (projectFiles.get(0).getProjectType()==null){
		
			throw new ServiceException("Project file:"+projectFiles.get(0)+" must have a project type. Actual project type is null");
		
		}else{
			
			if (projectFiles.get(0).getProjectType().getProjectInstanceLoaderQname()==null){
				
				throw new ServiceException("Cannot load a project instance with a null project instance loader Qname");
				
			}
			
		}
		
		if (projectFiles.get(0).getFormat()==null){
			throw new ServiceException("Project file:"+projectFiles.get(0)+" must have a format. Actual format is null");
		}
		
		
		
		String loaderQname 					= projectFiles.get(0).getProjectType().getProjectInstanceLoaderQname();
		
		IProjectInstanceLoader piloader 	= (IProjectInstanceLoader) classLoaderService.getInstanceLoader(loaderQname);
		

		
		List<IProjectInstance> storedIns	= ProjectServiceHelper.retrieveProjectInstances(projectFiles);
		
		List<IProjectInstance> instances 	= piloader.getProjectInstances(storedIns);
		
		if (instances==null){
			throw new ServiceException("Project instance loader :"+loaderQname+" returned null. Please check implementation");
		}
		
		return instances;
	}

	@Override
	public IProjectFile saveProjectInstance(IProjectInstance projectInstance)
			throws ServiceException {
		
		String writerQname					= projectInstance.getProjectType().getProjectInstanceWriteQname();
		
		IProjectInstanceWriter piwriter		= classLoaderService.getInstanceWriter(writerQname);
		
		File directory						= webeditorService.getProjectDirectory(userService.getLoggedUser(), projectInstance);
		
		File file 							= piwriter.writeProjectFile(projectInstance,directory);
		
		if (file==null){
			throw new ServiceException("Project instance writer :"+writerQname+" returned null. Please check implementation");
		}
		
		IProjectFile pf						= ProjectServiceHelper.convert(file);
		
		
		pf.setId(String.valueOf(System.currentTimeMillis()));
		pf.setProjectType(projectInstance.getProjectType());
		pf.setFormat(projectInstance.getProjectType().getProjectTypeFormat());
		projectInstance.setFile(pf);
		projectDao.saveProjectInstance(projectInstance);
		
		return pf;
	}

	@Override
	public IProjectInstance getProjectInstanceById(String id)
			throws ServiceException {
		
		return projectDao.getProjectById(id);
	}

	
	/**
	 * TODO: MetaData Should be uploaded in a temporary directory and attached
	 * when the project is saved 
	 * 
	 */
	@Override
	public IProjectInstanceMetaData addMetaData(String uploadRef,
			IProjectInstance projectInstance,IProjectInstanceFormat metaDataFormat) throws ServiceException {
		
		if (projectInstance.getProjectType()==null){
			throw new ServiceException("Project instance must have an associated project type");
		}
		
		if (projectInstance.getProjectType().getMetaDataLoaderQnames()==null){
			throw new ServiceException("Project Type:"+projectInstance.getProjectType().getName()+" does not specify any meta data loader qnames");
		}
		
		
		//retrieve loader for actual format
		String qname 							= projectInstance.getProjectType().getMetaDataLoaderQnames().get(metaDataFormat);
		
		if (qname==null){
			throw new ServiceException("Project Type: "+projectInstance.getProjectType().getName()+" does not specify any meta data loader qname for format: "+metaDataFormat.getName());
		}
		
		currentProjectInstance = projectInstance;
		
		//get project instance metadata
		IProjectInstanceMetaDataLoader loader  	= classLoaderService.getMetaDataLoader(qname);
		
		//retrieve temporary uploaded file  
		File uploadedFile 						= UploadService.getInstance(session).getUploadedFileFromSession(uploadRef, userService.getLoggedUser(), projectInstance.getProjectType());
		
		
		IProjectInstanceMetaData metaData		= loader.getMetaData(uploadedFile, metaDataFormat);
		
		if (metaData==null){
			throw new ServiceException("Meta data loader with Qname:"+qname+" returned null. Please, make sure the loader have been properly implemeted");
		}
		
		//write files in final destination
		File attachmentsDir = webeditorService.getAttachedFilesDirectory(userService.getLoggedUser(), projectInstance);
		
		File finalFile     = new File(attachmentsDir.getAbsolutePath()+File.separator+uploadedFile.getName());

		if (uploadedFile.renameTo(finalFile)==false) throw new ServiceException("Could not move uploaded file in final destination");
		
		
		//assign file to ProjectInstanceMetaData
		IProjectFile metaDataFile 				= ProjectServiceHelper.convert(finalFile);
		
		metaData.setFile(metaDataFile);
		
		

		
		
		return metaData;
	}

	

}
