/****************************************************************************
 *
 *  Copyright (c) 2012, EBM WebSourcing
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *****************************************************************************/

package org.ow2.petals.admin.api;

import java.net.URL;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map.Entry;
import java.util.Properties;

import org.ow2.petals.admin.api.artifact.Artifact;
import org.ow2.petals.admin.api.artifact.ArtifactException;
import org.ow2.petals.admin.api.artifact.ArtifactUtils;
import org.ow2.petals.admin.api.artifact.Component;
import org.ow2.petals.admin.api.artifact.ServiceAssembly;
import org.ow2.petals.admin.api.artifact.SharedLibrary;
import org.ow2.petals.admin.api.artifact.lifecycle.ArtifactLifecycle;
import org.ow2.petals.admin.api.artifact.lifecycle.ArtifactLifecycleFactory;
import org.ow2.petals.admin.api.exception.ArtifactAdministrationException;
import org.ow2.petals.admin.api.exception.ArtifactNotFoundException;
import org.ow2.petals.admin.api.exception.ArtifactTypeIsNeededException;
import org.ow2.petals.admin.api.exception.UnsupportedArtifactTypeException;

/**
 * 
 * @author Nicolas Oddoux - EBM WebSourcing
 */
public abstract class ArtifactAdministration {

    protected ArtifactAdministration() {

    }

    private final void deployAndStart(Artifact artifact, URL artifactUrl)
            throws ArtifactAdministrationException {
        ArtifactLifecycle artifactLifecycle = ArtifactLifecycleFactory.newInstance(artifact);
        artifactLifecycle.deploy(artifactUrl);
        artifactLifecycle.start();
    }

    public void deployAndStartArtifact(URL artifactUrl) throws ArtifactAdministrationException {
        try {
            Artifact artifact = ArtifactUtils.createArtifact(artifactUrl);
            deployAndStart(artifact, artifactUrl);
        } catch (ArtifactException ae) {
            throw new ArtifactAdministrationException(ae);
        }
    }

    public void deployAndStartArtifact(URL artifactUrl, Properties configurationProperties)
            throws ArtifactAdministrationException {
        try {
            Artifact artifact = ArtifactUtils.createArtifact(artifactUrl, configurationProperties);
            deployAndStart(artifact, artifactUrl);
        } catch (ArtifactException ae) {
            throw new ArtifactAdministrationException(ae);
        }

    }

    public void deployAndStartArtifact(URL artifactUrl, URL configurationFile)
            throws ArtifactAdministrationException {
        try {
            Artifact artifact = ArtifactUtils.createArtifact(artifactUrl, configurationFile);
            deployAndStart(artifact, artifactUrl);
        } catch (ArtifactException ae) {
            throw new ArtifactAdministrationException(ae);
        }
    }

    public void startArtifact(URL artifactUrl) throws ArtifactAdministrationException {
        try {
            Artifact artifact = ArtifactUtils.createArtifact(artifactUrl);
            ArtifactLifecycle artifactLifecycle = ArtifactLifecycleFactory.newInstance(artifact);
            artifactLifecycle.start();
        } catch (ArtifactException ae) {
            throw new ArtifactAdministrationException(ae);
        }
    }

    /**
     * <p>Start an artifact.</p>
     * <p>
     * The artifact type is needed to distinct betwwen two JBI artifacts having the same
     * identifier.
     * </p>
     * 
     * @param type
     *            The nature (SL, SA, BC, SA) of the artifact to stop.
     * 
     * @param name
     *            The JBI identifier of the artifact to stop.
     * @throws UnsupportedArtifactTypeException
     *             <code>type</code> is a not supported artifact type.
     * @throws ArtifactNotFoundException
     *             No artifact found.
     * @throws ArtifactTypeIsNeededException
     *             Several artifacts match the name, the artifact type MUST be
     *             provided to be able to find the right artifact.
     * @throws ArtifactAdministrationException
     *             An error occurs retrieving the artifact.
     */
    public void startArtifact(String type, String name) throws UnsupportedArtifactTypeException,
            ArtifactNotFoundException, ArtifactTypeIsNeededException,
            ArtifactAdministrationException {
        Artifact artifact = getArtifact(type, name);
        ArtifactLifecycle artifactLifecycle = ArtifactLifecycleFactory.newInstance(artifact);
        artifactLifecycle.start();
    }

    /**
     * Stop artifact
     * 
     * @param artifactUrl
     * @throws ArtifactAdministrationException
     */
    public void stopArtifact(URL artifactUrl) throws ArtifactAdministrationException {
        try {
            Artifact artifact = ArtifactUtils.createArtifact(artifactUrl);
            ArtifactLifecycle artifactLifecycle = ArtifactLifecycleFactory.newInstance(artifact);
            artifactLifecycle.stop();
        } catch (ArtifactException ae) {
            throw new ArtifactAdministrationException(ae);
        }
    }

    /**
     * <p>Stop an artifact.</p>
	 * <p>The artifact type is needed to distinct betwwen two JBI artifacts having the same
     * identifier.</p>
     * 
     * @param type
     *            The nature (SL, SA, BC, SA) of the artifact to stop.
     * 
     * @param name
     *            The JBI identifier of the artifact to stop.
     * @throws UnsupportedArtifactTypeException
     *             <code>type</code> is a not supported artifact type.
     * @throws ArtifactNotFoundException
     *             No artifact found.
     * @throws ArtifactTypeIsNeededException
     *             Several artifacts match the name, the artifact type MUST be
     *             provided to be able to find the right artifact.
     * @throws ArtifactAdministrationException
     *             An error occurs retrieving the artifact.
     */
    public void stopArtifact(String type, String name) throws UnsupportedArtifactTypeException,
            ArtifactNotFoundException, ArtifactTypeIsNeededException,
            ArtifactAdministrationException {
        Artifact artifact = getArtifact(type, name);
        ArtifactLifecycle artifactLifecycle = ArtifactLifecycleFactory.newInstance(artifact);
        artifactLifecycle.stop();
    }

    public void stopAndUndeployArtifact(URL artifactUrl) throws ArtifactAdministrationException {
        try {
            Artifact artifact = ArtifactUtils.createArtifact(artifactUrl);
            this.stopAndUndeployArtifact(artifact);
        } catch (ArtifactException ae) {
            throw new ArtifactAdministrationException(ae);
        }
    }

    /**
	 * <p>Stop and undeploy an artifact.</p>
	 * <p>The artifact type is needed to distinct betwwen two JBI artifacts having the same
     * identifier.</p>
     * @throws UnsupportedArtifactTypeException
     *             <code>type</code> is a not supported artifact type.
     * @throws ArtifactNotFoundException
     *             No artifact found.
     * @throws ArtifactTypeIsNeededException
     *             Several artifacts match the name, the artifact type MUST be
     *             provided to be able to find the right artifact.
     * @throws ArtifactAdministrationException
     *             An error occurs retrieving the artifact.
     */
    public void stopAndUndeployArtifact(String type, String name)
            throws UnsupportedArtifactTypeException, ArtifactNotFoundException,
            ArtifactTypeIsNeededException, ArtifactAdministrationException {
        Artifact artifact = getArtifact(type, name);
        this.stopAndUndeployArtifact(artifact);
    }

    private void stopAndUndeployArtifact(Artifact artifact) throws ArtifactAdministrationException {
        ArtifactLifecycle artifactLifecycle = ArtifactLifecycleFactory.newInstance(artifact);
        artifactLifecycle.stop();
        artifactLifecycle.shutdown();
        artifactLifecycle.undeploy();
    }

    public Artifact getArtifactInfo(URL artifactUrl) throws ArtifactAdministrationException {
        try {
            Artifact artifact = ArtifactUtils.createArtifact(artifactUrl);
            ArtifactLifecycle artifactLifecycle = ArtifactLifecycleFactory.newInstance(artifact);
            artifactLifecycle.updateState();
            return artifact;
        } catch (ArtifactException ae) {
            throw new ArtifactAdministrationException(ae);
        }
    }

    /**
     * <p>
     * Get information about an artifact already deployed.
     * </p>
     * <p>
     * The artifact type is needed to distinct betwwen two JBI artifacts having the same
     * identifier.
     * </p>
     * 
     * @param type
     *            artifact type (SL,BC,SE,SA). Can be <code>null</code>.
     * @param name
     *            Artifact identifier. Not <code>null</code>.
     * @throws UnsupportedArtifactTypeException
     *             <code>type</code> is a not supported artifact type.
     * @throws ArtifactNotFoundException
     *             No artifact found.
     * @throws ArtifactTypeIsNeededException
     *             Several artifacts match the name, the artifact type MUST be
     *             provided to be able to find the right artifact.
     * @throws ArtifactAdministrationException
     *             An error occurs retrieving the artifact.
     */
    public Artifact getArtifactInfo(String type, String name)
            throws UnsupportedArtifactTypeException, ArtifactNotFoundException,
            ArtifactTypeIsNeededException, ArtifactAdministrationException {
        Artifact artifact = getArtifact(type, name);
        ArtifactLifecycle artifactLifecycle = ArtifactLifecycleFactory.newInstance(artifact);
        artifactLifecycle.updateState();
        return artifact;
    }

    /**
     * Deploy and start all specified artifacts taking into account the artifact
     * type.
     * 
     * @param artifactsUrl
     *            artifacts to deploy and start
     * @throws ArtifactAdministrationException
     */
    public void deployAndStartArtifacts(URL[] artifactsUrl) throws ArtifactAdministrationException {
        HashMap<URL, Artifact> sls = new HashMap<URL, Artifact>();
        HashMap<URL, Artifact> components = new HashMap<URL, Artifact>();
        HashMap<URL, Artifact> sas = new HashMap<URL, Artifact>();
        HashMap<URL, Artifact> unknows = new HashMap<URL, Artifact>();

        for (URL url : artifactsUrl) {
            Artifact artifact;
            try {
                artifact = ArtifactUtils.createArtifact(url);
                if (artifact instanceof SharedLibrary) {
                    sls.put(url, artifact);
                } else if (artifact instanceof Component) {
                    components.put(url, artifact);
                } else if (artifact instanceof ServiceAssembly) {
                    sas.put(url, artifact);
                } else {
                    unknows.put(url, artifact);
                }
            } catch (ArtifactException e) {
                throw new ArtifactAdministrationException(e);
            }
        }

        LinkedHashMap<URL, Artifact> maps = new LinkedHashMap<URL, Artifact>();
        maps.putAll(sls);
        maps.putAll(components);
        maps.putAll(sas);
        maps.putAll(unknows);

        for (Entry<URL, Artifact> entry : maps.entrySet()) {
            this.deployAndStart(entry.getValue(), entry.getKey());
        }
    }

    public abstract void stopAndUndeployAllArtifacts() throws ArtifactAdministrationException;

    public abstract List<Artifact> listArtifacts() throws ArtifactAdministrationException;

    /**
     * <p>
     * Create an artifact with this type and name. The artifact to return is
     * resolved by the underlying implementation.
     * </p>
     * <p>
     * The artifact type is needed to distinct betwwen two JBI artifacts having the same
     * identifier.
     * </p>
     * 
     * @param type
     *            artifact type (SL,BC,SE,SA). Can be <code>null</code>.
     * @param name
     *            Artifact identifier. Not <code>null</code>.
     * @return
     * @throws UnsupportedArtifactTypeException
     *             <code>type</code> is a not supported artifact type.
     * @throws ArtifactNotFoundException
     *             No artifact found.
     * @throws ArtifactTypeIsNeededException
     *             Several artifacts match the name, the artifact type MUST be
     *             provided to be able to find the right artifact.
     * @throws ArtifactAdministrationException
     *             An error occurs retrieving the artifact.
     */
    public abstract Artifact getArtifact(String type, String name)
            throws UnsupportedArtifactTypeException, ArtifactNotFoundException,
            ArtifactTypeIsNeededException, ArtifactAdministrationException;
}
