/****************************************************************************
 *
 *  Copyright (c) 2012, EBM WebSourcing
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *****************************************************************************/

package org.ow2.petals.admin.api.artifact;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Properties;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.apache.commons.io.FileUtils;
import org.ow2.petals.admin.api.artifact.Component.ComponentType;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import com.ebmwebsourcing.easycommons.lang.UncheckedException;

/**
 * 
 * @author Nicolas Oddoux - EBM WebSourcing
 */
public abstract class ArtifactUtils {

    /**
     * Property key to set {@link ArtifactUtils} implementation name
     */
    public static final String FACTORY_PROPERTY_KEY = "org.ow2.petals.admin.api.artifact.factory";

    /**
     * Path to the jbi.xml file in a JBI artifact archive
     */
    private static final String JBI_XML_PATH = "META-INF/jbi.xml";

    /**
     * XPath expression to determine if a JBI configuration is tied to a binding
     * component
     */
    private static final String XPATH_JBI_BC_TYPE = "/*[local-name()='jbi']/*[local-name()='component'][@type='binding-component']";

    /**
     * XPath expression to determine if a JBI configuration is tied to a shared
     * library
     */
    private static final String XPATH_JBI_SL_TYPE = "/*[local-name()='jbi']/*[local-name()='shared-library']";

    /**
     * XPath expression to determine if a JBI configuration is tied to a service
     * assembly
     */
    private static final String XPATH_JBI_SA_TYPE = "/*[local-name()='jbi']/*[local-name()='service-assembly']";

    /**
     * XPath expression to determine if a JBI configuration is tied to a service
     * engine
     */
    private static final String XPATH_JBI_SE_TYPE = "/*[local-name()='jbi']/*[local-name()='component'][@type='service-engine']";

    /**
     * XPath expression to get all service unit name declared in a
     * ServiceAssembly descriptor.
     */
    private static final String XPATH_JBI_SU_NAME_AND_TARGETCOMPONENT_LIST = "//*[local-name()='service-unit']/*[local-name()='identification']/*[local-name()='name'] | //*[local-name()='service-unit']/*[local-name()='target']/*[local-name()='component-name']"; // "//*[local-name()='service-unit']/*[local-name()='identification']/*[local-name()='name']";

    /**
     * XPath expression to access the name of a JBI artifact
     */
    private static final String XPATH_JBI_NAME = "//*[local-name()='identification']/*[local-name()='name']";

    /**
     * Create artifact from the specified URL
     * 
     * @param artifactUrl
     * @return instance of artifact
     * @throws ArtifactException
     */
    public final static Artifact createArtifact(URL artifactUrl) throws ArtifactException {
        assert artifactUrl != null;
        return createArtifact(artifactUrl, new Properties());
    }

    /**
     * Create artifact from the specified URL with the specified properties
     * files (Only useful for {@link Component})
     * 
     * @param artifactUrl
     * @param configurationFile
     * @return
     * @throws ArtifactException
     */
    public final static Artifact createArtifact(URL artifactUrl, URL configurationFile)
            throws ArtifactException {
        assert artifactUrl != null;
        assert configurationFile != null;

        try {
            Properties configurationProperties = new Properties();
            File tmpConfigurationFile = copyURLToTempZipFile(configurationFile);
            configurationProperties.load(new FileReader(tmpConfigurationFile));
            return createArtifact(artifactUrl, configurationProperties);
        } catch (Exception e) {
            throw new ArtifactException(e);
        }
    }

    /**
     * Create artifact from the specified URL with the specified properties
     * files (Only useful for {@link Component})
     * 
     * @param artifactUrl
     * @param configurationProperties
     * @return
     * @throws ArtifactException
     */
    public final static Artifact createArtifact(URL artifactUrl, Properties configurationProperties)
            throws ArtifactException {
        assert artifactUrl != null;
        assert configurationProperties != null;

        try {
            File tmpJbiFile = copyURLToTempZipFile(artifactUrl);
            Node jbiConfig = extractJbiXml(tmpJbiFile);
            tmpJbiFile.delete();

            XPath xpath = XPathFactory.newInstance().newXPath();
            if (((Boolean) xpath.evaluate(XPATH_JBI_BC_TYPE, jbiConfig, XPathConstants.BOOLEAN))
                    .booleanValue()) {
                return new Component(getName(jbiConfig), ComponentType.BC, configurationProperties);
            } else if (((Boolean) xpath.evaluate(XPATH_JBI_SE_TYPE, jbiConfig,
                    XPathConstants.BOOLEAN)).booleanValue()) {
                return new Component(getName(jbiConfig), ComponentType.BC, configurationProperties);
            } else if (((Boolean) xpath.evaluate(XPATH_JBI_SL_TYPE, jbiConfig,
                    XPathConstants.BOOLEAN)).booleanValue()) {
                return new SharedLibrary(getName(jbiConfig));
            } else if (((Boolean) xpath.evaluate(XPATH_JBI_SA_TYPE, jbiConfig,
                    XPathConstants.BOOLEAN)).booleanValue()) {
                NodeList lst = (NodeList) xpath.evaluate(
                        XPATH_JBI_SU_NAME_AND_TARGETCOMPONENT_LIST, jbiConfig,
                        XPathConstants.NODESET);
                ArrayList<ServiceUnit> sus = new ArrayList<ServiceUnit>();
                for (int i = 0; i < lst.getLength(); i = i + 2) {
                    Node suName = lst.item(i);
                    Node targetComponentName = lst.item(i + 1);
                    sus.add(new ServiceUnit(suName.getTextContent(), "NOVERSION",
                            targetComponentName.getTextContent()));
                }
                return new ServiceAssembly(getName(jbiConfig), sus);
            } else {
                String errorMessage = String.format("%s: not a valid JBI archive",
                        artifactUrl.toString());
                throw new ArtifactException(errorMessage);
            }
        } catch (Exception e) {
            throw new ArtifactException(e);
        }
    }

    private final static Node extractJbiXml(File archive) throws ZipException, IOException,
            ParserConfigurationException, SAXException {
        ZipFile zipfile = new ZipFile(archive);
        ZipEntry jbixml = zipfile.getEntry(JBI_XML_PATH);
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        DocumentBuilder db = dbf.newDocumentBuilder();
        Document doc = db.parse(zipfile.getInputStream(jbixml));
        doc.getDocumentElement().normalize();
        return doc;
    }

    private final static File copyURLToTempZipFile(URL url) throws IOException {
        File result = File.createTempFile("jbi", ".zip");
        result.deleteOnExit();
        FileUtils.copyURLToFile(url, result);
        return result;
    }

    /**
     * Get the JBI name of this artifact
     * 
     * @return the JBI name (not null)
     */
    private final static String getName(Node jbiConfig) {
        try {
            XPath xp = XPathFactory.newInstance().newXPath();
            String result = (String) xp.evaluate(XPATH_JBI_NAME, jbiConfig, XPathConstants.STRING);
            if (result == null) {
                throw new UncheckedException(
                        "The artifact name node is mandatory in the file jbi.xml");
            }
            return result;
        } catch (XPathExpressionException xpee) {
            throw new UncheckedException(xpee);
        }
    }

}
