/**
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package org.ow2.petals.kernel.ws.client;

import java.util.HashMap;
import java.util.Map;


/**
 * @author Adrien Ruffié - eBM WebSourcing
 *
 */
public class KernelWSFactory {
    
    // Factory Instance
    private static KernelWSFactory instance = null;
    
    static{
        instance = new KernelWSFactory();
    }
    /** Available services stored in
     * these maps, keys represent
     * the specified address where
     * the service client is bound
     */
    private Map<String,TopologyServiceClient> topologyServiceClients = null;
    private Map<String,DeploymentServiceClient> deploymentServiceClients = null;
    private Map<String,EndpointServiceClient> endpointServiceClients = null;
    private Map<String,InformationServiceClient> informationServiceClients = null;
    private Map<String,InstallationServiceClient> installationServiceClients = null;
    private Map<String,JBIArtefactsServiceClient> jbiArtefactsServiceClients = null;
    private Map<String,MonitoringManagementServiceClient> monitoringManagementServiceClients = null;
    private Map<String,RemoteManagementServiceClient> remoteManagementServiceClients = null;
    private Map<String,RuntimeServiceClient> runtimeServiceClients = null;
    private Map<String,ServiceAssemblyStateServiceClient> serviceAssemblyStateServiceClients = null;
    private Map<String,ArtifactRepositoryServiceClient> artifactRepositoryServiceClients = null;
    
    private KernelWSFactory(){
        //NOP
        this.topologyServiceClients = new HashMap<String,TopologyServiceClient>();
        this.deploymentServiceClients = new HashMap<String,DeploymentServiceClient>();
        this.endpointServiceClients = new HashMap<String,EndpointServiceClient>();
        this.informationServiceClients = new HashMap<String,InformationServiceClient>();
        this.installationServiceClients = new HashMap<String,InstallationServiceClient>();
        this.jbiArtefactsServiceClients = new HashMap<String,JBIArtefactsServiceClient>();
        this.monitoringManagementServiceClients = new HashMap<String,MonitoringManagementServiceClient>();
        this.remoteManagementServiceClients = new HashMap<String,RemoteManagementServiceClient>();
        this.runtimeServiceClients = new HashMap<String,RuntimeServiceClient>();
        this.serviceAssemblyStateServiceClients = new HashMap<String,ServiceAssemblyStateServiceClient>();
        this.artifactRepositoryServiceClients = new HashMap<String,ArtifactRepositoryServiceClient>();
    }
    
    /**
     * function can allow to recover the
     * factory instance
     */
    public static synchronized KernelWSFactory getInstance() {
        return instance;
    }

    /**
     * Get the topology service client
     * for the specified address
     * @param String address
     * @return TopologyServiceClient
     */
    public synchronized TopologyServiceClient getTopologyServiceClient(final String address){
        TopologyServiceClient client = this.topologyServiceClients.get(address);
        if(client == null){
            client = new TopologyServiceClient(address);
            this.topologyServiceClients.put(address, client);
        }
        return client;
    }
    
    /**
     * Get the deployment service client
     * for the specified address
     * @param String address
     * @return DeploymentServiceClient
     */
    public synchronized DeploymentServiceClient getDeploymentServiceClient(final String address){
        DeploymentServiceClient client = this.deploymentServiceClients.get(address);
        if(client == null){
            client = new DeploymentServiceClient(address);
            this.deploymentServiceClients.put(address, client);
        }
        return client;
    }
    
    /**
     * Get the endpoint service client
     * for the specified address
     * @param String address
     * @return EndpointServiceClient
     */
    public synchronized EndpointServiceClient getEndpointServiceClient(final String address){
        EndpointServiceClient client = this.endpointServiceClients.get(address);
        if(client == null){
            client = new EndpointServiceClient(address);
            this.endpointServiceClients.put(address, client);
        }
        return client;
    }
    
    /**
     * Get the information service client
     * for the specified address
     * @param String address
     * @return InformationServiceClient
     */
    public synchronized InformationServiceClient getInformationServiceClient(final String address){
        InformationServiceClient client = this.informationServiceClients.get(address);
        if(client == null){
            client = new InformationServiceClient(address);
            this.informationServiceClients.put(address, client);
        }
        return client;
    }
    
    /**
     * Get the installation service client
     * for the specified address
     * @param String address
     * @return InstallationServiceClient
     */
    public synchronized InstallationServiceClient getInstallationServiceClient(final String address){
        InstallationServiceClient client = this.installationServiceClients.get(address);
        if(client == null){
            client = new InstallationServiceClient(address);
            this.installationServiceClients.put(address, client);
        }
        return client;
    }
    
    /**
     * Get the jbi artefacts service client
     * for the specified address
     * @param String address
     * @return JBIArtefactsServiceClient
     */
    public synchronized JBIArtefactsServiceClient getJBIArtefactsServiceClient(final String address){
        JBIArtefactsServiceClient client = this.jbiArtefactsServiceClients.get(address);
        if(client == null){
            client = new  JBIArtefactsServiceClient(address);
            this.jbiArtefactsServiceClients.put(address, client);
        }
        return client;
    }
    
    /**
     * Get the monitoring management service 
     * client for the specified address
     * @param String address
     * @return MonitoringManagementServiceClient
     */
    public synchronized MonitoringManagementServiceClient getMonitoringManagementServiceClient(final String address){
        MonitoringManagementServiceClient client = this.monitoringManagementServiceClients.get(address);
        if(client == null){
            client = new MonitoringManagementServiceClient(address);
            this.monitoringManagementServiceClients.put(address, client);
        }
        return client;
    }
    
    /**
     * Get the remote management service 
     * client for the specified address
     * @param String address
     * @return RemoteManagementServiceClient
     */
    public synchronized RemoteManagementServiceClient getRemoteManagementServiceClient(final String address){
        RemoteManagementServiceClient client = this.remoteManagementServiceClients.get(address);
        if(client == null){
            client = new RemoteManagementServiceClient(address);
            this.remoteManagementServiceClients.put(address, client);
        }
        return client;
    }
    
    /**
     * Get the runtime service client
     * for the specified address
     * @param String address
     * @return RuntimeServiceClient
     */
    public synchronized RuntimeServiceClient getRuntimeServiceClient(final String address){
        RuntimeServiceClient client = this.runtimeServiceClients.get(address);
        if(client == null){
            client = new RuntimeServiceClient(address);
            this.runtimeServiceClients.put(address, client);
        }
        return client;
    }
    
    /**
     * Get the service assembly state service
     * client for the specified address
     * @param String address
     * @return ServiceAssemblyStateServiceClient
     */
    public synchronized ServiceAssemblyStateServiceClient getServiceAssemblyStateServiceClient(final String address){
        ServiceAssemblyStateServiceClient client = this.serviceAssemblyStateServiceClients.get(address);
        if(client == null){
            client = new ServiceAssemblyStateServiceClient(address);
            this.serviceAssemblyStateServiceClients.put(address, client);
        }
        return client;
    }
    
    /**
     * Get the service artifact repository service
     * client for the specified address
     * @param String address
     * @return ArtifactRepositoryServiceClient
     */
    public synchronized ArtifactRepositoryServiceClient getArtifactRepositoryServiceClient(final String address){
        ArtifactRepositoryServiceClient client = this.artifactRepositoryServiceClients.get(address);
        if(client == null){
            client = new ArtifactRepositoryServiceClient(address);
            this.artifactRepositoryServiceClients.put(address, client);
        }
        return client;
    }
    
    /**
     * Remove the topology service client
     * for the specified address
     * @param String address
     */
    public synchronized void removeTopologyServiceClient(final String address){
        this.topologyServiceClients.remove(address);
    }
    
    /**
     * Remove the deployment service client
     * for the specified address
     * @param String address
     */
    public synchronized void removeDeploymentServiceClient(final String address){
        this.deploymentServiceClients.remove(address);
    }
    
    /**
     * Remove the endpoint service client
     * for the specified address
     * @param String address
     */
    public synchronized void removeEndpointServiceClient(final String address){
        this.endpointServiceClients.remove(address);
    }
    
    /**
     * Remove the information service client
     * for the specified address
     * @param String address
     */
    public synchronized void removeInformationServiceClient(final String address){
        this.informationServiceClients.remove(address);
    }
    
    /**
     * Remove the installation service client
     * for the specified address
     * @param String address
     */
    public synchronized void removeInstallationServiceClient(final String address){
        this.installationServiceClients.remove(address);
    }
    
    /**
     * Remove the installation service client
     * for the specified address
     * @param String address
     */
    public synchronized void removeJBIArtefactsServiceClient(final String address){
        this.jbiArtefactsServiceClients.remove(address);
    }
    
    /**
     * Remove the monitoring management service
     * client for the specified address
     * @param String address
     */
    public synchronized void removeMonitoringManagementServiceClient(final String address){
        this.monitoringManagementServiceClients.remove(address);
    }
    
    /**
     * Remove the remote management service
     * client for the specified address
     * @param String address
     */
    public synchronized void removeRemoteManagementServiceClient(final String address){
        this.remoteManagementServiceClients.remove(address);
    }
    
    /**
     * Remove the runtime service client
     * for the specified address
     * @param String address
     */
    public synchronized void removeRuntimeServiceClient(final String address){
        this.runtimeServiceClients.remove(address);
    }
    
    /**
     * Remove the service assembly state service
     * client for the specified address
     * @param String address
     */
    public synchronized void removeServiceAssemblyStateServiceClient(final String address){
        this.serviceAssemblyStateServiceClients.remove(address);
    }
    
    /**
     * Remove the artifact repository service
     * client for the specified address
     * @param String address
     */
    public synchronized void removeArtifactRepositoryServiceClient(final String address){
        this.artifactRepositoryServiceClients.remove(address);
    }
}
