/**
 * easySchema - SOA Tools Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * $id.java
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.commons.schema.api;

import java.net.URI;
import java.util.Map;

import javax.xml.transform.Source;
import javax.xml.transform.dom.DOMSource;

import org.w3c.dom.Document;
import org.xml.sax.InputSource;

/**
 * This interface describes a collection of methods
 * that enable conversion of a Schema document (in XML,
 * following the Schema schema described in the Schema
 * specification) into a Schema model.
 *
 * @author Nicolas Salatge - eBM WebSourcing
 */
public interface SchemaReader
{
	
    /**
     * Constants for the Message Exchange Patterns.
     * 
     */
    public enum FeatureConstants {
    	VERBOSE("com.ebmwebsourcing.commons.schema.verbose"),
        IMPORT_DOCUMENTS("com.ebmwebsourcing.commons.schema.importDocuments");

        private final String value;

        /**
         * Creates a new instance of {@link FeatureConstants}
         * 
         * @param value
         */
        private FeatureConstants(final String value) {
            this.value = value;
        }

        /**
         * 
         * @return
         */
        public String value() {
            return value;
        }
        
        /**
         * Please use this equals method instead of using :<code>
         * value().equals(value)
         * </code>
         * which is almost 10 times slower...
         * 
         * @param value
         * @return
         */
        public boolean equals(String value) {
            return toString().equals(value.toString());
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.lang.Enum#toString()
         */
        public String toString() {
            return this.value;
        }
    }
	
	
    /**
     * Sets the specified feature to the specified value.
     * <p>
     * The minimum features that must be supported are:
     * <p>
     * <table border=1>
     *   <tr>
     *     <th>Name</th>
     *     <th>Description</th>
     *     <th>Default Value</th>
     *   </tr>
     *   <tr>
     *     <td><center>com.ebmwebsourcing.commons.schema.verbose</center></td>
     *     <td>If set to true, status messages will be displayed.</td>
     *     <td><center>type: boolean - default value: false</center></td>
     *   </tr>
     *   <tr>
     *     <td><center>com.ebmwebsourcing.commons.schema.importDocuments</center></td>
     *     <td>If set to true, imported WSDL documents will be
     *         retrieved and processed.</td>
     *     <td><center>type: boolean - default value: true</center></td>
     *   </tr>
     *   <tr>
     *     <td><center>com.ebmwebsourcing.commons.schema.pathDirectoryOfImportLocations</center></td>
     *     <td>If the location is set, imported WSDL documents will be
     *         retrieved at this location (Set the importDocuments Features at true).</td>
     *     <td><center>type: String</center></td>
     *   </tr>
     * </table>
     * <p>
     * All feature names must be fully-qualified, Java package style. All
     * names starting with com.ebmwebsourcing. are reserved for features defined
     * by the specification. It is recommended that implementation-
     * specific features be fully-qualified to match the package name
     * of that implementation. For example: com.abc.featureName
     *
     * @param name the name of the feature to be set.
     * @param value the value to set the feature to.
     * @throws IllegalArgumentException if the feature name is not recognized.
     * @see #getFeature(String)
     */
    public void setFeature(FeatureConstants name, Object value)
      throws IllegalArgumentException;

    /**
     * Gets the value of the specified feature.
     *
     * @param name the name of the feature to get the value of.
     * @return the value of feature
     * @throws IllegalArgumentException if the feature name is not recognized.
     * @see #setFeature(String, boolean)
     */
    public Object getFeature(FeatureConstants name);
    
    
    /**
     * Gets all features.
     *
     * @return the features
     * @see #setFeature(String, boolean)
     */
    public Map<FeatureConstants, Object> getFeatures();


  /**
   * Read the Schema document accessible via the specified
   * URI into a Schema definition.
   *
   * @param schemaURI a URI (can be a filename or URL) pointing to a
   * Schema XML definition.
   * @return the definition.
   */
  public Schema readSchema(URI schemaURI) throws SchemaException;

  /**
   * Read the specified Schema document into a Schema definition.
   *
   * @param documentBaseURI the document base URI of the Schema definition
   * described by the document. Will be set as the documentBaseURI
   * of the returned Definition. Can be null, in which case it
   * will be ignored.
   * @param wsdlDocument the Schema document, an XML
   * document obeying the Schema schema.
   * @return the definition described in the document.
   */
  public Schema readSchema(Document wsdlDocument)
    throws SchemaException;

  /**
   * Read a Schema document into a Schema definition.
   *
   * @param schemaURI the uri of the schema
   * @param inputSource an InputSource pointing to the
   * Schema document, an XML document obeying the Schema schema.
   * @return the definition described in the document pointed to
   * by the InputSource.
   */
  public Schema readSchema(URI schemaURI, DOMSource inputSource) throws SchemaException;


}