/**
 * easySchema - SOA Tools Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * $id.java
 * -------------------------------------------------------------------------
 */
package com.ebmwebsourcing.commons.schema.api.abstractElmt;

import java.io.File;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.logging.Logger;

import javax.xml.XMLConstants;
import javax.xml.namespace.NamespaceContext;
import javax.xml.namespace.QName;

import org.jdom.Document;
import org.jdom.Namespace;

import com.ebmwebsourcing.commons.schema.SchemaFactory;
import com.ebmwebsourcing.commons.schema.api.Attribute;
import com.ebmwebsourcing.commons.schema.api.Element;
import com.ebmwebsourcing.commons.schema.api.Import;
import com.ebmwebsourcing.commons.schema.api.Include;
import com.ebmwebsourcing.commons.schema.api.Schema;
import com.ebmwebsourcing.commons.schema.api.SchemaException;
import com.ebmwebsourcing.commons.schema.api.Type;
import com.ebmwebsourcing.commons.schema.api.SchemaReader.FeatureConstants;

/**
 * @author Nicolas Salatge - eBM WebSourcing
 */
public abstract class AbstractSchema<E> extends AbstractSchemaElement<E> implements Schema {

	private static Logger log = Logger.getLogger(AbstractSchema.class.getName());
	
	private Namespace xsi = org.jdom.Namespace.getNamespace("xsi", XMLConstants.W3C_XML_SCHEMA_INSTANCE_NS_URI);


	/**
	 * Features
	 */
	protected Map<FeatureConstants, Object> features = new HashMap<FeatureConstants, Object>();

	/**
	 * the jdom document
	 */
	protected Document document;

	/**
	 * the namespace context
	 */
	protected NamespaceContext namespaceContext;

	/**
	 * list of imports
	 */
	protected List<Import> imports = new ArrayList<Import>();

	/**
	 * list of includes
	 */
	protected List<Include> includes = new ArrayList<Include>();

	/**
	 * the list of types
	 */
	public List<Type> types = new ArrayList<Type>();


	/**
	 * the list of elements
	 */
	public List<Element> elements = new ArrayList<Element>();


	/**
	 * the list of attributes
	 */
	public List<Attribute> attributes = new ArrayList<Attribute>();


	/**
	 * the baseUri
	 */
	protected URI baseURI;
	
    /**
     * schema location
     */
    protected Map<String, String> schemaLocations = new HashMap<String, String>();

	/**
	 *
	 * @param schema
	 * @param doc
	 */
	public AbstractSchema(URI baseURI, org.w3._2001.xmlschema.Schema schema, Document doc) {
		this.document = doc;
		this.baseURI = baseURI;
		this.namespaceContext = new com.ebmwebsourcing.commons.schema.api.extensions.NamespaceContext();


		// get all namespaces
		List<Namespace> namespaces = this.document.getRootElement().getAdditionalNamespaces();
		for (Namespace ns : namespaces) {
			if((((ns.getPrefix() == null)||(ns.getPrefix().equals(""))))&&
					(this.document.getRootElement().getNamespace().getURI().equals(ns.getURI()))) {
				((com.ebmwebsourcing.commons.schema.api.extensions.NamespaceContext)this.namespaceContext).addNamespace(this.document.getRootElement().getNamespace().getPrefix(), this.document.getRootElement().getNamespace().getURI());
			} else {
				((com.ebmwebsourcing.commons.schema.api.extensions.NamespaceContext)this.namespaceContext).addNamespace(ns.getPrefix(), ns.getURI());
			}
		}
		
        // get schemaLocation
        if (this.document.getRootElement().getAttributeValue("schemaLocation", xsi) != null) {
            String val = this.document.getRootElement().getAttributeValue("schemaLocation", xsi);
            final StringTokenizer st = new StringTokenizer(val, " ");
            String location = null;
            String ns = null;
            while (st.hasMoreTokens()) {
                ns = st.nextToken();
                if (st.hasMoreTokens()) {
                    location = st.nextToken();
                } else {
                    location = null;
                }
                if(ns != null) {
                    schemaLocations.put(ns, location);
                }
                ns = null;
            }
        }
	}


	protected void addImportElementsInAllList() {
		for(Import impt: this.imports) {
			if(impt.getSchema() != null) {
				this.types.addAll(impt.getSchema().getTypes());
				this.elements.addAll(impt.getSchema().getElements());
				this.attributes.addAll(impt.getSchema().getAttributes());
			}
		}
	}


	protected void addIncludeElementsInAllList() {
		for(Include incl: this.includes) {
			if(incl.getSchema() != null) {
				this.types.addAll(incl.getSchema().getTypes());
				this.elements.addAll(incl.getSchema().getElements());
				this.attributes.addAll(incl.getSchema().getAttributes());
			}
		}
	}


	/**
	 * Import operation
	 */
	public void addImport(Import impt) {
		if(this.imports == null) {
			this.imports = new ArrayList<Import>();
		}
		this.imports.add(impt);
	}

	public List<Import> getImports() {
		if(this.imports == null) {
			this.imports = new ArrayList<Import>();
		}
		return this.imports;
	}

	public List<Import> getImports(String namespaceUri) {
		List<Import> res = new ArrayList<Import>();
		for (Import impt : this.imports) {
			if(impt.getNamespaceURI().equals(namespaceUri)) {
				res.add(impt);
			}
		}
		return res;
	}


	/**
	 * Include operation
	 * @throws WSDLException
	 */
	public void addInclude(Include incl) throws SchemaException {
		if(this.includes == null) {
			this.includes = new ArrayList<Include>();
		}
		this.includes.add(incl);
	}

	public List<Include> getIncludes() {
		if(this.includes == null) {
			this.includes = new ArrayList<Include>();
		}
		return this.includes;
	}

	public List<Include> getIncludes(String locationUri) {
		List<Include> res = new ArrayList<Include>();
		for (Include incl : this.includes) {
			if(incl.getLocationURI().equals(locationUri)) {
				res.add(incl);
			}
		}
		return res;
	}



	/*
	 * Types methods
	 */

	/*
	 * (non-Javadoc)
	 * @see com.ebmwebsourcing.commons.schema.api.Schema#getTypes()
	 */
	public List<Type> getTypes() {
		if(types == null) {
			this.types = new ArrayList<Type>();
		}
		return this.types;
	}

	/*
	 * (non-Javadoc)
	 * @see com.ebmwebsourcing.commons.schema.api.Schema#getType(javax.xml.namespace.QName)
	 */
	public Type getType(QName type) {
		Type res = null;
		if(types == null) {
			this.types = new ArrayList<Type>();
		}
		for (Type t : this.types) {
			if(t.getQName().equals(type)) {
				res = t;
				break;
			}
		}

		if((res == null)&&(SchemaFactory.getDefaultSchema() != null)) {
			for(Type t: SchemaFactory.getDefaultSchema().getTypes()) {
				if(t.getQName().equals(type)) {
					res = t;
					break;
				}
			}
		}
		return res;
	}

	/*
	 * (non-Javadoc)
	 * @see com.ebmwebsourcing.commons.schema.api.Schema#addType(com.ebmwebsourcing.commons.schema.api.Type)
	 */
	public void addType(Type type) {
		if(types == null) {
			this.types = new ArrayList<Type>();
		}
		this.types.add(type);
	}


	/*
	 * Attributes methods
	 */

	/*
	 * (non-Javadoc)
	 * @see com.ebmwebsourcing.commons.schema.api.Schema#getAttributes()
	 */
	public List<Attribute> getAttributes() {
		if(this.attributes == null) {
			this.attributes = new ArrayList<Attribute>();
		}
		return this.attributes;
	}

	/*
	 * (non-Javadoc)
	 * @see com.ebmwebsourcing.commons.schema.api.Schema#getAttribute(javax.xml.namespace.QName)
	 */
	public Attribute getAttribute(QName element) {
		Attribute res = null;
		if(attributes == null) {
			this.attributes = new ArrayList<Attribute>();
		}
		for (Attribute e : this.attributes) {
			if((e.getName().equals(element.getLocalPart()))&&
					(e.getNamespaceUri().equals(element.getNamespaceURI()))) {
				res = e;
				break;
			}
		}
		return res;
	}

	/*
	 * (non-Javadoc)
	 * @see com.ebmwebsourcing.commons.schema.api.Schema#addAttribute(com.ebmwebsourcing.commons.schema.api.Attribute)
	 */
	public void addAttribute(Attribute elmt) {
		if(attributes == null) {
			this.attributes = new ArrayList<Attribute>();
		}
		this.attributes.add(elmt);
	}




	/*
	 * Elements methods
	 */

	/*
	 * (non-Javadoc)
	 * @see com.ebmwebsourcing.commons.schema.api.Schema#getElements()
	 */
	public List<Element> getElements() {
		if(elements == null) {
			this.elements = new ArrayList<Element>();
		}
		return this.elements;
	}

	/*
	 * (non-Javadoc)
	 * @see com.ebmwebsourcing.commons.schema.api.Schema#getAttribute(javax.xml.namespace.QName)
	 */
	public Element getElement(QName element) {
		Element res = null;
		if(elements == null) {
			this.elements = new ArrayList<Element>();
		}
		for (Element e : this.elements) {
			if(e.getQName().equals(element)) {
				res = e;
				break;
			}
		}
		return res;
	}

	/*
	 * (non-Javadoc)
	 * @see com.ebmwebsourcing.commons.schema.api.Schema#addElement(com.ebmwebsourcing.commons.schema.api.Element)
	 */
	public void addElement(Element elmt) {
		if(elements == null) {
			this.elements = new ArrayList<Element>();
		}
		this.elements.add(elmt);
	}


	/*
	 * Namespace method
	 */

	/*
	 * (non-Javadoc)
	 * @see com.ebmwebsourcing.commons.schema.api.Schema#getAllNamespaces()
	 */
	public NamespaceContext getAllNamespaces() {
		return this.namespaceContext;
	}


	/**
	 * @return the features
	 */
	public Map<FeatureConstants, Object> getFeatures() {
		if(features == null) {
			this.features = new HashMap<FeatureConstants, Object>();
		}
		return features;
	}


	/**
	 * @param features the features to set
	 */
	public void setFeatures(Map<FeatureConstants, Object> features) {
		this.features = features;
	}



	/**
	 * @return the document
	 */
	public Document getDocument() {
		return document;
	}


	/**
	 * methods for baseURI
	 */

	public URI getDocumentBaseURI() {
		URI res = null;
		try {
			if(this.baseURI.getPath().lastIndexOf("/") != -1) {
				res = new URI(this.baseURI.getPath().substring(0, this.baseURI.getPath().lastIndexOf("/") + 1));
			}
		} catch (URISyntaxException e) {
			log.warning("BaseURI is null : " + this.baseURI);
		}
		return res;
	}

	public void setDocumentBaseURI(URI documentBaseURI) {
		this.baseURI = documentBaseURI;
	}
	
    /**
     * @return the schemaLocations
     */
    public Map<String, String> getSchemaLocation() {
        return schemaLocations;
    }

	public String toString() {
		return this.baseURI.toString();
	}
}
