/**
 * easyWSDL - SOA Tools Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * $id.java
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.commons.wsdl.api;

import java.net.URI;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.NamespaceContext;
import javax.xml.namespace.QName;

import com.ebmwebsourcing.commons.wsdl.util.Constants.WSDLVersionConstants;

/**
 * This interface represents a WSDL definition.
 * 
 * @author Nicolas Salatge - eBM WebSourcing
 */
public interface Description extends WSDLElement {
    /**
     * Set the document base URI of this definition. Can be used to represent
     * the origin of the Definition, and can be exploited when resolving
     * relative URIs (e.g. in &lt;import&gt;s).
     * 
     * @param documentBaseURI
     *            the document base URI of this definition
     */
    void setDocumentBaseURI(URI documentBaseURI);

    /**
     * Get the document base URI of this definition.
     * 
     * @return the document base URI
     */
    URI getDocumentBaseURI();

    /**
     * Set the name of this definition.
     * 
     * @param name
     *            the desired name
     * @throws WSDLException
     */
    void setQName(QName name) throws WSDLException;

    /**
     * Get the name of this definition.
     * 
     * @return the definition name
     * @throws WSDLException
     */
    QName getQName() throws WSDLException;

    /**
     * Set the target namespace in which WSDL elements are defined.
     * 
     * @param targetNamespace
     *            the target namespace
     */
    void setTargetNamespace(String targetNamespace);

    /**
     * Get the target namespace in which the WSDL elements are defined.
     * 
     * @return the target namespace
     */
    String getTargetNamespace();

    /**
     * This is a way to add a namespace association to a definition. It is
     * similar to adding a namespace prefix declaration to the top of a
     * &lt;wsdl:definition&gt; element. This has nothing to do with the
     * &lt;wsdl:import&gt; element; there are separate methods for dealing with
     * information described by &lt;wsdl:import&gt; elements.
     * 
     * @param prefix
     *            the prefix to use for this namespace (when rendering this
     *            information as XML). Use null or an empty string to describe
     *            the default namespace (i.e. xmlns="...").
     * @param namespaceURI
     *            the namespace URI to associate the prefix with. If you use
     *            null, the namespace association will be removed.
     */
    void addNamespace(String prefix, String namespaceURI);

    /**
     * Remove the namespace URI associated with this prefix.
     * 
     * @param prefix
     *            the prefix of the namespace to be removed.
     * @return the namespace URI which was removed.
     */
    String removeNamespace(String prefix);

    /**
     * Get all namespace associations in this definition. The keys are the
     * prefixes, and the namespace URIs are the values. This is unrelated to the
     * &lt;wsdl:import&gt; element.
     * 
     * @see #addNamespace(String, String)
     */
    NamespaceContext getNamespaces();

    /**
     * Set the types section.
     */
    void setTypes(Types types);

    /**
     * Get the types section.
     * 
     * @return the types section
     */
    Types getTypes();

    /**
     * Add an import to this WSDL description.
     * 
     * @param importDef
     *            the import to be added
     */
    void addImport(Import importDef);

    /**
     * Remove an import from this WSDL description.
     * 
     * @param importDef
     *            the import to be removed
     * @return the removed ImportImpl
     */
    Import removeImport(Import importDef);

    /**
     * Get the list of imports for the specified namespaceURI.
     * 
     * @param namespaceURI
     *            the namespaceURI associated with the desired imports.
     * @return a list of the corresponding imports, or null if there weren't any
     *         matching imports
     */
    List<Import> getImports(String namespaceURI);

    /**
     * Get a map of lists containing all the imports defined here. The map's
     * keys are the namespaceURIs, and the map's values are lists. There is one
     * list for each namespaceURI for which imports have been defined.
     */
    List<Import> getImports();

    /**
     * Add an include to this WSDL description.
     * 
     * @param includeDef
     *            the include to be added
     * @throws WSDLException
     */
    void addInclude(Include includeDef) throws WSDLException;

    /**
     * Remove an include from this WSDL description.
     * 
     * @param includeDef
     *            the include to be removed
     * @return the removed include
     * @throws WSDLException
     */
    Include removeInclude(Include includeDef) throws WSDLException;

    /**
     * Get the list of includes for the specified locationURI.
     * 
     * @param locationURI
     *            the locationURI associated with the desired includes.
     * @return a list of the corresponding includes, or null if there weren't
     *         any matching includes
     */
    List<Include> getIncludes(String locationURI);

    /**
     * Get a map of lists containing all the includes defined here. The map's
     * keys are the namespaceURIs, and the map's values are lists. There is one
     * list for each locationURI for which imports have been defined.
     */
    List<Include> getIncludes();

    /**
     * Add a binding to this WSDL description.
     * 
     * @param binding
     *            the binding to be added
     */
    void addBinding(Binding binding);

    /**
     * Get the specified binding. Also checks imported documents.
     * 
     * @param name
     *            the name of the desired binding.
     * @return the corresponding binding, or null if there wasn't any matching
     *         binding
     */
    Binding getBinding(QName name);

    /**
     * Remove the specified binding from this definition.
     * 
     * @param name
     *            the name of the binding to remove
     * @return the binding previously associated with this qname, if there was
     *         one; may return null
     */
    Binding removeBinding(QName name);

    /**
     * Get all the bindings defined in this Definition.
     */
    List<Binding> getBindings();

    /**
     * Add a portType to this WSDL description.
     * 
     * @param interfaceType
     *            the portType to be added
     */
    void addInterface(InterfaceType interfaceType);

    /**
     * Get the specified portType. Also checks imported documents.
     * 
     * @param name
     *            the name of the desired portType.
     * @return the corresponding portType, or null if there wasn't any matching
     *         portType
     */
    InterfaceType getInterface(QName name);

    /**
     * Remove the specified portType from this definition.
     * 
     * @param name
     *            the name of the portType to remove
     * @return the portType previously associated with this qname, if there was
     *         one; may return null
     */
    InterfaceType removeInterface(QName name);

    /**
     * Get all the portTypes defined in this Definition.
     */
    List<InterfaceType> getInterfaces();

    /**
     * Add a service to this WSDL description.
     * 
     * @param service
     *            the service to be added
     */
    void addService(Service service);

    /**
     * Get the specified service. Also checks imported documents.
     * 
     * @param name
     *            the name of the desired service.
     * @return the corresponding service, or null if there wasn't any matching
     *         service
     */
    Service getService(QName name);

    /**
     * Remove the specified service from this definition.
     * 
     * @param name
     *            the name of the service to remove
     * @return the service previously associated with this qname, if there was
     *         one; may return null
     */
    Service removeService(QName name);

    /**
     * Get all the services defined in this Definition.
     */
    List<Service> getServices();

    /**
     * Create a new binding.
     * 
     * @return the newly created binding
     */
    Binding createBinding();

    /**
     * Create a new binding fault.
     * 
     * @return the newly created binding fault
     */
    BindingFault createBindingFault();

    /**
     * Create a new binding operation.
     * 
     * @return the newly created binding operation
     */
    BindingOperation createBindingOperation();

    /**
     * Create a new fault.
     * 
     * @return the newly created fault
     */
    Fault createFault();

    /**
     * Create a new import.
     * 
     * @return the newly created import
     */
    Import createImport();

    /**
     * Create a new input.
     * 
     * @return the newly created input
     */
    Input createInput();

    /**
     * Create a new operation.
     * 
     * @return the newly created operation
     */
    Operation createOperation();

    /**
     * Create a new output.
     * 
     * @return the newly created output
     */
    Output createOutput();

    /**
     * Create a new part.
     * 
     * @return the newly created part
     */
    Part createPart();

    /**
     * Create a new endpoint.
     * 
     * @return the newly created endpoint
     */
    Endpoint createEndpoint();

    /**
     * Create a new interface.
     * 
     * @return the newly created interface
     */
    InterfaceType createInterface();

    /**
     * Create a new service.
     * 
     * @return the newly created service
     */
    Service createService();

    /**
     * Create a new types section.
     * 
     * @return the newly created types section
     */
    Types createTypes();

    /**
     * get the version of the wsdl
     */
    WSDLVersionConstants getVersion();
    
    
    /**
     * get the schema location
     * @return
     */
    Map<String, String> getSchemaLocation();
}
