/**
 * easyWSDL - SOA Tools Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * $id.java
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.commons.wsdl.api.abstractElmt;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Map.Entry;
import java.util.logging.Logger;

import javax.xml.XMLConstants;
import javax.xml.namespace.NamespaceContext;
import javax.xml.namespace.QName;

import org.jdom.Document;
import org.jdom.Namespace;

import com.ebmwebsourcing.commons.schema.api.abstractElmt.AbstractSchema;
import com.ebmwebsourcing.commons.wsdl.api.Binding;
import com.ebmwebsourcing.commons.wsdl.api.Description;
import com.ebmwebsourcing.commons.wsdl.api.Import;
import com.ebmwebsourcing.commons.wsdl.api.Include;
import com.ebmwebsourcing.commons.wsdl.api.InterfaceType;
import com.ebmwebsourcing.commons.wsdl.api.Service;
import com.ebmwebsourcing.commons.wsdl.api.Types;
import com.ebmwebsourcing.commons.wsdl.api.WSDLException;
import com.ebmwebsourcing.commons.wsdl.api.WSDLReader.FeatureConstants;

/**
 * @author Nicolas Salatge - eBM WebSourcing
 */
public abstract class AbstractDescriptionImpl<E> extends AbstractWSDLElementImpl<E> implements
        Description {

    /**
	 * 
	 */
    private static final long serialVersionUID = 1L;

    private static Logger log = Logger.getLogger(AbstractSchema.class.getName());
    
    private Namespace xsi = org.jdom.Namespace.getNamespace("xsi", XMLConstants.W3C_XML_SCHEMA_INSTANCE_NS_URI);

    /**
     * Features
     */
    protected Map<FeatureConstants, Object> features = new HashMap<FeatureConstants, Object>();

    /**
     * the namespace context
     */
    protected NamespaceContext namespaceContext;

    /**
     * the jdom document
     */
    protected Document document;

    /**
     * list of imports
     */
    protected List<Import> imports = new ArrayList<Import>();

    /**
     * list of includes
     */
    protected List<Include> includes = new ArrayList<Include>();

    /**
     * list of services
     */
    protected List<Service> services = new ArrayList<Service>();

    /**
     * list of binding
     */
    protected List<Binding> bindings = new ArrayList<Binding>();

    /**
     * list of interface
     */
    protected List<InterfaceType> interfaces = new ArrayList<InterfaceType>();

    /**
     * types
     */
    protected Types types;

    /**
     * baseURI
     */
    protected URI baseURI;
    
    
    /**
     * schema location
     */
    protected Map<String, String> schemaLocations = new HashMap<String, String>();

    /**
     * 
     * @param schema
     * @param doc
     */
    @SuppressWarnings("unchecked")
    public AbstractDescriptionImpl(final URI baseURI, final E model, final Document doc,
            final Map<FeatureConstants, Object> features) throws WSDLException {
        this.model = model;
        this.features = features;
        this.document = doc;
        this.baseURI = baseURI;

        // get all namespaces
        this.namespaceContext = new com.ebmwebsourcing.commons.schema.api.extensions.NamespaceContext();
        final List<Namespace> namespaces = this.document.getRootElement()
                .getAdditionalNamespaces();
        for (final Namespace ns : namespaces) {
            if ((((ns.getPrefix() == null) || (ns.getPrefix().equals(""))))
                    && (this.document.getRootElement().getNamespace().getURI().equals(ns.getURI()))) {
                ((com.ebmwebsourcing.commons.schema.api.extensions.NamespaceContext) this.namespaceContext)
                        .addNamespace(this.document.getRootElement().getNamespace().getPrefix(),
                                this.document.getRootElement().getNamespace().getURI());
            } else {
                ((com.ebmwebsourcing.commons.schema.api.extensions.NamespaceContext) this.namespaceContext)
                        .addNamespace(ns.getPrefix(), ns.getURI());
            }
        }
        
        // get schemaLocation
        if (this.document.getRootElement().getAttributeValue("schemaLocation", xsi) != null) {
            String val = this.document.getRootElement().getAttributeValue("schemaLocation", xsi);
            final StringTokenizer st = new StringTokenizer(val, " ");
            String location = null;
            String ns = null;
            while (st.hasMoreTokens()) {
                ns = st.nextToken();
                if (st.hasMoreTokens()) {
                    location = st.nextToken();
                } else {
                    location = null;
                }
                if(ns != null) {
                    schemaLocations.put(ns, location);
                }
                ns = null;
            }
        }
    }

    protected void addImportElementsInAllList() {
        for (final Import impt : this.imports) {
            if (impt.getDescription() != null) {
                this.services.addAll(impt.getDescription().getServices());
                this.bindings.addAll(impt.getDescription().getBindings());
                this.interfaces.addAll(impt.getDescription().getInterfaces());
                if (this.types != null) {
                    this.types.getSchemas().addAll(impt.getDescription().getTypes().getSchemas());
                } else {
                    this.types = impt.getDescription().getTypes();
                }
            }
        }
    }

    protected void addIncludeElementsInAllList() {
        for (final Include incl : this.includes) {
            if (incl.getDescription() != null) {
                this.services.addAll(incl.getDescription().getServices());
                this.bindings.addAll(incl.getDescription().getBindings());
                this.interfaces.addAll(incl.getDescription().getInterfaces());
                if (this.types != null) {
                    this.types.getSchemas().addAll(incl.getDescription().getTypes().getSchemas());
                } else {
                    this.types = incl.getDescription().getTypes();
                }
            }
        }
    }

    /**
     * ImportImpl operation
     */
    public void addImport(final Import impt) {
        if (this.imports == null) {
            this.imports = new ArrayList<Import>();
        }
        this.imports.add(impt);
    }

    public List<Import> getImports() {
        if (this.imports == null) {
            this.imports = new ArrayList<Import>();
        }
        return this.imports;
    }

    public List<Import> getImports(final String namespaceUri) {
        final List<Import> res = new ArrayList<Import>();
        for (final Import impt : this.imports) {
            if (impt.getNamespaceURI().equals(namespaceUri)) {
                res.add(impt);
            }
        }
        return res;
    }

    /**
     * IncludeImpl operation
     * 
     * @throws WSDLException
     */
    public void addInclude(final Include incl) throws WSDLException {
        if (this.includes == null) {
            this.includes = new ArrayList<Include>();
        }
        this.includes.add(incl);
    }

    public List<Include> getIncludes() {
        if (this.includes == null) {
            this.includes = new ArrayList<Include>();
        }
        return this.includes;
    }

    public List<Include> getIncludes(final String locationUri) {
        final List<Include> res = new ArrayList<Include>();
        for (final Include incl : this.includes) {
            if (incl.getLocationURI().equals(locationUri)) {
                res.add(incl);
            }
        }
        return res;
    }

    /**
     * ServiceImpl operation
     */
    public void addService(final Service service) {
        if (this.services == null) {
            this.services = new ArrayList<Service>();
        }
        this.services.add(service);
    }

    public List<Service> getServices() {
        if (this.services == null) {
            this.services = new ArrayList<Service>();
        }
        return this.services;
    }

    public Service getService(final QName name) {
        Service res = null;
        for (final Service s : this.services) {
            if (s.getQName().equals(name)) {
                res = s;
                break;
            }
        }
        return res;
    }

    /**
     * BindingImpl operation
     */
    public void addBinding(final Binding binding) {
        if (this.bindings == null) {
            this.bindings = new ArrayList<Binding>();
        }
        this.bindings.add(binding);
    }

    public List<Binding> getBindings() {
        if (this.bindings == null) {
            this.bindings = new ArrayList<Binding>();
        }
        return this.bindings;
    }

    public Binding getBinding(final QName name) {
        Binding res = null;
        for (final Binding b : this.bindings) {
            if (b.getQName().equals(name)) {
                res = b;
                break;
            }
        }
        return res;
    }

    /**
     * Interface operation
     */
    public void addInterface(final InterfaceType itf) {
        if (this.interfaces == null) {
            this.interfaces = new ArrayList<InterfaceType>();
        }
        this.interfaces.add(itf);
    }

    public List<InterfaceType> getInterfaces() {
        if (this.interfaces == null) {
            this.interfaces = new ArrayList<InterfaceType>();
        }
        return this.interfaces;
    }

    public InterfaceType getInterface(final QName name) {
        InterfaceType res = null;
        for (final InterfaceType i : this.interfaces) {
            if (i.getQName().equals(name)) {
                res = i;
                break;
            }
        }
        return res;
    }

    /**
     * @return the features
     */
    public Map<FeatureConstants, Object> getFeatures() {
        if (this.features == null) {
            this.features = new HashMap<FeatureConstants, Object>();
        }
        return this.features;
    }

    /**
     * @return the feature value
     */
    public Object getFeatureValue(final FeatureConstants feature) {
        if (this.features == null) {
            this.features = new HashMap<FeatureConstants, Object>();
        }

        return this.features.get(feature);
    }

    /**
     * @param features
     *            the features to set
     */
    public void setFeatures(final Map<FeatureConstants, Object> features) {
        this.features = features;
    }

    public Types getTypes() {
        return this.types;
    }

    public void setTypes(final Types types) {
        this.types = types;
    }

    /**
     * @return the document
     */
    public Document getDocument() {
        return this.document;
    }

    /**
     * @param document
     *            the document to set
     */
    public void setDocument(final Document document) {
        this.document = document;
    }

    /**
     * methods for namespaces
     */
    public NamespaceContext getNamespaces() {
        return this.namespaceContext;
    }

    /**
     * methods for baseURI
     */

    public URI getDocumentBaseURI() {
        URI res = null;
        try {
            if (this.baseURI.getPath().lastIndexOf("/") != -1) {
                res = new URI(this.baseURI.getPath().substring(0,
                        this.baseURI.getPath().lastIndexOf("/") + 1));
            }
        } catch (final URISyntaxException e) {
            AbstractDescriptionImpl.log.warning("BaseURI is null : " + this.baseURI);
        }
        return res;
    }

    public void setDocumentBaseURI(final URI documentBaseURI) {
        this.baseURI = documentBaseURI;
    }
    
    /**
     * @return the schemaLocations
     */
    public Map<String, String> getSchemaLocation() {
        return schemaLocations;
    }


    @Override
    public String toString() {
        return this.baseURI.toString();
    }
}
